/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pkt-bpf.c,v 1.16 2002/10/20 10:39:28 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:		pkt-bpf.c
 * created on:		Sat Jul  6 11:04:26 CEST 2002
 * created by:		peter
 * project: 		libpnet6
 *----------------------------------------------------------------------*/

# include "../local.h"

# ifdef PNET_HAVE_BPF						/* { */

# include "../pkt.h"
# include "../pnet6pkt.h"

/*----------------------------------------------------------------------*/
/* Monitoring packets on the network, using the Berkeley Packet Filter. */
/*----------------------------------------------------------------------*/

int
pkt_get_ll_type( struct pnet_pktacc *pa )
{
    u_int type = -1;

    if ( ioctl( pa->pa_fd, BIOCGDLT, &type) )
	{ FATALERR("ioctl(...,BIOCGDLT,...)"); return -1; }

    switch ( type )
    {
    case DLT_NULL:
	pa->pa_llh_len	= 4;
# ifdef DLT_LOOP
    case DLT_LOOP:
	pa->pa_llh_len	= 4;
# endif
	break;

    case DLT_EN10MB:
	pa->pa_llh_len	= 14;
	break;

    case DLT_FDDI:
	pa->pa_llh_len  = 21;
	break;
    }
    pa->pa_iftype = (int) type;

    pdbg(E_INFO,"pkt_get_ll_type(): interface type: %s (%d)\n",
		 pkt_get_ll_name(pa), type );
    pdbg(E_INFO,"pkt_get_ll_type(): llh_len = %d\n", pa->pa_llh_len );
    return 0;
}
int
pkt_set_promiscuous( struct pnet_pktacc * pa, int on )
{
    int ret = 0;

    DBG(dbg("pkt_set_promiscuous(pa=%X,on=%s)\n",XX(pa),PONOFF(on)));

    if ( on )
    {
	if ( (ret = ioctl( pa->pa_fd, BIOCPROMISC, NULL )) )
	    { FATALERR("ioctl(...,BIOCPROMISC,...)"); ret = -1; }
    }

    if (ret == 0)
	pa->pa_ifisprom = on != 0;

    return - (ret != 0);
}
int
pkt_set_read_timeout( PNetPktAccess *pa, int msec )
{
    struct timeval 	tv;

    if ( msec <= 0 )
    {
	u_int on = 1;

	if ( ioctl( pa->pa_fd, BIOCIMMEDIATE, &on ) )
	    { FATALERR("ioctl(...,BIOCIMMEDIATE,...)"); return -1; }

	pdbg(E_INFO,"packet read timeout off.\n");
    }
    else
    {
	tv.tv_sec	= msec / 1000;
	tv.tv_usec	= (msec % 1000) * 1000;

	if ( ioctl( pa->pa_fd, BIOCSRTIMEOUT, &tv ) )
	    { FATALERR("ioctl(...,BIOCSRTIMEOUT,...)"); return -1; }

	pdbg(E_INFO,"packet read timeout set to %d.%d seconds\n",
	     tv.tv_sec,tv.tv_usec / 1000);
    }

    return 0;
}
int
pkt_get_stats( PNetPktAccess * pa, pnet_uint * caught, pnet_uint * dropped )
{
    struct bpf_stat	stats;

    if ( ioctl( pa->pa_fd, BIOCGSTATS, &stats ) )
	{ FATALERR("ioctl(...,BIOCGSTATS,...)"); return -1; }

    if (caught)
	*caught = stats.bs_recv;
    if (dropped)
	*dropped = stats.bs_drop;
    return 0;
}
struct pnet_pktacc*
pkt_open_dev( const char *ifname, int tries, int glen )
{
    struct pnet_pktacc *	pa;
    int         		count;
    char       		 	dev[12];
    struct ifreq 		ifr;
    int				fd = -1;
    struct bpf_version		bpf_version;
# ifdef PNET_TRU64
    u_short			eiocmbis_bits;
# endif
    
    for (count = 0; count < tries; count++)
    {
# if defined PNET_TRU64
	sprintf(dev,"/dev/pf/pfilt%d",count);
# else
        sprintf(dev,"/dev/bpf%d",count);
# endif
        pdbg(E_DBG4,"pkt_open_dev(): trying device %s\n", dev);

# ifdef HAVE_PFOPEN
#   define bpf_open	pfopen
# else
#   define bpf_open   	open
# endif
	if ( (fd = bpf_open( dev, O_RDWR )) > 0 )
	{
	    pdbg(E_DBG1,"pkt_open_dev(): got device %s\n",dev );
	    break;
	}

	perr(E_WARN,"device %s: %s\n",dev,SYSERR());

	if ( errno == ENOENT )
	{
	   perr(E_FATAL,"No more bpf devices available\n");
	   return NULL;
	}
    }

    if ( count == tries )
	{ FATALERR("pkt_open_dev()"); return NULL; }

    strncpy( ifr.ifr_name, ifname, IFNAMSIZ );

    if ( ioctl( fd, BIOCSETIF, &ifr) )
    {
	pdbg(E_FATAL,"pkt_open_dev(): cannot bind to interface %s\n",ifname);
	LOGSYSERR("pkt_open_dev()");
        close( fd );
        return NULL;
    }

    DBG(dbg("Bound to interface %s\n",ifname));

    /* Check filter version of kernel */
    if ( ioctl( fd, BIOCVERSION, &bpf_version ) )
    {
	perr(E_FATAL, "pkt_open_dev(): cannot determine BPF version.\n");
	LOGSYSERR("pkt_open_dev()");
	close( fd );
	return NULL;
    }
    /*
     * bpf(4) says that filters are OK if the kernel major matches our major
     * and the kernel minor is bigger than or equal to our minor.
     * In this case, the kernel values are given in bpf_version.
     */
    if ( bpf_version.bv_major != BPF_MAJOR_VERSION ||
	 bpf_version.bv_minor <  BPF_MINOR_VERSION )
    {
	pdbg(E_WARN,"pkt_open_dev(): kernel BPF version outdated.\n");
    }
    else
    {
	pdbg(E_INFO,"BPF version %d.%d\n",
		    bpf_version.bv_major, bpf_version.bv_minor );
    }

# ifdef PNET_TRU64	/* { */
    /* The Tru64 UNIX Packet Filter does not prepend a header to each */
    /* packet, while the BSD Packet Filter does. We need to use the   */
    /* EIOCMBIS ioctl() and set the ENBPFHDR mode bit.		  */

    eiocmbis_bits = ENBPFHDR | ENPROMISC | ENCOPYALL;
    if ( ioctl( fd, EIOCMBIS, &eiocmbis_bits ) )
    {
	perr(E_FATAL, "ioctl(..., EIOCMBIS, ... ) failed\n" );
	LOGSYSERR("pkt_open_dev()");
	close( fd );
	return NULL;
    }
# endif			/* } */

    STDMALLOC( pa, sizeof(struct pnet_pktacc), NULL );

    pa->pa_fd	= fd;

    strncpy( pa->pa_ifname, ifname, IFNAMSIZ );

    if ( ioctl( fd, BIOCGBLEN, &pa->pa_buflen ) )
    { 
	perr(E_INFO,"pkt_open_dev(): can't grab bpf buffer size.\n");
	pa->pa_buflen	= PKTBUFSIZ;
    }

    pdbg(E_INFO,"pkt_open_dev(): using capture buffer size of %d\n",
	 pa->pa_buflen);

    STDMALLOC( pa->pa_buf, pa->pa_buflen, NULL );

    pa->pa_glen = glen;
    pa->pa_cooked = PKTACC_NOT_COOKED;

    pkt_get_ll_type( pa );

    return pa;
}
void
pkt_close_dev( PNetPktAccess * pa )
{
    pdbg(E_DBG4,"pkt_close_dev(): closing device %s\n", pa->pa_ifname );

    if ( pa->pa_ifisprom )
	pkt_set_promiscuous( pa, 0 );

    close( pa->pa_fd );
    STDFREE( pa->pa_buf );
    STDFREE( pa );
}
int
pkt_set_bpf_filter( PNetPktAccess *pa, BPF_FILTER *insn, int len )
{
    struct bpf_program 	prgm;
    prgm.bf_len = len;
    prgm.bf_insns = insn;

    if ( ioctl( pa->pa_fd, BIOCSETF, &prgm) )
	{ FATALERR("ioctl(...,BIOCSETF,...)"); return -1; }

    pdbg(E_DBG4,"pkt_set_bpf_filter(): installed filter ok\n");

    return 0;
}
int
pkt_next_pkt( PNetPktAccess *pa, PNetPacket *pkt )
{
    struct bpf_hdr * bh;

    if ( pa->pa_sbuf >= pa->pa_ebuf )
    {
	/* Read new data from interface */
	int got = read( pa->pa_fd, pa->pa_buf, pa->pa_buflen );

	DBG(dbg("Got %d bytes from interface %s\n",got,pa->pa_ifname));

	if ( got <= 0 )
	    return got; 	/* timeout or error */

	/* Mark start and end of buffer */
	pa->pa_sbuf = pa->pa_buf;
	pa->pa_ebuf = pa->pa_sbuf + got;
    }
    else
    {
	/* pa's buffer contains more packets. Skip to the next one here */

	bh = (struct bpf_hdr*) pa->pa_sbuf;	/* Current pkt header */

	DBG(dbg("Got %d bytes from buffer\n",bh->bh_caplen));

	/* Advance pa's buffer start pointer to just behind this packet */
	pa->pa_sbuf += BPF_WORDALIGN( bh->bh_hdrlen + bh->bh_caplen );

	if ( pa->pa_sbuf >= pa->pa_ebuf )
	{
	    pa->pa_sbuf = pa->pa_ebuf;
	    return 0;			/* No more packets */
	}
    }

    /* Get data from next packet */
    bh = (struct bpf_hdr*) pa->pa_sbuf;

    pkt->pkt_tssec  = bh->bh_tstamp.tv_sec;
    pkt->pkt_tsusec = bh->bh_tstamp.tv_usec;
    pkt->pkt_grablen= bh->bh_caplen;
    pkt->pkt_datalen= bh->bh_datalen;
    DBG( dbg("pkt_next_pkt(): pkt: grablen=%d, datalen=%d\n",
	     pkt->pkt_grablen,pkt->pkt_datalen ) );
    /* Skip past bpf header */
    pkt->pkt_buf = pa->pa_sbuf + bh->bh_hdrlen;
    pkt->pkt_pa	 = pa;
   
    /* pkt_set_type( pkt ); */

    return 1;		/* More packet data present */
}
int
pkt_output( PNetPktAccess *pa, pnet_byte *buf, pnet_uint blen )
{
    return write( pa->pa_fd, buf, blen );
}
# if 0
int
pkt_set_type( PNetPacket * pkt )
{
    byte * b;

    if ( ! pkt->pkt_pa )
	return -1;

    b = pkt_get_payload( pkt );

    switch ( pkt->pkt_pa->pa_iftype )
    {
    case DLT_NULL:

	if ( pkt->pkt_buf[0] == AF_INET )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv4;
	    pkt->pkt_stype= ((struct ip*) b)->ip_p;
	}
	else if ( pkt->pkt_buf[0] == AF_INET6 )
	{
	    pkt->pkt_type = PNET_PKTTYPE_IPv6;
	    pkt->pkt_stype= ((struct ip6_hdr*)b)->ip6_nxt;
	}
	return 0;

    case DLT_EN10MB:
	{
	    struct ether_header *eh = (struct ether_header*) pkt->pkt_buf;

	    if ( eh->ether_type == ETHERTYPE_IP )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv4;
		pkt->pkt_stype= ((struct ip*) b)->ip_p;
	    }
	    else if ( eh->ether_type == ETHERTYPE_IPV6 )
	    {
		pkt->pkt_type = PNET_PKTTYPE_IPv6;
		pkt->pkt_stype= ((struct ip6_hdr*)b)->ip6_nxt;
	    }
	    else if ( eh->ether_type == ETHERTYPE_ARP  )
	    {
		pkt->pkt_type = PNET_PKTTYPE_ARP;
		pkt->pkt_stype= ((struct arphdr*)b)->ar_hrd;
	    }
	    else if ( eh->ether_type == ETHERTYPE_REVARP  )
	    {
		pkt->pkt_type= PNET_PKTTYPE_RARP;
	    }
	}
	return 0;
    }

    return -1;
}

# endif
# endif 				/* } */
