/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: nperr.h,v 1.4 2002/09/06 15:56:15 kingofgib Exp $
 */


# ifndef _NETWORK_PROTOCOL_ERRORS_H_
# define _NETWORK_PROTOCOL_ERRORS_H_

# ifdef STD_WIN98
# include <winsock.h>
# else
# include <winsock2.h>
# endif

# define NP_ERR_EACCES		0  // WSAEACCES  (10013) Permission denied. 
# define NP_ERR_EADDRINUSE	1  // WSAEADDRINUSE  (10048) Address already in use. 
# define NP_ERR_EADDRNOTAVAIL	2  // WSAEADDRNOTAVAIL (10049) Cannot assign requested address
# define NP_ERR_EAFNOSUPPORT	3  // WSAEAFNOSUPPORT  (10047) Address family not supported by protocol family
# define NP_ERR_EALREADY	4  // WSAEALREADY (10037) Operation already in progress.
# define NP_ERR_ECONNABORTED   5  // WSAECONNABORTED (10053) Software caused connection abort. 
# define NP_ERR_ECONNREFUSED   6  // WSAECONNREFUSED (10061) Connection refused
# define NP_ERR_ECONNRESET     7  // WSAECONNRESET (10054) Connection reset by peer
# define NP_ERR_EDESTADDRREQ   8  //WSAEDESTADDRREQ (10039) Destination address required
# define NP_ERR_EFAULT         9  // WSAEFAULT (10014) Bad address
# define NP_ERR_EHOSTDOWN      10 // WSAEHOSTDOWN (10064) Host is down. 
# define NP_ERR_EHOSTUNREACH   11 //WSAEHOSTUNREACH (10065) No route to host. 
# define NP_ERR_EINPROGRESS    12 // WSAEINPROGRESS (10036) Operation now in progress. 
# define NP_ERR_EINTR          13 // WSAEINTR (10004) Interrupted function call. 
# define NP_ERR_EINVAL         14 // WSAEINVAL (10022) Invalid argument. 
# define NP_ERR_EISCONN        15 // WSAEISCONN (10056) Socket is already connected.
# define NP_ERR_EMFILE         16 // WSAEMFILE (10024) Too many open files (sockets).
# define NP_ERR_EMSGSIZE       17 // WSAEMSGSIZE (10040) Message too long. 
# define NP_ERR_ENETDOWN       18 // WSAENETDOWN (10050) Network is down. 
# define NP_ERR_ENETRESET      19 // WSAENETRESET (10052) Network dropped connection on reset.
# define NP_ERR_ENETUNREACH    20 // WSAENETUNREACH (10051) Network is unreachable.
# define NP_ERR_ENOBUFS        21 // WSAENOBUFS (10055) No buffer space available.
# define NP_ERR_ENOPROTOOPT    22 // WSAENOPROTOOPT (10042) Bad protocol option.
# define NP_ERR_ENOTCONN       23 // WSAENOTCONN (10057) Socket is not connected.
# define NP_ERR_ENOTSOCK       24 // WSAENOTSOCK (10038) Socket operation on non-socket.
# define NP_ERR_EOPNOTSUPP     25 // WSAEOPNOTSUPP (10045) Operation not supported.
# define NP_ERR_EPFNOSUPPORT   26 //WSAEPFNOSUPPORT (10046) Protocol family not supported.
# define NP_ERR_EPROCLIM       27 // WSAEPROCLIM (10067) Too many processes.
# define NP_ERR_EPROTONOSUPPORT 28 //WSAEPROTONOSUPPORT (10043) Protocol not supported.
# define NP_ERR_EPROTOTYPE     29 // WSAEPROTOTYPE (10041) Protocol wrong type for socket.
# define NP_ERR_ESHUTDOWN      30 // WSAESHUTDOWN (10058) Cannot send after socket shutdown.
# define NP_ERR_ESOCKTNOSUPPORT 31 // WSAESOCKTNOSUPPORT (10044) Socket type not supported.
# define NP_ERR_ETIMEDOUT      32 // WSAETIMEDOUT (10060) Connection timed out.
# define NP_ERR_EWOULDBLOCK    33 // WSAEWOULDBLOCK (10035) Resource temporarily unavailable.
# define NP_ERR_HOST_NOT_FOUND 34 // WSAHOST_NOT_FOUND (11001) Host not found.
# define NP_ERR_NOTINITIALISED 35 // WSANOTINITIALISED (10093) Successful WSAStartup() not yet performed.
# define NP_ERR_NO_DATA        36 // WSANO_DATA (11004) Valid name, no data record of requested type.
# define NP_ERR_NO_RECOVERY    37 // WSANO_RECOVERY (11003) This is a non-recoverable error.
# define NP_ERR_SYSNOTREADY    38 // WSASYSNOTREADY (10091) Network subsystem is unavailable.
# define NP_ERR_TRY_AGAIN      39 // WSATRY_AGAIN (11002) Non-authoritative host not found.
# define NP_ERR_VERNOTSUPPORTED  40 // WSAVERNOTSUPPORTED (10092) WINSOCK.DLL version out of range.
# define NP_ERR_EDISCON        41 // WSAEDISCON (10094) Graceful shutdown in progress.

# define NP_ERR_LAST_ERROR     42

typedef struct err_entry
{
    int     np_code;
    int     win_code;
} ErrorEntry;

ErrorEntry   NP_ErrorTranslateTable[] =
{
    { NP_ERR_EACCES,       WSAEACCES           },      // 0
    { NP_ERR_EADDRINUSE,   WSAEADDRINUSE       },
    { NP_ERR_EADDRNOTAVAIL,WSAEADDRNOTAVAIL,   },
    { NP_ERR_EAFNOSUPPORT, WSAEAFNOSUPPORT,    },
    { NP_ERR_EALREADY,     WSAEALREADY         },
    
    { NP_ERR_ECONNABORTED, WSAECONNABORTED     },      // 5
    { NP_ERR_ECONNREFUSED, WSAECONNREFUSED     },
    { NP_ERR_ECONNRESET,   WSAECONNRESET       },
    { NP_ERR_EDESTADDRREQ, WSAEDESTADDRREQ     },
    { NP_ERR_EFAULT,       WSAEFAULT           },

    { NP_ERR_EHOSTDOWN,    WSAEHOSTDOWN        },      // 10
    { NP_ERR_EHOSTUNREACH, WSAEHOSTUNREACH     },  
    { NP_ERR_EINPROGRESS,  WSAEINPROGRESS      },
    { NP_ERR_EINTR,        WSAEINTR            }, 
    { NP_ERR_EINVAL,       WSAEINVAL           },

    { NP_ERR_EISCONN,      WSAEISCONN          },      // 15
    { NP_ERR_EMFILE,       WSAEMFILE           },
    { NP_ERR_EMSGSIZE,     WSAEMSGSIZE         },
    { NP_ERR_ENETDOWN,     WSAENETDOWN         }, 
    { NP_ERR_ENETRESET,    WSAENETRESET        },

    { NP_ERR_ENETUNREACH,  WSAENETUNREACH      },      // 20
    { NP_ERR_ENOBUFS,      WSAENOBUFS          },
    { NP_ERR_ENOPROTOOPT,  WSAENOPROTOOPT      },
    { NP_ERR_ENOTCONN,     WSAENOTCONN         },
    { NP_ERR_ENOTSOCK,     WSAENOTSOCK         },

    { NP_ERR_EOPNOTSUPP,   WSAEOPNOTSUPP       },      // 25
    { NP_ERR_EPFNOSUPPORT, WSAEPFNOSUPPORT     },
    { NP_ERR_EPROCLIM,     WSAEPROCLIM         },
    { NP_ERR_EPROTONOSUPPORT,WSAEPROTONOSUPPORT},
    { NP_ERR_EPROTOTYPE,   WSAEPROTOTYPE       },

    { NP_ERR_ESHUTDOWN,    WSAESHUTDOWN        },      // 30
    { NP_ERR_ESOCKTNOSUPPORT, WSAESOCKTNOSUPPORT},
    { NP_ERR_ETIMEDOUT,    WSAETIMEDOUT        },
    { NP_ERR_EWOULDBLOCK,  WSAEWOULDBLOCK      },
    { NP_ERR_HOST_NOT_FOUND,WSAHOST_NOT_FOUND  },

    { NP_ERR_NOTINITIALISED,WSANOTINITIALISED  },      // 35
    { NP_ERR_NO_DATA,      WSANO_DATA          },
    { NP_ERR_NO_RECOVERY,  WSANO_RECOVERY      },
    { NP_ERR_SYSNOTREADY,  WSASYSNOTREADY      },
    { NP_ERR_TRY_AGAIN,    WSATRY_AGAIN        },

    { NP_ERR_VERNOTSUPPORTED,WSAVERNOTSUPPORTED},      // 40
    { NP_ERR_EDISCON,      WSAEDISCON          },
//    { NP_ERR_LAST_ERROR,   NP_ERR_LAST_ERROR  }     // 42
};

static char * NP_ErrorTable[] = 
{
    // 0
    "Permission denied.",                                // NP_ERR_EACCES
    "Address already in use.",                           // NP_ERR_EADDRINUSE 
    "Cannot assign requested address.",                  // NP_ERR_EADDRNOTAVAIL
    "Address family not supported by protocol family.",  // NP_ERR_EAFNOSUPPORT
    "Operation already in progress.",                    // NP_ERR_EALREADY
    "Software caused connection abort." ,                // NP_ERR_ECONNABORTED

    // 5
    "Connection refused.",                               // NP_ERR_ECONNREFUSED
    "Connection reset by peer.",                         // NP_ERR_ECONNRESET
    "Destination address required." ,                    // NP_ERR_EDESTADDRREQ
    "Bad address."  ,                                    // NP_ERR_EFAULT
    "Host is down." ,                                    // NP_ERR_EHOSTDOWN
    // 10

    "No route to host."         ,                        // NP_ERR_EHOSTUNREACH
    "Operation now in progress.",                        // NP_ERR_EINPROGRESS
    "Interrupted function call.",                        // NP_ERR_EINTR
    "Invalid argument."         ,                        // NP_ERR_EINVAL
    "Socket is already connected.",                      // NP_ERR_EISCONN
// 15
    "Too many open files (sockets).",                    // NP_ERR_EMFILE
    "Message too long."           ,                      // NP_ERR_EMSGSIZE
    "Network is down."            ,                      // NP_ERR_ENETDOWN
    "Network dropped connection on reset.",              // NP_ERR_ENETRESET
    "Network is unreachable."      ,                     // NP_ERR_ENETUNREACH
// 20
    "No buffer space available." ,                       // NP_ERR_ENOBUFS
    "Bad protocol option."      ,                        // NP_ERR_ENOPROTOOPT
    "Socket is not connected."   ,                       // NP_ERR_ENOTCONN
    "Socket operation on non-socket."  ,                 // NP_ERR_ENOTSOCK
    "Operation not supported."       ,                   // NP_ERR_EOPNOTSUPP
// 25
    "Protocol family not supported."  ,                  // NP_ERR_EPFNOSUPPORT
    "Too many processes."         ,                      // NP_ERR_EPROCLIM
    "Protocol not supported."    ,                       // NP_ERR_EPROTONOSUPPORT
    "Protocol wrong type for socket."  ,                 // NP_ERR_EPROTOTYPE
    "Cannot send after socket shutdown."  ,              // NP_ERR_ESHUTDOWN
// 30
    "Socket type not supported."   ,                     // NP_ERR_ESOCKTNOSUPPORT
    "Connection timed out."      ,                       // NP_ERR_ETIMEDOUT
    "Resource temporarily unavailable."   ,              // NP_ERR_EWOULDBLOCK
    "Host not found."                 ,                  // NP_ERR_HOST_NOT_FOUND
    "Successful WSAStartup() not yet performed.",        // NP_ERR_NOTINITIALISED
// 35
    "Valid name, no data record of requested type." ,    // NP_ERR_NO_DATA
    "This is a non-recoverable error."   ,               // NP_ERR_NO_RECOVERY
    "Network subsystem is unavailable."  ,               // NP_ERR_SYSNOTREADY
    "Non-authoritative host not found."  ,               // NP_ERR_TRY_AGAIN
    "WINSOCK.DLL version out of range."  ,               // NP_ERR_VERNOTSUPPORTED
// 40
    "Graceful shutdown in progress."    ,                // NP_ERR_EDISCON
    "Unknown error."                                    // NP_ERR_LAST_ERROR  
// 42
};   

static int
TranslateWSAError(int wsa_err_code)
{
    int i = 0;
    int tab_size = sizeof(NP_ErrorTranslateTable) / sizeof(ErrorEntry);

    for (i = 0; i < tab_size; i++)
        if (NP_ErrorTranslateTable[i].win_code == wsa_err_code)
            return NP_ErrorTranslateTable[i].np_code;
     
    return i; //        NP_ERR_LAST_ERROR
}

# endif /* ! _NETWORK_PROTOCOL_ERRORS_H_ */
