/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * $Id: mcast.c,v 1.27 2002/10/10 13:20:53 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:		mcast.c
 * created on:		Tue Jul 16 00:01:34 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

# ifndef STDWin32
# ifdef PNET_SUN
#   define BSD_COMP
# endif
#   include <sys/ioctl.h>
#   include <net/if.h>
# endif

/*----------------------------- (    7 lines ) --------------------------*/
static int fill_ip4_request(PNETADDR,const char *,struct ip_mreq *);
static int fill_ipv6_request(PNETADDR,const char *,void *);
static int mcast_hops(PNETSOCK,int *,int);
static int mcast_join_ipv4(PNETSOCK,PNETADDR,const char *);
static int mcast_join_ipv6(PNETSOCK,PNETADDR,const char *);
static int mcast_loop(PNETSOCK,int *,int);

/*----------------------------------------------------------------------*/
/* Multicasting routines						*/
/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Join a multicast address						*/
/*----------------------------------------------------------------------*/

# ifndef IPV6_ADD_MEMBERSHIP
#   ifdef IPV6_JOIN_GROUP
#	define IPV6_ADD_MEMBERSHIP	IPV6_JOIN_GROUP
#   endif
# endif

# ifndef IPV6_DROP_MEMBERSHIP
#   ifdef IPV6_JOIN_GROUP
#	define IPV6_DROP_MEMBERSHIP	IPV6_LEAVE_GROUP
#   endif
# endif

/*----------------------------------------------------------------------*/
/* Fill in an IPv4 multicast request structure. Parameters are:		*/
/* pa: address of multicast group;					*/
/* ifnamidx: name or index of interface on which we are multicasting;	*/
/* imr: request struct to fill in.					*/
/*----------------------------------------------------------------------*/
static int
fill_ip4_request( PNETADDR pa, const char * ifnamidx, struct ip_mreq *imr)
{
    PNetIf*		pif;
    PNetIfAddr*		pia;

    /* Copy multicast address 			*/
    if ( pa )
	memcpy( &imr->imr_multiaddr, &pa->pa_sin_addr, sizeof(struct in_addr));

    /* If ifnamidx contains an interface index, look up the name of 
     * the interface, and find the interface address */

    if ( ! ifnamidx )
    {
	/* Let kernel choose which interaface to bind to our address	*/
	imr->imr_interface.s_addr = htonl( INADDR_ANY );
    }
    else 
    {
	/* Given an interface name or index, look up and check if such	*/
	/* an interface exists, whether it has an AF_INET address	*/
	/* and whether it's multicast capable				*/ 

	if( !( pif = net_get_if_by_name( AF_INET, ifnamidx )) )
	{
	    perr(E_FATAL,"Cannot find interface %s\n",ifnamidx );
	    return -1;
	}

	if ( ! pnetIfIsMulticast( pif ) )
	{
	    perr(E_FATAL,"Interface %s (%d) is not multicast enabled\n",
	    	pif->pif_name,pif->pif_index);
	    return -1;
	}

	/* Look up an AF_INET address for the given inteface 	*/
	if ( ! (pia = net_if_get_next_address_type( pif, NULL, AF_INET )) )
	{
	    perr(E_FATAL,"Cannot obtain address for interface %s (%d)\n",
	    		  pif->pif_name, pif->pif_index );
	    return -1;
	}

	memcpy( &imr->imr_interface, &pia->pia_addr4.sin_addr.s_addr, 
		sizeof( struct in_addr ) );
    }
    SLOG( inet_ntoa( imr->imr_multiaddr ) );
    SLOG( inet_ntoa( imr->imr_interface ) );

    return 0;
}
/*----------------------------------------------------------------------*/
/* Join an IPv4 multicast-group (address).				*/
/*----------------------------------------------------------------------*/
static int
mcast_join_ipv4( PNETSOCK ps, PNETADDR pa, const char * ifnamidx )
{
    struct ip_mreq	imr;

    if ( fill_ip4_request( pa, ifnamidx, &imr ) )
	{ return -1; }

    if ( setsockopt( ps->sd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
		     (void*)&imr, sizeof( imr ) ) < 0 )
	{ NETERR("pnetMCastJoin()"); return -1; }

    return 0;
}
/*----------------------------------------------------------------------*/
/* Fill in an IPv6 multicast request structure. Same as w/ IPv4.	*/
/*----------------------------------------------------------------------*/
static int
fill_ipv6_request( PNETADDR pa, const char * ifnamidx,
		   void * pimr )
{
# ifdef HAVE_STRUCT_IPV6_MREQ
    struct ipv6_mreq*	imr = (struct ipv6_mreq*) pimr;
    if ( pa )
	memcpy( &imr->ipv6mr_multiaddr,
		&pa->pa_sin6_addr     ,
		sizeof(struct in6_addr));

    if ( ! ifnamidx )
    	imr->ipv6mr_interface = 0;
    else
    {
	PNetIf*		pif;
	PNETIFADDR	ifaddr;

	if ( !(pif = net_get_if_by_name( AF_INET6, ifnamidx )) )
	{
	    perr(E_FATAL,"Cannot find interface %s\n",ifnamidx );
	    return -1;
	}

	if ( ! pnetIfIsMulticast( pif ) )
	{
	    perr(E_FATAL,"Interface %s (%d) is not multicast enabled\n",
		pif->pif_name,pif->pif_index);
	    return -1;
	}

	ifaddr = pnetIfGetNextAddr( pif, NULL );

	while ( ifaddr && ifaddr->pia_family == AF_INET )
	    ifaddr = pnetIfGetNextAddr( pif, ifaddr );

	if ( ! ifaddr )
	{
	    perr(E_FATAL,"Interface %s (%d) has not IPv6 enabled.\n",
			  pif->pif_name,pif->pif_index);
	    return -1;
	}

	/* For link local multicast address, fill in scope of inteface
	 * address. */

	if ( pa && pnetAddrIsMCLinkLocal( pa ) )
	    pa->pa_scope_id = ifaddr->pia_scope_id;

	imr->ipv6mr_interface = pif->pif_index;
    }
    LLOG( imr->ipv6mr_interface );

    return 0;
# endif
}
static int
mcast_join_ipv6( PNETSOCK ps, PNETADDR pa, const char * ifnamidx )
{
# ifdef HAVE_STRUCT_IPV6_MREQ
    struct ipv6_mreq	imr;

    if ( fill_ipv6_request( pa, ifnamidx, &imr ) )
	{ FATALERR("pnetMCastJoin()"); return -1; }

    if( setsockopt( ps->sd, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, 
		    (void*)&imr, sizeof( imr ) ) )
    {
	NETERR("pnetMCastJoin()");
	return -1;
    }
    return 0;
# endif
}
/*----------------------------------------------------------------------*/
/* Join a multicast group on the given interface. The group's address	*/
/* is specified by pa, which must be a valid multicast address. The 	*/
/* interface is given as ifnamidx, which is a string containing either	*/
/* an interface name, or an interface index. If ifnamidx is NULL, then	*/
/* the interface is chosen by the kernel.				*/
/*----------------------------------------------------------------------*/
int
pnetMCastJoin( PNETSOCK ps, PNETADDR pa, const char * ifnamidx )
{
    char buf[ PNET_ADDR_STRLEN ];
    int ret = 0;

    if ( ps->type != SOCK_DGRAM )
	{ TYPEERR("pnetMCastJoin()"); return -1; }

    if ( ps->fam == AF_INET )
	ret = mcast_join_ipv4( ps, pa, ifnamidx );
    else if ( ps->fam == AF_INET6 )
	ret = mcast_join_ipv6( ps, pa, ifnamidx );
    else
	{ FAMERR("pnetMCastJoin()"); return -1; }

    if ( !ret )
    {
	/* Set reuseaddr for the port, and bind the address in the end */
	PNetIf * pif = NULL;

	if ( pnetSockSetReuseaddr( ps, 1 ) ) 
	    { NETERR("pnetMCastJoin()"); return -1; }

	if ( sock_bind( ps, pa ) )
	    { NETERR("pnetMCastJoin()"); return -1; }

	ps->multicast = 1;

	if ( ifnamidx )
	    pif = net_get_if_by_name( ps->fam, ifnamidx );

	pdbg(E_INFO,"Joined multicast address %s on interface %s\n",
		    pnet_ntop( pa, buf, sizeof( buf ) ),
		    pif ? pif->pif_name : "ANY" );
    }

    return ret;
}
/*----------------------------------------------------------------------*/
/* Leave a multicast group. The group is on the specified local 	*/
/* interface, which is passed as ifnamidx. ifnamidx can also hold 	*/
/* a string containing the index of the interface. 			*/
/* If ifnamidx = NULL, then the first matching multicast group 		*/
/* membership is dropped.						*/
/*----------------------------------------------------------------------*/
int
pnetMCastLeave( PNETSOCK ps, PNETADDR pa, const char * ifnamidx )
{
    char buf[ PNET_ADDR_STRLEN ];
    int ret = 0;

    if ( ps->type != SOCK_DGRAM )
	{ TYPEERR("pnetMCastJoin()"); return -1; }

    if ( ps->fam == AF_INET )
    {
	struct ip_mreq	imr;

	if ( fill_ip4_request( pa, ifnamidx, &imr ) )
	    { FATALERR("pnetMCastLeave()"); ret = -1; }

	if ( setsockopt( ps->sd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
			 (void*)&imr, sizeof( imr ) ) < 0 )
	    { NETERR("pnetMCastLeave()"); ret = -1; }
    }
    else if ( ps->fam == AF_INET6 )
    {
# ifdef HAVE_STRUCT_IPV6_MREQ
	struct ipv6_mreq	imr;

	if ( fill_ipv6_request( pa, ifnamidx, &imr ) )
	    { FATALERR("pnetMCastLeave()"); ret = -1; }

	if( setsockopt( ps->sd, IPPROTO_IPV6, IPV6_DROP_MEMBERSHIP, 
			(void*)&imr, sizeof( imr ) ) )
	    { NETERR("pnetMCastLeave()"); ret =  -1; }
# endif
    }
    else
	{ FAMERR("pnetMCastLeave()"); ret = -1; }

    if ( ! ret )
    {
	pdbg(E_INFO,"Left multicast address %s on interface %s\n",
		    pnet_ntop( pa, buf, sizeof( buf ) ), ifnamidx ? ifnamidx : "ANY" );
    }
    return ret;
}

/*----------------------------------------------------------------------*/
/* Specify default interface for outgoing mcast datagrams.		*/
/* Pass NULL as argument to remove any interface specifications, and 	*/
/* let kernel choose the outgoing interface				*/
/*----------------------------------------------------------------------*/

int
pnetMCastSetInterface( PNETSOCK ps, const char * ifnamidx )
{
    int ret = 0;

    if ( ps->type != SOCK_DGRAM )
	{ TYPEERR("pnetMCastSetInterface()"); return -1; }

    if ( ps->fam == AF_INET )
    {
	struct ip_mreq	imr;

	if ( fill_ip4_request( NULL, ifnamidx, &imr ) )
	    { FATALERR("pnetMCastSetInterface()"); ret = -1; }

	if ( setsockopt( ps->sd, IPPROTO_IP, IP_MULTICAST_IF,
			 (void*)&imr.imr_interface,
		 	 sizeof( struct in_addr ) ) < 0 )
	    { NETERR("pnetMCastSetInterface()"); ret = -1; }
    }
    else if ( ps->fam == AF_INET6 )
    {
# ifdef HAVE_STRUCT_IPV6_MREQ
	struct ipv6_mreq	imr;

	if ( fill_ipv6_request( NULL, ifnamidx, &imr ) )
	    { FATALERR("pnetMCastSetInterface()"); ret = -1; }

	if( setsockopt( ps->sd, IPPROTO_IPV6, IPV6_MULTICAST_IF, 
			(void*)&imr.ipv6mr_interface, sizeof( u_int ) ) )
	    { NETERR("pnetMCastSetInterface()"); ret =  -1; }
# endif
    }
    else
	{ FAMERR("pnetMCastSetInterface()"); ret = -1; }

    if ( ! ret )
    {
	pdbg(E_INFO,"Default interface for outgoing multicasts: %s\n",
		     ifnamidx ? ifnamidx : "ANY");
    }
    return ret;
}
/*----------------------------------------------------------------------*/
/* Set multicast datagrams hop limit 					*/
/*----------------------------------------------------------------------*/

static int
mcast_hops( PNETSOCK ps, int * hops, int set )
{
    DBG( dbg("mcast_hops(ps=%lX,hops=%lX,set=%s)\n",
	      XX(ps),XX(hops),ponoff(set)) );

    if ( ps->type != SOCK_DGRAM )
	{ TYPEERR("mcast_hops()"); return -1; }

    if ( set && !hops )
    {
	perr(E_FATAL,"Request to set multicast hops, no value provided\n");
	return -1;
    }

    if ( ps->fam == AF_INET )
    {
	u_char    ttl = (u_char) *hops;
	socklen_t len = sizeof( u_char );

	if ( set && setsockopt(ps->sd, IPPROTO_IP, IP_MULTICAST_TTL,
			       (void*)&ttl, len ) < 0 )
	    { NETERR("mcast_hops(ipv4): set failed"); return -1; }
	else
	{
	    if ( getsockopt( ps->sd, IPPROTO_IP, IP_MULTICAST_TTL,
			       (void*)&ttl, &len ) < 0 )
	    {
		NETERR("mcast_hops(ipv4): get failed");
		return -1;
	    }
	    *hops = (int) ttl;
	}
	LLOG( ttl ); LLOG( *hops );
    }
    else if ( ps->fam == AF_INET6 )
    {
# ifdef IPV6_MULTICAST_HOPS
	socklen_t len = sizeof( int );

	if ( set && setsockopt(ps->sd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
			       (void*)hops, len ) < 0 )
	    { NETERR("mcast_hops(ipv6): set failed"); return -1; }
	else if ( getsockopt(ps->sd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
			       (void*)hops, &len ) < 0 )
	    { NETERR("mcast_hops(ipv6): get failed"); return -1; }
# endif
    }
    else
	{ FAMERR("mcast_hops()"); return -1; }
    
    if ( set )
	pdbg(E_INFO,"Multicast hops set to %d\n",*hops);
    else 
	pdbg(E_INFO,"Multicast hops = %d\n",*hops);

    return 0;
}
int
pnetMCastSetHops( PNETSOCK ps, int hops )
{
    return mcast_hops( ps, &hops, 1 );
}
int
pnetMCastGetHops( PNETSOCK ps, int * pHops )
{
    return mcast_hops( ps, pHops, 0 );
}
/*----------------------------------------------------------------------*/
/* Enable or disable localloop back for multicast transmissions.	*/
/* (i.e. whether this socket will receive its own multicasts		*/
/*----------------------------------------------------------------------*/
static int
mcast_loop( PNETSOCK ps, int * on, int set )
{
    if ( ps->type != SOCK_DGRAM )
	{ TYPEERR("mcast_loop()"); return -1; }

    *on = !! *on;

    if ( ps->fam == AF_INET )
    {
	u_char uc = (u_char) *on;
	socklen_t len = sizeof( u_char );

	if ( set && setsockopt(ps->sd, IPPROTO_IP, IP_MULTICAST_LOOP,
			(void*)&uc, sizeof( uc ) ) < 0 )
	{
	    NETERR("mcast_loop(): set failed");
	    return -1;
	}
	else
	{
	    if ( getsockopt(ps->sd, IPPROTO_IP, IP_MULTICAST_LOOP,
			(void*)&uc, &len ) < 0 )
	    {
		NETERR("mcast_loop(): get failed");
		return -1;
	    }
	    *on = (int) uc;
	}
    }
    else if ( ps->fam == AF_INET6 )
    {
# ifdef IPV6_MULTICAST_LOOP
	socklen_t len = sizeof( int );
	if ( set && setsockopt(ps->sd, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
			(void*)&on, sizeof( on ) ) < 0 )
	    { NETERR("mcast_loop()"); return -1; }
	else if ( getsockopt(ps->sd, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
			(void*)&on, &len ) < 0 )
	    { NETERR("mcast_loop()"); return -1; }
# endif
    }
    else
	{ FAMERR("mcast_loop()"); return -1; }
    
    if ( set )
	pdbg(E_INFO,"Multicast loop set to %s\n",ponoff( *on ) );
    else 
	pdbg(E_INFO,"Multicast loop  = %s\n",ponoff( *on ) );

    return 0;
}
int
pnetMCastSetLoop( PNETSOCK ps, int on )
{
    return mcast_loop( ps, &on, 1 );
}
int
pnetMCastGetLoop( PNETSOCK ps, int * on )
{
    return mcast_loop( ps, on, 0 );
}
