/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: if-sysctl.c,v 1.17 2002/10/04 09:17:54 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            if-sysctl.c
 * created on:          Wed Aug  7 21:44:53 CEST 2002
 * created by:          peter
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Interface information.						*/
/* Use SYSCTL								*/
/*----------------------------------------------------------------------*/

/*
 * It seems this must be defined prior to include net/if.h on AIX in 
 * order to have the IFF_MULTICAST constant available. It works, but I'm
 * not all too sure if its the right thing to do here. 
 */
# ifdef PNET_AIX
#   define IP_MULTICAST
# endif

# include "../local.h"

# if defined HAVE_SYSCTL || defined HAVE_GETKERNINFO	/* { */

# include "iflocal.h"

# ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
# endif

# include <net/if.h>

# include <sys/param.h>

# ifdef HAVE_SYS_SYSCTL_H
#   include <sys/sysctl.h>
# endif

# include <net/route.h>

# include <net/if_dl.h>

# ifdef HAVE_GETKERNINFO
#   include <sys/ndd_var.h>
#   include <sys/kinfo.h>
# endif

static int
if_get_mtu( int sd, PNetIf* pif )
{
    struct ifreq	ifr;

    strncpy( ifr.ifr_name, pif->pif_name, IFNAMSIZ );

# ifdef SIOCGIFMTU
    if ( if_ioctl( sd, SIOCGIFMTU, &ifr) )
	{ FATALERR("ioctl(...,SIOCGIFMTU,...)"); return -1; }
# endif

    pif->pif_mtu = ifr.ifr_mtu;

    return 0;
}

static int
if_get_hwparams( PNetIf *pif, struct sockaddr_dl *sdl )
{
    int 	alen;		/* Length of address */
    byte * 	p;		/* Pointer to address */
    int		i;
    int		sd;

    p = (byte*) LLADDR(sdl);

    alen = sdl->sdl_alen;
    pif->pif_index = sdl->sdl_index;
    pif->pif_type  = sdl->sdl_type;

    if (pif->hwaddr_len > IF_HWADDRLEN)
    {
	perr(E_FATAL,"if_get_hwparams(): hardware address for %s too long\n",
		    pif->pif_name);
	return -1; 
    }

    /* Skip addresses of all 0's */
    for (i = 0; i < alen && !p[i]; i++) /* Empty */;

    if (i == alen)		/* Address is empty */
	perr(E_DBG4,"Hardware address for %s is empty.\n",pif->pif_name);
    else
    {
	pif->hwaddr_len = alen;
	memcpy( pif->hwaddr, p, alen );
	pif->pif_pflags |= PIF_HAS_HWADDR;
    }

    if ( ( sd = socket( AF_INET, SOCK_DGRAM, 0 ) ) < 0 )
	{ NETERR("if_get_hwparams_sdl()"); }

    if_get_mtu( sd, pif );

    close( sd );

    return 0;
}

/*----------------------------------------------------------------------*/
/* Set an interface's netmask						*/
/* This pretty tricky, too tricky for my taste, but anyway, here we go:	*/
/* - if we're using sysctl(), then for AF_INET, the sockaddr_dl struct 	*/
/*   we get here is a truncated one, having a length of 0, 5, 6, 7 or 8 */
/*   The family is undefined. 0 means a mask of 0.0.0.0, 5 means a mask */
/*   of %d.0.0.0, etc...						*/
/* - I am not sure of what this means for AF_INET6. As far as I know, 	*/
/*   AF_INET6 should not have netmasks.	It has 'prefix lengths', which, */
/*   we can calculate from the netmask that the sdl contains. Not sure 	*/
/*   how right or wrong thiis is....					*/
/*									*/
/* NOTE: pia->pia_family must be set prior to calling this. 		*/
/*----------------------------------------------------------------------*/
static int
if_set_netmask( PNetIfAddr *pia, SockAddr *sa )
{
    int mlen = 0;
    byte *s  = (byte*)&sa->sa_data[2];	/* Start of netmask portion	*/
    byte *t;

    /* Netmasks can have lengths of 0 (implied all 0 netmask),		*/
    /* or 5 through 8 (for IPv4), 					*/

    if ( pia->pia_family == AF_INET )
	mlen = sa->sa_len;
    else if ( pia->pia_family == AF_INET6 )
    {
	InetAddr6* in6 = (InetAddr6*) sa;
	s = (byte*)&in6->sin6_addr;
    }

    memset( pia->pia_netmask, 0, sizeof(pia->pia_netmask));
    t = &pia->pia_netmask[1];

    /* Remember, there's 4 leading bytes in a sockaddr_dl struct, if we
     * assume that it supports sa_len. Until we encounter a case where 
     * this is not so, this code won't get fixed
     */
    if (mlen > 4)
	memcpy(t,s,mlen - 4);

    /* Compute the prefix length: netmask must consist of a contiguous
     * run of 1s.
     */
    if ( pia->pia_family == AF_INET6 )
    {
	byte b = 0xFF;
	byte m = 0x01;

	pia->pia_prefixlen = 0;

	while ( ! (*s ^ b) )
	    pia->pia_prefixlen += 8, s++;

	/* Will the prefix length ever not be a multiple of 8?		*/
	/* Until sure, we need to collect some odd bits left over	*/
	/* in the last non-FF/00 byte.					*/

	if ( *s && pia->pia_prefixlen < 128 )
	    do
		pia->pia_prefixlen += m & *s, m <<= 1;
	    while ( m & *s );
    }

    if ( pia->pia_family == AF_INET )
	pia->pia_netmask[0] = 1; /* Trick flag to know if set 	*/

    return 0;
}

static size_t
next_boundary(size_t x,size_t s)
{
    register int r = x % s;

    x += r ? (s - r) : 0;

    return x;
}

int
get_ifs_info( int fam )
{
    byte *		pnext;
    byte *		end;
    struct pnet_if *	pif_curr;
    struct pnet_if **	ppif;
    byte *		buf;
    size_t		len = 0;
    PNetIfAddr **	ppia= 0;
# ifdef HAVE_SYSCTL
    int 	mib[6] = { CTL_NET, AF_ROUTE, 0, 0, NET_RT_IFLIST, 0 } ; 
# endif

    /* Free old interfaces info first */
    if (pif_head)
	free_ifs_info(pif_head);

# ifdef HAVE_SYSCTL
    if (fam != AF_UNSPEC )
	mib[3] = fam;
# endif

    /* Get size of buffer from kernel */
# ifdef HAVE_SYSCTL
    if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
	{ NETERR("get_ifs_info(): sysctl()"); return -1; }
# elif HAVE_GETKERNINFO
    len = getkerninfo( KINFO_RT_IFLIST, 0, 0, 0 );
    if( ((int)len) < 0 )
	{ NETERR("get_ifs_info(): getkerninfo()"); return -1; }
# endif

    LLOG(len);

    STDMALLOC(buf,len,-1);

# ifdef HAVE_SYSCTL
    if ( sysctl(mib, 6, buf, &len, NULL, 0) < 0 )
	{ NETERR("get_ifs_info(): sysctl()"); STDFREE(buf); return -1; }
# elif HAVE_GETKERNINFO
    if ( getkerninfo( KINFO_RT_IFLIST, buf, &len, 0 ) < 0 )
	{ NETERR("get_ifs_info(): getkerninfo()"); STDFREE(buf); return -1; }
# endif

    end = buf + len;
    pnext = buf;

    /* Initialize the head of the list */
    pif_curr = pif_head = NULL;
    ppif = &pif_head;

    while ( pnext < end )
    {
	struct if_msghdr*	ifm;

	/* Get next msg header */
	ifm = (struct if_msghdr*) pnext;

	if (ifm->ifm_type == RTM_IFINFO)
	{
	    /* Round 1, get hardware address and name of interface */
	    struct sockaddr_dl *	sdl;

	    sdl = (struct sockaddr_dl*) (ifm + 1);

	    STDMALLOC( pif_curr, sizeof(PNetIf), -1);

	    *ppif = pif_curr;
	    ppif = &pif_curr->next;

	    /* Remember address of first address pointer (no pun intended) */
	    ppia = &pif_curr->pif_addrs;

	    pif_curr->pif_flags = ifm->ifm_flags;

	    if (sdl->sdl_family == AF_LINK)
	    {
		/* interface name */
		char * p;

		if (sdl->sdl_nlen > 0)
		{
		    strncpy(pif_curr->pif_name,sdl->sdl_data,sdl->sdl_nlen);
		    pif_curr->pif_name[sdl->sdl_nlen] = 0;
		}
		else
		{
		    sprintf(pif_curr->pif_name,"index %d",sdl->sdl_index);
		}

		/* Remove semi-colon */
		if ( (p = strrchr( pif_curr->pif_name, ':' )) )
		    *p = 0;

		if_get_hwparams( pif_curr, sdl );
	    }
	    if_debug( pif_curr );
	}
	else if (ifm->ifm_type == RTM_NEWADDR)
	{
	    struct ifa_msghdr*	ifam;
	    SockAddr*		sa;
	    SockAddr*		sanm = 0;
	    PNetIfAddr*		pia;

	    ifam = (struct ifa_msghdr*) pnext;

	    sa = (SockAddr*)(ifam + 1);

	    /* Allocate an address structure, and link it up to the others
	     * for this interface 			*/

	    STDMALLOC( pia, sizeof( PNetIfAddr ), -1 );

	    *ppia = pia;
	    ppia  = &pia->pia_next;

	    pia->pia_if = pif_curr;
# if 0
if (ifam->ifam_addrs & RTA_DST) printf("dst addr present \n");
if (ifam->ifam_addrs & RTA_GATEWAY) printf("gw present\n");
if (ifam->ifam_addrs & RTA_NETMASK) printf("netmask present\n");
if (ifam->ifam_addrs & RTA_GENMASK) printf("genmask present\n");
if (ifam->ifam_addrs & RTA_IFP    ) printf("if name present\n");
if (ifam->ifam_addrs & RTA_IFA    ) printf("if addr present\n");
if (ifam->ifam_addrs & RTA_AUTHOR ) printf("author present\n");
if (ifam->ifam_addrs & RTA_BRD    ) printf("baddr present\n");
# endif

# define SKIP_SA(sa)	(SockAddr*)((caddr_t)sa + \
				next_boundary(sa->sa_len,sizeof(u_long)))

	    if (ifam->ifam_addrs & RTA_NETMASK)
	    {
		sanm = sa;
		SLOG( net_sockaddrtop( sa ) );
		sa = SKIP_SA( sa );
	    }

	    if (ifam->ifam_addrs & RTA_IFA)
	    {
		pia->pia_family = sa->sa_family;
		memcpy( &pia->pia_address, sa, sa->sa_len );
		pif_curr->pif_pflags |= PIF_HAS_ADDR;
		sa = SKIP_SA(sa);

		if ( pia->pia_family == AF_INET6 )
		{
		    /* For a link-local address, the 2 and 3 bytes contain */
		    /* the interface index. We need to remember it here	*/	
		    /* and clear the two bytes, since LL addresses have */
		    /* those bytes empty 				*/

		    if ( IN6_IS_ADDR_LINKLOCAL( &pia->pia_addr6.sin6_addr ) )
		    {
			pnet_ushort *pui = 
				(pnet_ushort*) &pia->pia_addr6.sin6_addr;

			pui++;
			pia->pia_scope_id = ntohs( *pui );

			*pui = 0;
			if ( pia->pia_scope_id == 0 )
			    pia->pia_scope_id = pif_curr->pif_index;
		    }
		}
	    }

	    if ( sanm )
		if_set_netmask( pia, sanm );

	    if (ifam->ifam_addrs & RTA_BRD)
	    {
		if (pif_curr->pif_flags & IFF_BROADCAST)
		{
		    memcpy( pia->pia_broadaddr + 1,
			    &((InetAddr*)sa)->sin_addr,
			    sizeof( struct in_addr ) );
		    pif_curr->pif_pflags |= PIF_HAS_BADDR;
		    pia->pia_broadaddr[0] = 1;
		}
	    /*}
	      if (ifam->ifam_addrs & RTA_DST)
	      { */
		else if (pif_curr->pif_flags & IFF_POINTOPOINT)
		{
		    if ( sa->sa_len )
		    {
			LLOG( sa->sa_len );
			memcpy(&pif_curr->pif_daddr,sa,sa->sa_len);
			pif_curr->pif_pflags |= PIF_HAS_DADDR;
		    }
		}
	    }
	}
	else
	{
	    perr(E_FATAL,"get_ifs_info(): error reading interface list\n");
	    STDFREE( buf );
	    return -1;
	}

	pnext += ifm->ifm_msglen;
    }

    STDFREE(buf);

    return 0;
}
# ifndef IFF_SMART
# define IFF_SMART	0
# endif
# ifndef IFF_MULTICAST
# define IFF_MULTICAST	0
# endif
int pnetIfIsUp( PNETIF pif ) { return pif->pif_flags & IFF_UP; }
int pnetIfIsBroadcast( PNETIF pif ) { return pif->pif_flags & IFF_BROADCAST; }
int pnetIfIsDebug( PNETIF pif ) { return pif->pif_flags & IFF_DEBUG; }
int pnetIfIsLoopback( PNETIF pif ) { return pif->pif_flags & IFF_LOOPBACK; }
int pnetIfIsPointopoint( PNETIF pif ) { return pif->pif_flags & IFF_POINTOPOINT; }
int pnetIfIsSmart( PNETIF pif ) { return pif->pif_flags & IFF_SMART; }
int pnetIfIsRunning( PNETIF pif ) { return pif->pif_flags & IFF_RUNNING; }
int pnetIfIsNoArp( PNETIF pif ) { return pif->pif_flags & IFF_NOARP; }
int pnetIfIsPromiscuous( PNETIF pif ) { return pif->pif_flags & IFF_PROMISC; }
int pnetIfIsSimplex( PNETIF pif ) { return pif->pif_flags & IFF_SIMPLEX; }
int pnetIfIsMulticast( PNETIF pif ) { return pif->pif_flags & IFF_MULTICAST; }
# endif 						/* } */

# ifdef HAVE_NET_IF_TYPES_H
#   include <net/if_types.h>

const char *
if_get_description( PNetIf * pif )
{
    static char buf[sizeof("type: ") + 11];

    switch ( pif->pif_type )
    {
    case IFT_OTHER: 		return "Other";
# ifdef IFT_ETHER
    case IFT_ETHER:		return "Ethernet CSMACD";
# endif
# ifdef IFT_ISO88023
    case IFT_ISO88023:		return "CMSA CD";
# endif
# ifdef IFT_IS88024
    case IFT_ISO88024:		return "Token Bus";
# endif
# ifdef IFT_ISO88025
    case IFT_ISO88025:		return "Token Ring";
# endif
# ifdef IFT_ISO88026
    case IFT_ISO88026:		return "MAN";
# endif
# ifdef IFT_P10
    case IFT_P10:		return "Proteon 10MBit ring";
# endif
# ifdef IFT_P80
    case IFT_P80:		return "Proteon 80MBit ring";
# endif
# ifdef IFT_HY
    case IFT_HY:		return "Hyperchannel";
# endif
# ifdef IFT_FDDI
    case IFT_FDDI:		return "FDDI";
# endif
# ifdef IFT_ISDNBASIC
    case IFT_ISDNBASIC:		return "Basic ISDN";
# endif
# ifdef IFT_ISDNPRIMARY
    case IFT_ISDNPRIMARY:	return "Primary ISDN";
# endif
# ifdef IFT_PPP
    case IFT_PPP:		return "PPP";
# endif
# ifdef IFT_LOOP
    case IFT_LOOP:		return "Loopback";
# endif
    case IFT_SLIP:		return "SLIP";
# ifdef IFT_PARA
    case IFT_PARA:		return "Parallel Port";
# endif
# ifdef IFT_PARALLEL
    case IFT_PARALLEL:		return "Parallel Port";
# endif
    case IFT_ARCNET:		return "ARCnet";
# ifdef IFT_ARCNETPLUS
    case IFT_ARCNETPLUS:	return "ARCnet Plus";
# endif
# ifdef IFT_ARCNET_PLUS
    case IFT_ARCNET_PLUS:	return "ARCnet Plus";
# endif
    case IFT_ATM:		return "ATM";
# ifdef IFT_HIPPI
    case IFT_HIPPI:		return "HiPPi";
# endif
# ifdef IFT_HPPI_I
    case IFT_HPPI_I:		return "HiPPi";
# endif
    case IFT_MODEM:		return "Modem";
# ifdef IFT_GIF
    case IFT_GIF:		return "Gif";
# endif
# ifdef IFT_FAITH
    case IFT_FAITH:		return "Faith";
# endif
    }

    sprintf(buf,"type: 0x%08X", pif->pif_type );

    return buf;
}

# endif
