/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: if-lifconf.c,v 1.15 2002/10/03 06:37:35 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            if-lifconf.c
 * created on:          Thu Aug  8 12:26:10 EDT 2002
 * created by:          peter
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Interface information.						*/
/*									*/
/* This one is using LIFCONF, for Solaris 5.8 I believe			*/
/*									*/
/*----------------------------------------------------------------------*/

# include "../local.h"

# ifdef STDSunOS				/* { */

# include "iflocal.h"

# define BSD_COMP		/* Solaris needs this */
# include <sys/ioctl.h>
# include <net/if.h>

# if defined HAVE_NET_IF_TYPES_H

#   include <net/if_types.h>

# endif

static int
if_set_netmask( PNetIfAddr *pia , SockAddr * sa)
{
    if ( pia->pia_family == AF_INET )
    {
	pia->pia_netmask[0] = 1; /* Trick flag to know if set 	*/
        memcpy( pia->pia_netmask + 1,
                &((InetAddr*)sa)->sin_addr.s_addr, sizeof( struct in_addr ) );
    }

    return 0;
}

static int
lif_get_hwparams( PNetIf *pif, int sd )
{
    struct lifreq	lifr;

    strncpy( lifr.lifr_name, pif->pif_name, IFNAMSIZ );

    if ( if_ioctl( sd, SIOCGLIFINDEX, &lifr ) )
	{ FATALERR("lif_get_hwparams()"); return -1 ; }

    pif->pif_index = lifr.lifr_index;

    strncpy( lifr.lifr_name, pif->pif_name, IFNAMSIZ );

    if ( if_ioctl( sd, SIOCGLIFMTU, &lifr ) )
	{ FATALERR("lif_get_hwparams()"); return -1 ; }

    pif->pif_mtu = lifr.lifr_mtu;

# ifdef SIOCGENADDR
    strncpy( lifr.lifr_name, pif->pif_name, IFNAMSIZ );

    if ( if_ioctl( sd, SIOCGENADDR, &lifr ) )
	{ perr(E_WARN,"lif_get_hwparams(): %s\n", neterr()); return 0; }

    pif->hwaddr_len = 6;
    memcpy( pif->hwaddr, lifr.lifr_enaddr, 6 );
    pif->pif_pflags |= PIF_HAS_HWADDR;

# endif

    return 0;
}
/* Get long interface info, Solaris style */
static int
get_lifs_info( int fam )
{
    byte *              pnext;
    byte *              end ;
    struct pnet_if *	pif_curr;
    struct pnet_if **	ppif;
    int                 sd;
    struct lifconf      lifc;
    struct lifnum	lifn;
    PNetIfAddr**	ppia = 0;

    DBG( dbg("get_lifs_info( fam = %d )\n", fam ) );

    /* Free old interfaces info first */
    if (pif_head)
	free_ifs_info(pif_head);

    if ( (sd = socket( AF_INET, SOCK_DGRAM, IPPROTO_IP )) < 0)
        { NETERR( "get_lifs_info()" );  return -1; }

    /* Get number of interfaces */

    lifn.lifn_family = AF_UNSPEC;
    lifn.lifn_flags  = 0 ;

    if ( if_ioctl( sd, SIOCGLIFNUM, &lifn ) )
	{ NETERR( "get_lifs_info()" ); return -1; }

    pdbg(E_INFO, "Found %d configured interfaces\n", lifn.lifn_count );

    /* Allocate "enough" space for the buffer.                          */

    lifc.lifc_len = lifn.lifn_count * sizeof(struct lifreq);

    lifc.lifc_flags = 0x0;
    lifc.lifc_family = fam;

    STDMALLOC( lifc.lifc_buf , lifc.lifc_len, -1 );

    pdbg(E_DBG4,"Trying ioctl(%d,SIOCGLIFCONF,...): "
		    "buflen = %d\n",sd,lifc.lifc_len);

    if ( if_ioctl( sd, SIOCGLIFCONF, &lifc ) )
	{ NETERR("get_lifs_info()"); close(sd); return -1; }

    pnext = (byte*)lifc.lifc_buf;
    end = pnext + lifc.lifc_len;

    /* Initialize the head of the list */
    pif_curr = pif_head = NULL;
    ppif = &pif_head;

    while ( pnext < end )
    {
        struct lifreq *  ifr = (struct lifreq*) pnext;
        struct lifreq    ifrtmp;
        u_int       	 len = 0;
	char *		 p;
        struct sockaddr *sa;
	PNetIfAddr*	 pia;

        /* Get the actual length of the IP address stored in the ifr    */
        sa = (struct sockaddr*) &ifr->lifr_addr;

	len = addr_len( sa->sa_family );

        /* Find start of next struct ifreq                              */
	pnext += sizeof( struct lifreq );

        /* Skip family if not matching requested family */
        if (fam != AF_UNSPEC && fam != sa->sa_family)
            continue;

        if ( (pif_curr = net_get_if_by_name_bootstrap(fam, ifr->lifr_name,1)))
        {
            pdbg(E_DBG1,"Interface %s seen already\n", ifr->lifr_name);

            pia = NULL;

            while ( (pia = net_if_get_next_address( pif_curr, pia )) )
                ppia = &pia->pia_next;
        }
	else
	{
	    /* Allocate a new pif struct and link up the pointers */

	    STDMALLOC(pif_curr,sizeof(PNetIf),-1);
	    *ppif = pif_curr;
	    ppif = &pif_curr->next;

	    ppia = &pif_curr->pif_addrs;

	    /* Save interface name and family. */
	    strncpy(pif_curr->pif_name,ifr->lifr_name,IFNAMSIZ);

	    /* Remove semi-colon */
	    if ( (p = strrchr( pif_curr->pif_name, ':' )) )
		*p = 0;

	    pif_curr->pif_family = sa->sa_family;

	    ifrtmp = *ifr;

	    /* Get interface flags */
	    if ( if_ioctl(sd,SIOCGLIFFLAGS,&ifrtmp) )
		{ NETERR("SIOCGLIFFLAGS"); }

	    pif_curr->pif_flags = ifrtmp.lifr_flags;

	    /* Get interface hardware address type and index */
	    if ( lif_get_hwparams(pif_curr,sd) )
	    {
		printf("Error reading hardware configuration for "
		       "interface %s\n",ifr->lifr_name);
	    }

	    /* Get unicast address */
    /*
	    if ( if_ioctl(sd, SIOCGLIFADDR, &ifrtmp) == 0 )
	    {
		memcpy(&pif_curr->pif_addr,&ifr->lifr_addr,len);
		pif_curr->pif_pflags |= PIF_HAS_ADDR;
	    }

	    if ( if_ioctl(sd, SIOCGLIFNETMASK, &ifrtmp) == 0 )
	    {
		if_set_netmask( pif_curr, (SockAddr*)&ifrtmp.lifr_addr );
	    }
    */


	    /* Get point-to-point destination address */

	    if (pif_curr->pif_flags & IFF_POINTOPOINT)
	    {
		if ( if_ioctl( sd, SIOCGLIFDSTADDR, &ifrtmp) == 0 )
		{
		    memcpy(&pif_curr->pif_daddr4,&ifrtmp.lifr_dstaddr, len );
		    pif_curr->pif_pflags |= PIF_HAS_DADDR;
		}
	    }
	}

        /* Now, finally, get the unicast address, and its netmask       */
        /* Allocate an address structure, and link it up to the others  */
        /* for this interface                                           */

        STDMALLOC( pia, sizeof( PNetIfAddr ), -1 );

        *ppia = pia;
	pia->pia_if = pif_curr;

	/* Get broadcast address */
	if ( pif_curr->pif_flags & IFF_BROADCAST )
	{
	    if ( if_ioctl(sd,SIOCGLIFBRDADDR,&ifrtmp) == 0 )
	    {
		memcpy( pia->pia_broadaddr + 1, 
			&((InetAddr*)&ifrtmp.lifr_broadaddr)->sin_addr,
			sizeof( struct in_addr ));
		pif_curr->pif_pflags |= PIF_HAS_BADDR;
		pia->pia_broadaddr[0] = 1;
	    }
	}

	if ( if_ioctl(sd, SIOCGLIFADDR, &ifrtmp) == 0 )
        {
            pia->pia_family     = (*((SockAddr*)&ifr->lifr_addr)).sa_family;

            memcpy( &pia->pia_address, &ifr->lifr_addr, len);

            pif_curr->pif_pflags |= PIF_HAS_ADDR;
        }

        if ( if_ioctl(sd, SIOCGLIFNETMASK, &ifrtmp) == 0 )
            if_set_netmask( pia, (SockAddr*)&ifrtmp.lifr_addr );

    }

    close(sd);
    free( lifc.lifc_buf );

    return 0;
}

int
get_ifs_info( int fam )
{
    return get_lifs_info( fam );

    return 0;
}

# ifndef IFF_SMART
#   define IFF_SMART		0
# endif

# ifndef IFF_SIMPLEX
#   define IFF_SIMPLEX		0
# endif

int pnetIfIsUp( PNETIF pif ) { return pif->pif_flags & IFF_UP; }
int pnetIfIsBroadcast( PNETIF pif ) { return pif->pif_flags & IFF_BROADCAST; }
int pnetIfIsDebug( PNETIF pif ) { return pif->pif_flags & IFF_DEBUG; }
int pnetIfIsLoopback( PNETIF pif ) { return pif->pif_flags & IFF_LOOPBACK; }
int pnetIfIsPointopoint( PNETIF pif ) { return pif->pif_flags & IFF_POINTOPOINT; }
int pnetIfIsSmart( PNETIF pif ) { return pif->pif_flags & IFF_SMART; }
int pnetIfIsRunning( PNETIF pif ) { return pif->pif_flags & IFF_RUNNING; }
int pnetIfIsNoArp( PNETIF pif ) { return pif->pif_flags & IFF_NOARP; }
int pnetIfIsPromiscuous( PNETIF pif ) { return pif->pif_flags & IFF_PROMISC; }
int pnetIfIsSimplex( PNETIF pif ) { return pif->pif_flags & IFF_SIMPLEX; }
int pnetIfIsMulticast( PNETIF pif ) { return pif->pif_flags & IFF_MULTICAST; }

# endif							/* } */
