/*
 * $Id: snoop6.c,v 1.16 2002/12/02 08:22:02 kingofgib Exp $
 *
 * A very simple network sniffer based on libpnet6.
 *
 * Link with libpnet6 and libpthread.
 */
# include <pnet6.h>
# include <pnet6pkt.h>
# include <pnet6tlp.h>

# include <stdio.h>
# include <string.h>
# include <stdlib.h>

/*----------------------------------------------------------------------*/
/* Forward declarations							*/
/*----------------------------------------------------------------------*/

static unsigned long	total_bytes = 0;  /* Total bytes seen on the wire */
static pnet_uint	total_out   = 0;  /* Total bytes that left us */
static pnet_uint	total_in    = 0;  /* Total bytes that came in */
static pnet_uint	grab_length = 92; /* #bytes to read per packet */
static pnet_uint	timeout     = 400; /* Milliseconds */

static PNET_PKTACCESS  	pa;	/* Device used for packet access */
static PNetPacket	pkt;	/* Holds the currently read packet */


/*----------------------------- (    9 lines ) --------------------------*/
static int icmp6_dump(pnet_byte *,int);
static int icmp_dump(pnet_byte *,int);
static int ip4_dump(pnet_byte *);
static int ip6_dump(int,pnet_byte *);
static void print_timestamp(void);
static void sig_handler(int);
static int tcp_dump(void *,pnet_byte *,int);
static int udp_dump(void *,pnet_byte *,int);

static void
print_timestamp( void )
{
    char buf[128];

    printf("(%s) ", pnetGetTimeStamp( buf, sizeof( buf ) ));
}

static int
icmp_dump( pnet_byte * buf, int len )
{
    pnet_icmp *icmp = (pnet_icmp*)buf;

    printf( "ICMP: %s, %s\n",
	pnetICMP_TypeDesc( PNET_IPPROTO_ICMP, icmp->icmp_type ),
	pnetICMP_CodeDesc( PNET_IPPROTO_ICMP, icmp->icmp_type, icmp->icmp_code ) );
/*    pnetHexdumpX( stdout, (pnet_byte*)icmp, len, 4, 16 );  */
    return 0;
}
static int
icmp6_dump( pnet_byte * buf, int len )
{
    pnet_icmp6 *icmp6 = (pnet_icmp6*)buf;

    printf( "ICMP6: %s, %s\n",
	pnetICMP_TypeDesc( PNET_IPPROTO_ICMPV6, icmp6->icmp6_type ),
	pnetICMP_CodeDesc( PNET_IPPROTO_ICMPV6, icmp6->icmp6_type, icmp6->icmp6_code ) );
    /* pnetHexdumpX( stdout, (pnet_byte*)icmp6, len, 4, 16 );  */
    return 0;
}
static int
tcp_dump( void * pip, pnet_byte * buf, int len )
{
    pnet_tcp * tcp = (pnet_tcp*) buf;
    pnet_byte * popts;
    int		optlen;

    if ( len < (int)sizeof( pnet_tcp ))
    {
	printf(" TCP--- truncated header\n" );
	return 0;
    }
    optlen = pnetTCP_OptsLen( tcp );
    popts  = pnetTCP_Opts( tcp );

    printf("TCP: hl=%d,optlen=%d,s=%u,d=%u,seq=%lu,ack=%lu,w=%u,c=0x%X",
	    pnetTCP_DataOffset(tcp),pnetTCP_OptsLen( tcp ),
	    pnetTCP_SrcPort(tcp),pnetTCP_DstPort(tcp),
	    pnetTCP_Seq(tcp),
	    pnetTCP_Ack(tcp),pnetTCP_Win(tcp),pnetTCP_Checksum(tcp));

    if ( tcp->th_flags )
    {
	printf(", f: ");
	printf("%c", tcp->th_flags & TH_CC ? 'C' : '-' );
	printf("%c", tcp->th_flags & TH_CE ? 'E' : '-' );
	printf("%c", tcp->th_flags & TH_URG ? 'U' : '-' );
	printf("%c", tcp->th_flags & TH_ACK ? 'A' : '-' );
	printf("%c", tcp->th_flags & TH_PUSH ? 'P' : '-' );
	printf("%c", tcp->th_flags & TH_RST ? 'R' : '-' );
	printf("%c", tcp->th_flags & TH_SYN ? 'S' : '-' );
	printf("%c", tcp->th_flags & TH_FIN ? 'F' : '-' );
    }

    if ( optlen > 0 )
	printf("\n");
    while ( optlen > 0 )
    {
	switch ( *popts )
	{
	case 0:	printf("eoo"); break;
	case 1:	printf("nop,"); break;
	case 2:
	    printf("mss: %d,", pnet_ntohs( *(pnet_ushort*)( popts + 2 )));
	    break;
	case 3:
	    printf("wsf: %d,", *(popts + 2));
	    break;
	case 4:
	    printf("sack,"); break;
	case 5:
	    printf("sack block,"); break;
	case 6:
	    printf("echo,"); break;
	case 7:
	    printf("echoreply,"); break;
	case 8:
	    printf("tstamp,"); break;
	case 9:
	    printf("partial order connection permitted,"); break;
	case 10:
	    printf("partial order service profile %d,",*(popts + 2));
	    break;
	case 11: printf("CC,"); break;
	case 12: printf("CC new,"); break;
	case 13: printf("CC echo,"); break;
	case 14:
	    printf("Alt. chksum req.=%d,",*(popts + 2));
	    break;
	case 15:
	    printf("Alt. chksum data,");
	    break;
	}
	if ( *popts < 2 )
	    popts++, optlen--;
	else 
	{
	    optlen -= *(popts + 1);
	    popts += *(popts + 1);
	}
    }
    printf("\n");

    /*
     * We can only compute the TCP checksum if the entire packet fits inside
     * our grab buffer. The length of the entire datagram is returned by
     * pnetIP_Length().
     */
    if ( grab_length > (pnet_uint) pkt.pkt_datalen )
    {
	pnet_byte * data = (pnet_byte*) tcp + pnetTCP_DataOffset(tcp);
	int    datalen = pnetIP_PayloadLength( pip ) - pnetTCP_DataOffset(tcp);

	pnet_ushort checksum = pnetTCP_ComputeChecksum( pip, tcp,
							data, datalen );

	if ( checksum != tcp->th_sum )
	    printf("\nTCP CHECKSUM ERROR: found %X, should be %X\n",
		    tcp->th_sum,checksum);
    }

    return 0;
}
static int
udp_dump( void * pip, pnet_byte * buf, int len )
{
    pnet_udp *udp = (pnet_udp *)buf;

    if ( len < (int)sizeof( pnet_udp ) )
    {
	printf( "UPD---truncated header\n" );
	return 0;
    }
    printf("UDP: s=%u,d=%u,l=%u,s=%X\n",
	    pnet_ntohs( udp->uh_sport ), pnet_ntohs( udp->uh_dport ),
	    pnet_ntohs( udp->uh_ulen  ), pnet_ntohs( udp->uh_sum   ));

    if ( grab_length > (pnet_uint) pkt.pkt_datalen )
    {
	pnet_byte * data = (pnet_byte*)udp + sizeof( pnet_udp );
	int    datalen = pnetIP_PayloadLength( pip ) - sizeof( pnet_udp );
	pnet_ushort cksum = pnetUDP_ComputeChecksum( pip, udp, data, datalen );

	if ( cksum != udp->uh_sum )
	    printf("\nUDP CHECKSUM ERROR: found %X, should be %X\n",
		    udp->uh_sum,cksum);

    } 
    return 0;
}

static int
ip6_dump( int ts, pnet_byte * buf )
{
    PNETADDRSTORAGE pas1;
    PNETADDRSTORAGE pas2;
    char addr[ PNET_ADDR_BUFSIZ ];
    pnet_ip6 * ip6= (pnet_ip6*)buf;
    int len = pnetIP_PayloadLength( ip6 );
    int proto ;

    if ( len > (int) grab_length )
	len = grab_length ;
    pnetIP_Source( ip6, PNET_SADDR( pas1 ) );
    pnetIP_Destination( ip6, PNET_SADDR( pas2 ) );
    proto =  pnetIP_Protocol( ip6 );

    if ( ts )
	print_timestamp();

    printf("IP6: tc=%d, f=%d, l=%d, p=%d, hop=%d, %s -> ",
	    pnetIP6_TrafficClass( ip6 ), pnetIP6_Flow( ip6 ),
	    pnetIP_PayloadLength( ip6 ), proto,
	    pnetIP_Hoplimit( ip6 ),
	    pnet_ntop( PNET_SADDR(pas1), addr, sizeof( addr )) );
    printf("%s\n", pnet_ntop( PNET_SADDR(pas2), addr, sizeof( addr ) ) );

    if ( proto == PNET_IPPROTO_ICMPV6 )
	return icmp6_dump( pnetIP_Payload( ip6 ), len );
    else if ( proto == PNET_IPPROTO_TCP )
	return tcp_dump( ip6, pnetIP_Payload( ip6 ), len );
    else if ( proto == PNET_IPPROTO_UDP )
	return udp_dump( ip6, pnetIP_Payload( ip6 ), len );
    return 0;
}
static int
ip4_dump( pnet_byte *buf )
{
    PNETADDRSTORAGE pas1;
    PNETADDRSTORAGE pas2;
    char addr[ PNET_ADDR_BUFSIZ ];
    pnet_ip * ip= (pnet_ip*)buf;
    pnet_ushort	cksum = 0;
    int proto ;
    int len = pnetIP_PayloadLength(ip);

    if ( len > (int) grab_length )
	len = grab_length;

    if ( (cksum = pnetIP4_ComputeChecksum( ip )) != ip->ip_sum )
    {
	printf("WARN: IP checksum wrong: found %X, should be %X\n",
		cksum, ip->ip_sum );
		
    }
    proto =  pnetIP_Protocol( ip );

    pnetIP_Source( ip, PNET_SADDR( pas1) );
    pnetIP_Destination( ip, PNET_SADDR( pas2 ) );

    total_bytes += pnetIP_Length( ip );
    print_timestamp();
    printf("IP4 (%d): %d, %s -> ",pnetIP_HeaderLength( ip ),
           pnetIP_PayloadLength( ip ),
	   pnet_ntop( PNET_SADDR( pas1 ), addr, sizeof( addr )));
    printf("%s: ", pnet_ntop( PNET_SADDR(pas2) , addr, sizeof( addr )) );
    printf("id=%x,mf=%d,df=%d,off=%d,ttl=%d,p=%d,s=0x%X\n",
	    pnetIP4_Id( ip ), 
	    pnetIP4_MoreFrags( ip ), pnetIP4_DontFrag( ip ),
	    pnetIP4_FragOffset( ip ), pnetIP_TTL( ip ),
	    pnetIP_Protocol( ip ), pnetIP4_Checksum( ip ) );
    if ( proto == PNET_IPPROTO_ICMP )
        return icmp_dump( pnetIP_Payload( ip ), len );
    else if ( proto == PNET_IPPROTO_TCP )
        return tcp_dump( ip, pnetIP_Payload( ip ), len );
    else if ( proto == PNET_IPPROTO_UDP )
        return udp_dump( ip, pnetIP_Payload( ip ), len );
    else if ( proto == 41 )		/* IPv6-over-IPv4 */
	return ip6_dump( 0, pnetIP_Payload( ip ) );

    return 0;
}
/*----------------------------------------------------------------------*/
/* The following struct desribes a packet access format/device.		*/
/*----------------------------------------------------------------------*/

# include <signal.h>
/* Our signal handler, called when a signal is caught */

static void
sig_handler( int sig )
{
    if ( sig == SIGINT )
    {
	pnet_uint got,dropped;

	pnetPktGetStats( pa, &got, &dropped );
	printf("Packets received = %d, dropped = %d\n",got,dropped );
	printf("Total bytes received = %lu", total_bytes);
	if ( total_bytes > 1024 )
	    printf(" (%lu Kb)", total_bytes / 1024 );
	if ( total_bytes > 1024*1024 )
	    printf(" (%lu Mb)", total_bytes / (1024*1024) );
	printf("\n");
	pnetPktCloseInterface( pa );

	exit( 0 );
    }

    return;
}

int
main( int argc, const char ** argv )
{
    const char * 	ifname = "eth0";	/* Default interface */
    int			n = 1000;
    int 		c;
    int			raw_mode = 0;
    int			link_frame_len = 0;
    const char *	expr= "";	/* empty exrpression means all pkts */


    pnetInit();
    /* Install our sighandler for SIGINT */
    pnetSetSighandler( SIGINT, sig_handler );

    while ( (c = pnetGetopt( argc, argv, ":I:rle:g:T:" )) != -1 )
    {
	switch ( c )
	{
	case 'l':
	    {
		/* List all interfaces and exit */
		PNETIF pif = pnetGetIfInfo( PNET_IPv4 );

		printf("Available interfaces:\n");
		while ( pif )
		{
		    printf(" %s, %d (%s), %s\n",
			pnetIfName( pif ),
			pnetIfIndex( pif ),
			pnetIfTypeString( pif ),
			pnetIfIsRunning( pif ) ? "running" : "not running" ) ;
		    pif = pnetIfNext( pif );
		}
	    }
	    return 0;
	case 'I':
	    ifname = strdup( pnetOptarg );
	    break;

	case 'T':
	    timeout = atoi( pnetOptarg );
	    break;

	case 'g':
	    grab_length = atoi( pnetOptarg );
	    break;

	case 'e':
	    expr = strdup( pnetOptarg );
	    break;

	case 'r':
	    raw_mode = 1;
	    break;

	case '!':
            fprintf(stderr,"Option '%c' requires an argument\n", pnetOptopt );
            return 1;

        case '?':
            fprintf( stderr, "Unknown argument '%c'\n", pnetOptopt );
            return 1;
	}
    }

    /* 
     * Open a packet access interface, set to promiscuous mode if desired,
     * the given timeout, and the given grab length
     */

    if ( !(pa = pnetPktOpenInterface( ifname, 1, timeout, grab_length )) )
    {
	printf("Cannot open interface %s for packet access\n",ifname);
	return 1;
    }

    if ( pnetPktAddFilter( pa, expr ) )
    {
	fprintf( stderr, "Failed to install packet filter.\n" );
	return 1;
    }

    printf("Watching for ``%s'' packets on iface %s, using %s frames\n", 
		expr, ifname, raw_mode ? "raw" : "cooked" );
    printf("(timeout = %d msec, grab_length=%d)\n", timeout, grab_length );

    pnetPktSetReadMode( pa, raw_mode ? PNET_READ_RAW : PNET_READ_COOKED );
    pnetPktSetReadTimeout( pa, timeout );

    memset( &pkt, 0, sizeof( pkt ) );

    if ( raw_mode )
    {
	pnetPktLLLength( pa, &link_frame_len );
	printf( "Link frame length is %d bytes\n", link_frame_len );
    }

    while ( n )
    {
	if ( pnetPktNextPacket( pa, &pkt ) )
	{
	    /* If reading raw frames, skip the link header here */
	    if ( raw_mode )
		pkt.pkt_buf += link_frame_len;

	    if ( (pkt.pkt_buf[0] & 0xF0) == 0x60 )
	    {
		ip6_dump( 1, pkt.pkt_buf );
		continue;
	    }
	    if ( (pkt.pkt_buf[0] & 0xF0) == 0x40 )
	    {
		ip4_dump( pkt.pkt_buf );
		continue;
	    }
	    pnetHexdump( stdout, (pnet_byte*)pkt.pkt_buf, pkt.pkt_grablen , 0 );
	}
    }

    sig_handler( SIGINT );
    pnetPktCloseInterface( pa );

    return 0;
}
