/* -*- pftp-c -*- */
/* str.c - String helper functions */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDARG_H
# include <stdarg.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_CTYPE_H
# include <ctype.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include "str.h"

#ifndef safe_free
# define safe_free(x) { if (x) { free(x); x = NULL; } }
#endif

char *realloc_strncpy(char *str, const char *newstr, size_t len)
{
    if (newstr == NULL) {
	if (len == 0)
	    return str;
	
	safe_free(str);
	return NULL;
    }
    
    str = realloc(str, len + 1);
    memcpy(str, newstr, len);
    str[len] = '\0';

    return str;
}

char *realloc_strncat(char *str, const char *newstr, size_t len)
{
    size_t mylen;

    if (newstr == NULL) {
	if (len == 0)
	    return str;
	
	return str;
    }

    if (str == NULL)
	return realloc_strncpy(str, newstr, len);

    mylen = strlen(str);
    str = realloc(str, mylen + len + 1);
    memcpy(str + mylen, newstr, len);
    str[mylen + len] = '\0';
    
    return str;
}

void unbackslash(char *str)
{
    char *pos = NULL;
    size_t str_len = 0;
    
    if (str == NULL)
	return;

    pos = strchr(str, '\\');
    str_len = strlen(str);

    while (pos != NULL) {
	switch (pos[1]) {
        case 'n':
            pos[0] = '\n';
	    break;
        case 't':
            pos[0] = '\t';
	    break;
        case 'r':
            pos[0] = '\r';
	    break;
        case '\a':
            pos[0] = '\a';
	    break;
        default:
            pos[0] = pos[1];
        }
      
	memcpy(pos + 1, pos + 2, str_len - ((pos - str) + 1));
	str_len--;
	pos = strchr(pos + 1, '\\');
    }
}

size_t split(const char *str, const char *needle, char ***part, ...)
{
    char **dont = NULL;
    size_t donts = 0, *dont_len = NULL, needle_len = strlen(needle);
    size_t parts = 0, d;
    const char *start1 = str, *start2 = str, *pos = NULL;
    va_list list;
    
    va_start(list, part);
    
    while ((pos = va_arg(list, const char *))) {
	dont = realloc(dont, (donts + 1) * sizeof(char *));
	dont_len = realloc(dont_len, (donts + 1) * sizeof(size_t));
	dont[donts] = alloc_strcpy(pos);
	dont_len[donts] = strlen(dont[donts]);
	donts++;
    }
    
    for (;;) {
	d = 0;
	while (d < donts) {
	    if (strncmp(start2, dont[d], dont_len[d]) == 0) {
		if ((pos = strstr(start2 + dont_len[d], dont[d]))) {
		    start2 = pos + dont_len[d];
		    d = 0;
		    continue;
		} else {
		    start2 = start2 + strlen(start2);
		    break;
		}
	    } else {
		d++;
	    }
	}
	
	if ((pos = strstr(start2, needle))) {
	    if (pos - start1 > 0) {
		(*part) = realloc((*part), (parts + 1) * sizeof(char *));
		(*part)[parts] = alloc_strncpy(start1, pos - start1);
		parts++;
	    }

	    start1 = pos + needle_len;
	    while (strncmp(start1, needle, needle_len) == 0)
		start1 += needle_len;
	    start2 = start1;
	} else {
	    break;
	}
    }
    
    if (start1[0]) {
	(*part) = realloc((*part), (parts + 1) * sizeof(char *));
	(*part)[parts] = alloc_strcpy(start1);
	parts++;
    }
    
    for (d = 0; d < donts; d++)
	free(dont[d]);
    
    safe_free(dont);
    safe_free(dont_len);
    
    return parts;
}

void free_split(char ***part, size_t *parts)
{
    size_t c;
    
    if ((*part) != NULL) {
	for (c = 0; c < (*parts); c++)
	    safe_free((*part)[c]);
	
	free((*part));
	(*part) = NULL;
    }

    (*parts) = 0;
}

char *unsplit(const char **str, size_t str_count, const char *seporator)
{
    char *ret = NULL;
    size_t c = 0;
    size_t ret_len = 0, str_len = 0, sep_len;
    
    sep_len = strlen(seporator);
    
    for (c = 0; c < str_count; c++) {
	if (str[c] != NULL) {
	    str_len = strlen(str[c]);
	    ret = realloc(ret, ret_len + str_len + sep_len + 1);
	    memcpy(ret + ret_len, str[c], str_len);
	    ret_len += str_len;
	} else {
	    ret = realloc(ret, ret_len + sep_len + 1);
	}
	
	memcpy(ret + ret_len, seporator, sep_len);
	ret_len += sep_len;
    }
    
    if (ret != NULL)
	ret[ret_len - sep_len] = '\0';

    return ret;
}

int is_whitespace(char c)
{
    return (c == ' ' || c == '\t' || c == '\n' || c == '\r');
}

void trim_left(char *str)
{
    char *s;

    if (str == NULL)
	return;

    s = str;

    while (is_whitespace(*s))
	s++;

    memmove(str, s, strlen(s) + 1);
}

void trim_right(char *str)
{
    size_t str_len = 0;
    
    if (str == NULL)
	return;
    
    str_len = strlen(str);
    
    while (str_len > 0 && is_whitespace(str[str_len - 1]))
	str_len--;

    str[str_len] = '\0';    
}

void trim(char *str)
{
    char *start, *end;

    start = str;
    end = str + strlen(str);

    while (is_whitespace(*start))
	start++;

    while (end > start && is_whitespace(*(end-1)))
	end--;

    memmove(str, start, end - start);
    str[end - start] = '\0';
}

char *alloc_strreplace(const char *body, const char *needle, 
		       const char *new)
{
    size_t str_len = 0, new_len = 0, needle_len = 0;
    const char *lastpos = body;
    char *pos = NULL;
    char *str = NULL;
    
    if (body == NULL || needle == NULL || new == NULL)
	return NULL;

    pos = strstr(body, needle);
    new_len = strlen(new);
    needle_len = strlen(needle);

    while (pos != NULL) {
	str = realloc(str, str_len + (pos - lastpos) + new_len + 1);
	memcpy(str + str_len, lastpos, pos - lastpos);
	str_len += pos - lastpos;
	memcpy(str + str_len, new, new_len);
	str_len += new_len;
	str[str_len] = '\0';
	
	pos += needle_len;
	lastpos = pos;
	pos = strstr(lastpos, needle);
    }
    
    if (strlen(lastpos) > 0) {
	str = realloc(str, str_len + strlen(lastpos) + 1);
	strcpy(str + str_len, lastpos);
    }
    
    return str;
}

char *realloc_strreplace(char *str, const char *needle, const char *new)
{
    char *ret = NULL;
    
    ret = alloc_strreplace(str, needle, new);
    str = realloc_strcpy(str, ret);
    
    if (ret != NULL)
	free(ret);

    return str;
}

char *strtolower(char *str)
{
    size_t p = 0;
    if (!str) return NULL;
    
    while (str[p]) {
	if (xisupper(str[p])) str[p] = tolower(str[p]);
	p++;
    }
    
    return str;
}

char *strtoupper(char *str)
{
    size_t p = 0;
    if (!str) return NULL;

    while (str[p]) {
	if (xislower(str[p])) str[p] = toupper(str[p]);
	p++;
    }

    return str;
}

char *realloc_strcat(char *str, const char *newstr)
{
    return realloc_strncat(str, newstr, (newstr == NULL) ? 0 : strlen(newstr));
}

char *realloc_strcpy(char *str, const char *newstr)
{
    return realloc_strncpy(str, newstr, (newstr == NULL) ? 0 : strlen(newstr));
}

char *alloc_strncpy(const char *newstr, size_t len)
{
    return realloc_strncpy(NULL, newstr, len);
}

char *alloc_strcpy(const char *newstr)
{
    return alloc_strncpy(newstr, (newstr == NULL) ? 0 : strlen(newstr));
}
