/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_CTYPE_H
# include <ctype.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include <pftp_settings.h>

#if !HAVE_GETLINE
# include "getline.h"
#endif
#include "str.h"
#include "parse_utils.h"
#include "pftp_getconf.h"
#include "util.h"
#include "hosts_settings.h"

/* Set default configuration settings and return.
 * Secure connections will be used if possible as
 * default.
 */
static pftp_conf_t conf_defaults = NULL;

/* Init default settings. */
static void init_default(void);
static void free_default(void);

static pftp_conf_t parse_conf_file(FILE *file, const char *file_name);

/* Get configuration for the client. First look at ~/.pftprc
 * then /etc/pftprc and as last way out resort to configuration
 * defaults.
 */
pftp_conf_t pftp_get_conf(void)
{
#ifdef WIN32
    const char *homedrive = getenv("HOMEDRIVE");
    const char *homepath = getenv("HOMEPATH");
#else
    const char *home = getenv("HOME");
#endif
    char *conf = NULL;
    FILE *file = NULL;
    pftp_conf_t ret;
    init_default();
#ifdef WIN32
    if (homepath != NULL && homedrive != NULL) {
        conf = alloc_strcpy(homedrive);
        conf = realloc_strcat(conf, homepath);
	conf = realloc_strcat(conf, "\\.pftprc");
#else
    if (home != NULL) {
        conf = alloc_strcpy(home);
    	conf = realloc_strcat(conf, "/.pftprc");
#endif
	file = fopen(conf, "rb");
    }
#ifndef WIN32
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, checking /etc/pftprc.",
		   conf, strerror(errno));
	free(conf);
	conf = alloc_strcpy("/etc/pftprc");
	file = fopen(conf, "rb");
    }
#endif
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, using defaults.",
		   conf, strerror(errno));
	free(conf);
	return conf_defaults;
    }
    ret = parse_conf_file(file, conf);
    fclose(file);
    if (!strcmp(conf, "/etc/pftprc")) {
	free(conf);
	return ret;
    } else if (ret == conf_defaults) {
	free(conf);
#ifndef WIN32
	conf = alloc_strcpy("/etc/pftprc");
	file = fopen(conf, "rb");
	if (file != NULL) {
	    free(conf);
	    return parse_conf_file(file, conf);
	} else {
	    util_error("Couldn't open file %s: %s, using defaults.",
		       conf, strerror(errno));
	    free(conf);
#endif
	    return conf_defaults;
#ifndef WIN32
	}
#endif
    } else {
	free(conf);
	return ret;
    }
    free(conf);
    return conf_defaults;
}

void init_default()
{
    conf_defaults = malloc(sizeof(struct pftp_conf_s));
    memset(conf_defaults, 0, sizeof(struct pftp_conf_s));
    conf_defaults->ftp = newDefault();
    conf_defaults->ftp->secure_list = 1;
    conf_defaults->ftp->secure_data = 1;
    conf_defaults->ftp->use_secure = 1;
}

static void parse_conf_row(char *line, pftp_conf_t conf, int line_count,
			   const char *file_name)
{
    int valid = 0;
    char *name, *value;

    if (split_conf_row(line, &name, &value)) {
	util_error("%s:%d: Invalid line format %s", 
		   file_name, line_count, line);
	return;
    }

    if (!strcmp(name, "PASV")) {	
	if (!(set_yes_or_no(value, &(conf->ftp->pasv)))) {
	    return;
	}
    } else if (!strcmp(name, "PORT_START")) {
	long port_start	= parse_port(value, &valid);
	if (valid) {
	    conf->ftp->port_start = (uint16_t) port_start;
	    return;
	}
    } else if (!strcmp(name, "PORT_STOP")) {
	long port_stop = parse_port(value, &valid);
	if (valid) {
	    conf->ftp->port_stop = (uint16_t) port_stop;
	    return;
	}
    } else if (!strcmp(name, "IF")) {
	if (conf->ftp->bind_to) free(conf->ftp->bind_to);

	if (check_interface(value)) {
	    util_error("%s:%d: Invalid interface `%s'", file_name, line_count, 
		       value);   
	} else {
	    conf->ftp->bind_to = strdup(value);
	}

	return;
    } else if (!strcmp(name, "OUTWARD_IP")) {
	if (conf->ftp->myaddr) free(conf->ftp->myaddr);
	
	if (check_ip(value)) {
	    conf->ftp->myaddr = NULL;
	    util_error("%s:%d: Invalid outward ip `%s'", file_name, line_count,
		       value);   
	} else {
	    conf->ftp->myaddr = strdup(value);
	}

	return;
    } else if (!strcmp(name, "PRIO_LIST")) {
	free_split(&conf->ftp->prio_list, &conf->ftp->prio_list_len);
	conf->ftp->prio_list_len = split(value, " ", 
					 &conf->ftp->prio_list, NULL);
	if (check_list((const char **)conf->ftp->prio_list, 
		       conf->ftp->prio_list_len)) {
	    free_split(&conf->ftp->prio_list, 
		       &conf->ftp->prio_list_len);
	    util_error("%s:%d: Invalid priority list `%s'", file_name, 
		       line_count, value);
	}
	return;
    } else if (!strcmp(name, "IGNORE_LIST")) {
	free_split(&conf->ftp->ignore_list, 
		   &conf->ftp->ignore_list_len);
	conf->ftp->ignore_list_len = split(value, " ", 
					   &conf->ftp->ignore_list,
					   NULL);
	if (check_list((const char **)conf->ftp->ignore_list, 
		       conf->ftp->ignore_list_len)) {
	    free_split(&conf->ftp->ignore_list, 
		       &conf->ftp->ignore_list_len);
	    util_error("%s:%d: Invalid ignore list `%s'", file_name, 
		       line_count, value);
	}
	return;
    } else if (!strcmp(name, "USE_SECURE")) {
	if (!(set_yes_or_no(value, &(conf->ftp->use_secure))))
	    return;
    } else if (!strcmp(name, "SECURE_LIST")) {
	if (!(set_yes_or_no(value, &(conf->ftp->secure_list))))
	    return;
    } else if (!strcmp(name, "SECURE_DATA")) {
	if (!(set_yes_or_no(value, &(conf->ftp->secure_data))))
	    return;
    } else if (!strcmp(name, "IMPLICIT_SSL")) {
	if (!(set_yes_or_no(value, &(conf->ftp->implicid_ssl))))
	    return;
    } else if (!strcmp(name, "GUI_WIDTH")) {
	unsigned long tmp = parse_ulong(value, &valid);
	if (valid) {
	    conf->gui_width = tmp;
	    return;
	}
    } else if (!strcmp(name, "GUI_HEIGHT")) {
	unsigned long tmp = parse_ulong(value, &valid);
	if (valid) {
	    conf->gui_height = tmp;
	    return;
	}
    } else if (!strcmp(name, "HORIZONTAL_PANEL1_POS")) {
	unsigned long tmp = parse_ulong(value, &valid);
	if (valid) {
	    conf->horzpan1 = tmp;
	    return;
	}
    } else if (!strcmp(name, "HORIZONTAL_PANEL2_POS")) {    
	unsigned long tmp = parse_ulong(value, &valid);
	if (valid) {
	    conf->horzpan2 = tmp;
	    return;
	}
    } else if (!strcmp(name, "PAGER")) {    
	conf->pager = realloc_strcpy(conf->pager, value);
	return;
    } else if (line[0] == '\0') {
	return;
    }
    util_error("%s:%d: Error parsing %s", file_name, line_count, line);
}

static void conf_cpy(pftp_conf_t DEST, pftp_conf_t SOURCE)
{
    memcpy(DEST, SOURCE, sizeof(struct pftp_conf_s));
    DEST->ftp = newDefault();
    cpyDefault(DEST->ftp, SOURCE->ftp);
}

void pftp_free_conf(pftp_conf_t conf)
{
    if (conf) {
	if (conf->ftp)
	    freeDefault(&conf->ftp);
	free(conf);
	if (conf == conf_defaults)
	    conf_defaults = NULL;	
    }
}

void free_default(void) 
{
    pftp_free_conf(conf_defaults);
}

pftp_conf_t parse_conf_file(FILE *file, const char *file_name)
{
    char *line = NULL;
    int line_count = 0;
    size_t N = 0;
    int first_line_ok = -1;
    pftp_conf_t ret =  malloc(sizeof(struct pftp_conf_s));
    conf_cpy(ret, conf_defaults);

    if (!feof(file)) {
	while (first_line_ok == -1) {
	    line_count++;
	    if (getline(&line, &N, file) == -1) {
		util_error("%s: Error reading config file.", file_name);
		if (line) free(line);
		pftp_free_conf(ret);
		ret = conf_defaults;
		return ret;
	    }
	    trim(line);
	    if (line[0] == '#') {
		continue;
	    } else if (!strncmp(line, "[pftp]", 6)) {
		break;
	    } else {
		util_error("%s: Invalid config file (doesn't begin with [pftp]).", 
			   file_name);
		if (line) free(line);
		pftp_free_conf(ret);
		ret = conf_defaults;
		return ret;
	    }
	}
    }
    while (!feof(file)) {
	line_count++;
	if (getline(&line, &N, file) == -1)
	    break;
	trim(line);
	if (line[0] == '#' || line[0] == '\0') {
	    continue;
	} else {
	    parse_conf_row(line, ret, line_count, file_name);
	}
    }

    if (line) free(line);
    return ret;
}

void pftp_close_conf(void)
{
    free_default();
}
