/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_LIMITS_H
# include <limits.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_FNMATCH_H
# include <fnmatch.h>
#endif
#if HAVE_STDARG_H
# include <stdarg.h>
#endif
#if HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_DIRENT_H
# include <dirent.h>
#else
# if HAVE_NDIR_H
#  include <ndir.h>
# else
#  if HAVE_SYS_DIR_H
#   include <sys/dir.h>
#  else
#   if HAVE_SYS_NDIR_H
#    include <sys/ndir.h>
#   endif
#  endif
# endif
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#if HAVE_PWD_H
# include <pwd.h>
#endif
#if HAVE_GRP_H
# include <grp.h>
#endif
#if HAVE_UTIME_H
# include <utime.h>
#endif
#if HAVE_SYS_UTIME_H
# include <sys/utime.h>
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif


#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include <pftp.h>

#include "pftp_item.h"
#include "pftputil.h"
#include "str.h"
#include "pftp_client.h"
#include "util.h"

#ifndef LLONG_MAX
# define LLONG_MAX LONG_LONG_MAX
#endif

#if HAVE_TIMEGM
#else
static time_t timegm(struct tm *tm)
{
    time_t ret;
    char *oldtz, *tmp;
    tmp = getenv("TZ");
    if (tmp)
	oldtz = strdup(tmp);
    else
	oldtz = NULL;
    setenv("TZ", "UTC", 1);
    tzset();
    ret = mktime(tm);
    if (oldtz) {
	setenv("TZ", oldtz, 1);
	free(oldtz);
    } else {
#if HAVE_UNSETENV
	unsetenv("TZ");
#else
	putenv("TZ");
#endif
    }
    tzset();
    
    return ret;
}
#endif

/* Local type used by begin_trans */
typedef struct {
    size_t count;
    char **path;
    pftp_server_t *ftp;
} clear_cache_t;

/* Function from libutil.c */
int util_comm_central(pftp_msg_t msg, pftp_server_t ftp, void *userdata, 
		      pftp_param_t param1, pftp_param_t param2);

/* Local functions */
#ifdef WIN32
static void get_user_groupname(const char *filename, char **user, char **group);
#else
static void get_username(uid_t uid, char **name);
static void get_groupname(gid_t gid, char **name);
#endif
static pftp_server_t _getFTP(const char *name);
static void add_curdir_to_clear_cache(clear_cache_t *clear_cache, 
				      pftp_server_t ftp);
static void sort_files(pftp_file_t *files, size_t len);

static int trans_cd(const char *ftp_name, const char *path) 
{
    if (!ftp_name) {
	int ret;
	if ((ret = chdir(path))) {
	    util_error("Error: %s.", strerror(errno));
	}
	return ret;
    } else { 
	pftp_server_t ftp = _getFTP(ftp_name);
	
	return (ftp ? pftp_cd(ftp, path, NULL) : -1);
    }
}

static int trans_rm(const char *ftp, const char *filename)
{
    if (ftp) {
	pftp_server_t _ftp;
	_ftp = _getFTP(ftp);
	if (_ftp) {
	    if (!pftp_rm(_ftp, filename)) {
		return 0;
	    }
	}
	
	return -1;
    } else {
	if (unlink(filename)) {
	    util_error("Unable to remove source file `%s': %s.",
		       filename, strerror(errno));
	    return -1;
	} else {
	    return 0;
	}
    }
}

static int trans_rmdir(const char *ftp, const char *filename)
{
    if (ftp) {
	pftp_server_t _ftp;
	_ftp = _getFTP(ftp);
	if (_ftp) {
	    if (!pftp_rmdir(_ftp, filename)) {
		return 0;
	    }
	}
	
	return -1;
    } else {
	if (rmdir(filename)) {
	    util_error("Unable to remove source directory `%s': %s.",
		       filename, strerror(errno));
	    return -1;
	} else {
	    return 0;
	}
    }
}

static pftp_directory_t *inter_ls(pftp_server_t ftp, const char *filemask, 
				  int hidden)
{
    pftp_directory_t *Cdir = malloc(sizeof(pftp_directory_t));
    memset(Cdir, 0, sizeof(pftp_directory_t));    
    if (pftp_ls(ftp, Cdir, 0, NULL) == 0) {
	pftp_apply_mask(Cdir, filemask, hidden);
	return Cdir;
    }
    
    pftp_free_fdt((*Cdir));
    free(Cdir);
    return NULL;
}

static pftp_file_type_t parse_file_type(mode_t M, int *link)
{
    if (link) {
	if(S_ISLNK(M)) {
	    (*link) = 1;
	    return pft_unknown;
	}
	(*link) = 0;
    }
    if(S_ISDIR(M))
	return pft_directory;
    if(S_ISCHR(M))
	return pft_character;
    if(S_ISBLK(M))
	return pft_block;
    if(S_ISREG(M))
	return pft_file;
    if(S_ISFIFO(M))
	return pft_fifo;
    if(S_ISSOCK(M))
	return pft_s_socket;
    
    return pft_unknown;      
}

#ifndef WIN32
static char *readlink_malloc(const char *filename)
{
    size_t size = 100;
    int ret;
    char *buffer = NULL;
    
    for (;;) {
	buffer = realloc(buffer, size);
	ret = readlink(filename, buffer, size);
	if (ret < 0) {
	    free(buffer);
	    return NULL;
	}
	if (ret < size) {
	    buffer[ret] = '\0';
	    return buffer;
	}
	
	size *= 2;
    }
}
#endif

static pftp_directory_t *local_dir_parse(char *dir, char **out_curpath)
{
    pftp_file_t file;
    struct dirent *d;
    struct stat64 buf;
    struct tm tm, *tmret;
    size_t base_len, tmp_len;
    char *tmp = NULL, *base = NULL;
    DIR *dstream;
    pftp_directory_t *res;
    
    if (dir[0] == '/') {
	base = strdup(dir);
	base_len = strlen(base);
	if (base_len > 1 && base[base_len - 1] == '/')
	    base[--base_len] = '\0';
    } else {
	char *curpath = NULL;
	size_t c_len, d_len;           
	
	if (pftp_get_path_dir(NULL, &curpath))
	    return NULL;
	
	c_len = strlen(curpath);
	d_len = strlen(dir);
	base_len = c_len + 1 + d_len;
	base = malloc(base_len + 1);
	memcpy(base, curpath, c_len);
	memcpy(base + c_len, "/", 1);
	memcpy(base + c_len + 1, dir, d_len + 1);

	pftp_compactPath(&base);

	base_len = strlen(base);
	
	free(curpath);
    }
    
    tmp = malloc(base_len + 2);
    memcpy(tmp, base, base_len + 1);
    if (base_len > 1) {
	memcpy(tmp + base_len, "/", 2);
	base_len++;
    }
    
    res = malloc(sizeof(pftp_directory_t));
    memset(res, 0, sizeof(pftp_directory_t));
    
    if ((dstream = opendir(base))) {
	while ((d = readdir(dstream))) {
	    tmp_len = base_len + strlen(d->d_name) + 1;
	    tmp = realloc(tmp, tmp_len);
	    memcpy(tmp + base_len, d->d_name, tmp_len - base_len);
	    lstat64(tmp, &buf);
	    file.name = strdup(d->d_name);
	    file.type = parse_file_type(buf.st_mode, &file.link);
	    if (file.link) {
#ifndef WIN32
		struct stat64 buf2;
		char *target = NULL;
		
		if (!stat64(tmp, &buf2)) {
		    file.type = parse_file_type(buf2.st_mode, NULL);
		}
		
		if ((target = readlink_malloc(tmp))) {
		    file.name = realloc_strcat(file.name, " -> ");
		    file.name = realloc_strcat(file.name, target);
		    free(target);
		}
#endif
	    }
	    file.perm = buf.st_mode & 0777;
	    file.links = buf.st_nlink;
	    file.user = file.group = NULL;
#if WIN32
	    get_user_groupname(tmp, &file.user, &file.group);
#else
	    get_username(buf.st_uid, &file.user);
	    get_groupname(buf.st_gid, &file.group);
#endif
	    file.size = buf.st_size;
#if HAVE_GMTIME_R
	    tmret = gmtime_r(&buf.st_mtime, &tm);
#else
	    tmret = gmtime(&buf.st_mtime);
	    if (tmret)
		memcpy(&tm, tmret, sizeof(struct tm));
#endif
	    
	    file.changed = malloc(13);
	    pftp_set_date(file.changed, &tm);
	    
	    res->length++;
	    res->files = realloc(res->files, 
				 sizeof(pftp_file_t) * res->length);
	    res->files[res->length - 1] = file;
	}
	closedir (dstream);
    } else {
	util_error("Error: Couldn't open the directory `%s'. %s", base, 
		   strerror(errno));
    }
    
    /* Sort filelist */
    sort_files(res->files, res->length);
    
    if (out_curpath) 
	(*out_curpath) = realloc_strcpy((*out_curpath), base);    
    
    if (base) free(base);
    if (tmp) free(tmp);
    
    return res;
}

/**
 * A function that returns a directory listing.
 * Works for both localhost and ftp
 */
pftp_directory_t *pftp_trans_ls(const char *ftp_name, const char *pattern,
				char **path, int hidden, int link_names)
{
    if (ftp_name) {
	pftp_server_t ftp = _getFTP(ftp_name);
	
	if (ftp)
	    return pftp_trans_ls2(ftp, pattern, path, hidden, link_names);
    } else {
	return pftp_trans_ls2(NULL, pattern, path, hidden, link_names);
    }
    
    return NULL;
}

static void remove_link_names(pftp_directory_t *dir)
{
    size_t f = 0;
    
    while (f < dir->length) {
	char *pos = strstr(dir->files[f].name, " -> ");
	if (pos)
	    pos[0] = '\0';
	f++;
    }
}
/*this function is called by pftp_trans_ls*/
pftp_directory_t *pftp_trans_ls2(pftp_server_t ftp, const char *pattern, 
				 char **path, int hidden, int link_names)
{
    pftp_directory_t *ret = NULL;
    char *_path = NULL, *pos = strrchr(pattern, '/');
    const char *real_pattern = pattern;

    if (pos) {
	real_pattern = ++pos;
	_path = malloc((pos - pattern) + 1);
	memcpy(_path, pattern, pos - pattern);    
	_path[pos - pattern] = '\0';
    }

    if (!ftp) {
	ret = local_dir_parse(_path ? _path : ".", path);
	if (ret) pftp_apply_mask(ret, 
				 strlen(real_pattern) ? real_pattern : "*", 
				 hidden);
    } else { 
	if (_path) {
	    char *old = NULL;
	    pftp_curdir(ftp, &old, 0);
	    if (old) {
		if (!pftp_cd(ftp, _path, NULL)) {
		    if (path)
			pftp_curdir(ftp, path, 0);
		    ret = inter_ls(ftp, 
				   strlen(real_pattern) ? real_pattern : "*", 
				   hidden);
		    pftp_cd(ftp, old, NULL);
		}
		free(old);
	    }
	} else {
	    if (path)
		pftp_curdir(ftp, path, 0);
	    ret = inter_ls(ftp, strlen(real_pattern) ? real_pattern : "*", 
			   hidden);
	}
    }
    
    if (ret && !link_names)
	remove_link_names(ret);
    
    return ret;
}

int pftp_get_path_dir(const char *ftpserver, char **path)
{
    if (ftpserver) {
	pftp_server_t ftp = _getFTP(ftpserver);
	
	if (!ftp) {
	    util_error("Error: Unable to connect to source. `%s'.", 
		       ftpserver);
	    return -1;
	}
	
	pftp_curdir(ftp, path, 0);
	
	if (!path) {
	    util_error("Error: Unable to get remote path.");
	    return -1;
	}
    } else {
	size_t size = 100;
	
	for (;;) {
	    (*path) = realloc((*path), size);
	    errno = 0;
	    if (getcwd((*path), size) == (*path))
		break;
	    if (errno != ERANGE) {
		util_error("Error: Unable to get local path. %s", 
			   strerror(errno));
		free((*path));
		return -1;
	    }
	    
	    size *= 2;
	}
    }
    
    return 0;
}

void pftp_apply_mask(pftp_directory_t *filelist, const char *mask, int hidden)
{
    size_t i;
    
/*    if (strcmp(mask, "*"))
      hidden = 0;*/

    for (i = 1; i <= filelist->length; i++) {
	if ((hidden 
	     &&
	     (fnmatch(mask, filelist->files[i - 1].name, 0)))
	    ||
	    (!hidden
	     &&
	     (((filelist->files[i - 1].name[0] == '.')
	       &&
	       ((mask[0] == '*')
		|| 
		(mask[0] == '?')))
	      ||
	      (fnmatch(mask, filelist->files[i - 1].name, 0))))) {
	    pftp_free_fft(filelist->files[i - 1]);
	    
	    if (i < filelist->length)
		memmove(filelist->files + (i - 1), filelist->files + i, 
			(filelist->length - i) * sizeof(pftp_file_t));
	    filelist->length--;
	    i--;
	}
    }    
}

void sort_files(pftp_file_t *files, size_t len)
{
    size_t i, j;
    pftp_file_t tmpf;
    for (i = 0; i < len; i++) {
	for (j = i + 1; j < len; j++) {
	    if (strcmp(files[i].name, files[j].name) == 1) {
		memcpy(&tmpf, files + i, sizeof(pftp_file_t));
		memmove(files + i, files + j, sizeof(pftp_file_t));
		memcpy(files + j, &tmpf, sizeof(pftp_file_t));
	    }   
	}
    }
}

int pftp_get_files_ordered(const char *from, const char *to, 
			   const char *srcpath, pftp_directory_t *FileList, 
			   pftp_que_t fileque, int last, unsigned int preserve,
			   pftp_item_action_t action, unsigned int mode)
{
    size_t I;
    char *destpath = NULL;
    pftp_directory_t dir;
    pftp_file_t tmpf;
    int ret = 0;
    size_t nrDirs = 0;

    dir.files = malloc(FileList->length * sizeof(pftp_file_t));
    dir.length = FileList->length;
    memcpy(dir.files, FileList->files, 
	   FileList->length * sizeof(pftp_file_t));
    
    /* Sort directories last and remove . and .. if found */
    if (dir.length > 1) {
	for (I = 1; I <= dir.length - nrDirs; I++) {
	    if (dir.files[I-1].type == pft_directory) {
		if (strcmp(dir.files[I-1].name, ".") == 0 ||
		    strcmp(dir.files[I-1].name, "..") == 0) {
		    /* no need to free here as pointers are really from
		       FileList
		       free_fft(dir.files[I-1]);
		    */
		    if (I < dir.length)
			memmove(dir.files + (I-1), dir.files + I, 
				(dir.length - I) * sizeof(pftp_file_t));
		    dir.length--;
		    I--;
		} else {
		    memcpy(&tmpf, dir.files + (I-1), sizeof(pftp_file_t));
		    memmove(dir.files + (I-1), 
			    dir.files + (dir.length - (1 + nrDirs)),
			    sizeof(pftp_file_t));
		    memcpy(dir.files + (dir.length - (1 + nrDirs)), &tmpf, 
			   sizeof(pftp_file_t));
		    nrDirs++;
		}
	    }
	}
    } else {
	for (I = 1; I <= dir.length; I++) {
	    if (strcmp(dir.files[I-1].name, ".") == 0 ||
		strcmp(dir.files[I-1].name, "..") == 0) {
		/* no need to free here as pointers are really from
		   FileList
		   free_fft(dir.files[I-1]);
		*/
		if (I < dir.length)
		    memmove(dir.files + (I-1), dir.files + I, 
			    (dir.length - I) * sizeof(pftp_file_t));
		dir.length--;
		I--;
	    }        
	}
    }
    
    /* Sort files */
    sort_files(dir.files, dir.length - nrDirs);
    /* Sort directories */
    sort_files(dir.files + (dir.length - nrDirs), nrDirs);
    
    if (pftp_get_path_dir(to, &destpath)) {
	free(dir.files);
	return -1;
    }
    
    if ((ret = pftp_que_add_dir(from, to, srcpath, destpath, &dir, fileque, 
				last, preserve, action, mode)))
	util_error("Error: Unable to add files to que.");
    
    free(destpath);
    free(dir.files);
    
    return ret;
}

static int trans_mask(const char *from, const char *to, const char *mask, 
		      pftp_que_t fileque, unsigned int preserve,
		      pftp_item_action_t action, unsigned int mode)
{
    char *srcpath = NULL;
    pftp_directory_t *dir = pftp_trans_ls(from, mask, &srcpath, 1, 0);
    int ret = -1;
    
    if (dir) {
	ret = pftp_get_files_ordered(from, to, srcpath, dir, fileque, 0,
				     preserve, action, mode);
	pftp_free_fdt((*dir));
	free(dir);
    }
    
    if (srcpath)
	free(srcpath);
    
    return ret;
}

void que_failed_trans(pftp_item_t item, pftp_que_t errorque)
{
    if (item->target) {
	util_error("%s -> %s failed.", item->filename, item->target);
    } else {
	util_error("%s failed.", item->filename);
    }
    pftp_que_add_last(item, errorque);
}

static unsigned int if_file_exist_get_response(pftp_item_t item, 
					       pftp_ExistingFilefunc_t dialog,
					       uint64_t *filesize)
{
    pftp_directory_t *filelist = NULL;
    unsigned int response = 0;
    const char *target = item->target ? item->target : item->filename;
    filelist = pftp_trans_ls(item->ftp_to, target, NULL, 1, 0);
    
    /* filelist == NULL if filelisting caused error */
    if (filelist) {
        /* if listing contains the file filelist->length > 0 */
	if (filelist->length) {
	    if (item->mode & PFTP_MODE_OVERWRITE_OK) {
		/* Overwrite without asking */
		response = PFTP_DLG_OVERWRITE;
	    } else {
	        /* If the existing file is 0 byte large, overwrite it */
	        if (filelist->files[0].size == 0)
		  		response = PFTP_DLG_OVERWRITE;
			else if (dialog)
		    	response = dialog(filelist->files[0], item);
			else
		    	response = PFTP_DLG_ABORT;
	    }
	    *filesize = filelist->files[0].size;
	} else {
	    response = 0; // doesn't exist
	}
	pftp_free_fdt(*filelist);
	free(filelist);
    } else { 
	response = 0; // doesn't exist    
    }
    
    return response;
}

static unsigned int change_directories(const char *ftp_to, 
                       const char *destpath,
                       const char *ftp_from, 
                       const char *srcpath,
                       int dstpath_important)
{
    if (dstpath_important) {
	if (trans_cd(ftp_to, destpath)) {
	    util_error("Error: Unable to change to destination directory");
	    return 1;
	}
    }
    
    if (trans_cd(ftp_from, srcpath)) {
	util_error("Error: Unable to change to source directory.");
	return 1;
    }
    
    return 0;
}

static void directory_trans(pftp_item_t item, pftp_que_t fileque,
			    pftp_que_t errorque, clear_cache_t *clear_cache)
{
    const char *target = item->target ? item->target : item->filename;
    int error = 0;
    
    switch (item->action)
    {
    case PFTP_COPY:
    case PFTP_MOVE:
	if (trans_cd(item->ftp_from, item->filename)) {
	    util_error("Error: Unable to change source directory");
	    error = 1;
	} else if (pftp_makePath(item->ftp_to, target)) {
	    util_error("Error: Unable to create destination directory");
	    error = 1;
	} else if (trans_cd(item->ftp_to, item->destpath)) {
	    error = 1;
	} else if (trans_cd(item->ftp_to, target)) {
	    util_error("Error: Unable to enter destination directory");
	    error = 1;
	} else if (trans_mask(item->ftp_from, item->ftp_to, "*", fileque,
			      item->preserve, item->action, item->mode)) {
	    util_error("Error: Unable to walk recursively");
	    error = 1;
	}
	break;
    case PFTP_DELETE_FROM:
	if (trans_cd(item->ftp_from, item->filename)) {
	    util_error("Error: Unable to change source directory");
	    error = 1;
	}
	
	/* Add directory back to que, to be deleted last */
	
	item->action = PFTP_DELETE_FROM_DIR;
	pftp_que_add_first(item, fileque);    
	
	if (trans_mask(item->ftp_from, NULL, "*", fileque, item->preserve, 
		       PFTP_DELETE_FROM, item->mode)) {
	    util_error("Error: Unable to walk recursively");
	    error = 1;
	}
	
	return;    
	break;
    case PFTP_DELETE_FROM_DIR:
	if (trans_rmdir(item->ftp_from, item->filename)) {
	    util_error("Error: Unable to remove directory.\n");
	    error = 1;
	}
	if (item->ftp_from)
	    add_curdir_to_clear_cache(clear_cache, _getFTP(item->ftp_from));
	break;
    }
    
    if (error) {
	que_failed_trans(item, errorque);
    } else {
	pftp_free_queitem(&item);
    }
}

static int checking_trans(pftp_item_t *item, pftp_ExistingFilefunc_t dialog,
			  uint64_t *filesize, intptr_t *exist_action,
			  pftp_que_t fileque, pftp_que_t errorque,
			  intptr_t *skipped)
{   
    (*exist_action) = if_file_exist_get_response(*item, dialog, filesize);
    if ((*exist_action) & PFTP_DLG_ABORT) {
	do {
	    pftp_que_add_last((*item), errorque);
	} while (((*item) = pftp_get_next(fileque)));
    } else if ((*exist_action) & PFTP_DLG_SKIP) {
//    (*skipped)++;
    } 
    if ((*exist_action) & PFTP_DLG_ALL)
	return ((*exist_action) & PFTP_DLG_ALL);
/*
  else
  pftp_free_queitem(item);
*/
    return 0;
}

static void clear_and_free_cache(clear_cache_t *clear_cache)
{
    size_t i;
    
    for (i = 0; i < clear_cache->count; i++) {
	pftp_clear_dir_cache(clear_cache->ftp[i], clear_cache->path[i]);
	free(clear_cache->path[i]);
    }
    
    if (clear_cache->ftp)
	free(clear_cache->ftp);
    if (clear_cache->path)
	free(clear_cache->path);

    memset(clear_cache, 0, sizeof(clear_cache_t));
}

static void add_dir_to_clear_cache(clear_cache_t *clear_cache, 
				   pftp_server_t ftp, const char *path)
{
    size_t i;
    
    if (!ftp) {
	/* Local, isn't cached. */
	return;
    }
    
    for (i = 0; i < clear_cache->count; i++) {
	if (clear_cache->ftp[i] == ftp &&
	    strcmp(clear_cache->path[i], path) == 0) {
	    /* Already in list */
	    return;
	}        
    }
    
    clear_cache->ftp = realloc(clear_cache->ftp, (clear_cache->count + 1) *
			       sizeof(pftp_server_t));
    clear_cache->path = realloc(clear_cache->path, (clear_cache->count + 1) *
				sizeof(char *));
    clear_cache->ftp[clear_cache->count] = ftp;
    clear_cache->path[clear_cache->count] = strdup(path);
    clear_cache->count++;
}

void add_curdir_to_clear_cache(clear_cache_t *clear_cache, pftp_server_t ftp)
{
    char *path = NULL;
    
    if (!ftp) {
	/* Local, isn't cached. */
	return;
    }
    
    pftp_curdir(ftp, &path, 0);
    if (!path) {
	/* Error, not really usefull to fix here */
	return; 
    }
    
    add_dir_to_clear_cache(clear_cache, ftp, path);
    free(path);
}

int does_file_exist(pftp_item_t item)
{
    pftp_directory_t *filelist = NULL;
    int does_exist = 0;
    const char *target = item->target ? item->target : item->filename;
    
    filelist = pftp_trans_ls(item->ftp_to, target, NULL, 1, 0);
    
    if (filelist) {
	if(filelist->length) {
	    does_exist = 1;
	} 
	pftp_free_fdt(*filelist);
	free(filelist);
    }
    return does_exist;
}

static int _finish_que_item(pftp_item_t item, clear_cache_t *clear_cache)
{
    const char *target;
    target = item->target ? item->target : item->filename;
    
    if (item->preserve & PFTP_PRESERVE_TIME) {
	struct tm date;
	
	pftp_get_date(item->changed, &date);
	
	if (item->ftp_to) {
	    pftp_server_t ftp;
	    ftp = _getFTP(item->ftp_to);
	    if (!(item->preserve & PFTP_PRESERVE_FORCE)) {
		util_comm_central(PFTP_NEXTSTATUS_IS_NOT_FATAL, NULL, NULL, 
				  NULL, NULL);
	    }
	    
	    if (pftp_mdtm(ftp, target, &date)) {
		if (item->preserve & PFTP_PRESERVE_FORCE) {
		    return -1;
		}
	    }
	} else {
	    struct utimbuf data;
	    data.modtime = data.actime = timegm(&date);
	    
	    if (utime(target, &data)) {
		util_error("WARN: Unable to set file modification time: %s.",
			   strerror(errno));
		
		if (item->preserve & PFTP_PRESERVE_FORCE) {
		    return -1;
		}
	    }
	}
    }
    if (item->preserve & PFTP_PRESERVE_ACCESS) {
	if (item->ftp_to) {        
	    pftp_server_t ftp;
	    ftp = _getFTP(item->ftp_to);
	    if (!(item->preserve & PFTP_PRESERVE_FORCE)) {
		util_comm_central(PFTP_NEXTSTATUS_IS_NOT_FATAL, NULL, NULL, 
				  NULL, NULL);
	    }
	    
	    if (pftp_chmod(ftp, target, item->access)) {
		if (item->preserve & PFTP_PRESERVE_FORCE) {
		    return -1;
		}
	    }
	} else {
	    if (chmod(target, item->access)) {
		util_error("WARN: Unable to set file permissions (%u%u%u): %s.",
			   item->access / (8 * 8), 
			   (item->access % (8 * 8)) / 8,
			   item->access % 8,
			   strerror(errno));
		
		if (item->preserve & PFTP_PRESERVE_FORCE) {
		    return -1;
		}
	    }
	}
    }
    
    switch (item->action) {
    case PFTP_COPY:
	return 0;
    case PFTP_MOVE:
	if (trans_rm(item->ftp_from, item->filename))
	    return -1;
	if (item->ftp_from)
	    add_curdir_to_clear_cache(clear_cache, _getFTP(item->ftp_from));
	return 0;
    case PFTP_DELETE_FROM:
	if (trans_rm(item->ftp_from, item->filename))
	    return -1;
	if (item->ftp_from)
	    add_curdir_to_clear_cache(clear_cache, _getFTP(item->ftp_from));
	return 0;
    case PFTP_DELETE_FROM_DIR:
	/* This will only be called for files */
	assert(0);
	return -1;
    }
    
    assert(0);
    
    return -1;
}

static void non_checking_trans_from_local(intptr_t *skipped,
					  clear_cache_t *clear_cache,
					  pftp_item_t item, 
					  unsigned int exist_action,
					  uint64_t filesize,
					  pftp_que_t errorque)
{
    FILE *fh = NULL;
    unsigned int err = 0, skip = 0;
    const char *target = item->target ? item->target : item->filename;
    
    if (!(fh = fopen64(item->filename, "rb"))) {
	/*Unable to open file*/
	util_error("Error: Unable to open file: %s.", strerror(errno));
	err = 1;
    } else {
        /*If you were able to open the file*/
	pftp_server_t ftp = _getFTP(item->ftp_to);
	if(does_file_exist(item)) {
	    if (exist_action & PFTP_DLG_SKIP) {
		(*skipped)++;
		skip = 1;
	    } else if (exist_action & PFTP_DLG_RESUME) {
		uint64_t tmp = filesize;
#ifdef WIN32
		if (tmp > LONG_MAX) {
		    fseeko64(fh, LONG_MAX, SEEK_SET);
		    tmp -= LONG_MAX;
		} 
#else
		if (tmp > LLONG_MAX) {
		    fseeko64(fh, LLONG_MAX, SEEK_SET);
		    tmp -= LLONG_MAX;
		} 
#endif
		fseeko64(fh, tmp, SEEK_SET);
	    } else if (exist_action & PFTP_DLG_OVERWRITE) {
		/*remove file*/
		if (pftp_rm(ftp, target)) {
		    util_error("Error: Unable to remove file before upload.");
		    err = 1;
		    /*Then try tu upload again*/
		}         
	    }
	}
	/* Try to put file unless skipped */
	if(!skip) {
	    if (pftp_put(ftp, target, 'B', fh, 0, item->size)) {
		util_error("Error: Unable to upload file.");
		err = 1;
	    } else {
		/* if put was successful, clear cache for this directory */
		if (_finish_que_item(item, clear_cache)) {
		    err = 1;
		}
		
		add_curdir_to_clear_cache(clear_cache, ftp);
	    }
	} else {
	    /* skipping */
	}
    }
    if (fh) {
	fclose(fh);
    }
    if (err) {    
	que_failed_trans(item, errorque);
    } else {
	pftp_free_queitem(&item);
    }
}


static void non_checking_trans_to_local(intptr_t *skipped,
					clear_cache_t *clear_cache,
					pftp_item_t item,
					unsigned int exist_action,
					uint64_t filesize,
					pftp_que_t errorque)
{
    FILE *fh = NULL;
    unsigned int err = 0;
    const char *target = item->target ? item->target : item->filename;
    if(does_file_exist(item)){
	if (exist_action & PFTP_DLG_SKIP) {
	    (*skipped)++;
	    pftp_free_queitem(&item);
	    return; 
	} else if (exist_action & PFTP_DLG_RESUME) {
	    if (!(fh = fopen64(target, "ab"))) {
		util_error("Error: Unable to create file: %s.", 
			   strerror(errno));
		err = 1;
	    } else {
		uint64_t tmp = filesize;
#ifdef WIN32
		if (tmp > LONG_MAX) {
		    fseeko64(fh, LONG_MAX, SEEK_SET);
		    tmp -= LONG_MAX;
		} 
#else
		if (tmp > LLONG_MAX) {
		    fseeko64(fh, LLONG_MAX, SEEK_SET);
		    tmp -= LLONG_MAX;
		} 
#endif
		fseeko64(fh, tmp, SEEK_SET);
	    }
	} else if((exist_action & PFTP_DLG_OVERWRITE)) {
	    if (!(fh = fopen64(target, "wb"))) {
		util_error("Error: Unable to create file: %s.", 
			   strerror(errno));
		err = 1;
	    } else {
		filesize = 0;
	    }
	}
    } else {
	if (!(fh = fopen64(target, "wb"))) {
	    util_error("Error: Unable to create file: %s.", 
		       strerror(errno));
	    err = 1;
	} else {
	    filesize = 0;
	}
    }
    
    if (!err) {
	if (pftp_get(_getFTP(item->ftp_from), item->filename, 'B', fh, 
		     filesize, item->size, NULL)) {
	    util_error("Error: Unable to download file.");
	    err = 1;
	}    
	
	fclose(fh);
	
	if (_finish_que_item(item, clear_cache)) {
	    err = 1;
	}
    }
    
    if (!err) {
        pftp_free_queitem(&item);
    } else {
	que_failed_trans(item, errorque);
    }
}

static void non_checking_trans_FXP(intptr_t *skipped,
				   clear_cache_t *clear_cache,
				   pftp_item_t item, 
				   unsigned int exist_action,
				   uint64_t filesize,
				   pftp_que_t errorque)
{
    FILE *fh = NULL;
    unsigned int err = 0;
    const char *target = item->target ? item->target : item->filename;
    pftp_server_t ftp;
    
    if(does_file_exist(item)){
	if (exist_action & PFTP_DLG_SKIP) {
	    (*skipped)++;
	    pftp_free_queitem(&item);
	    return ;
	} else if (exist_action & PFTP_DLG_RESUME) {
	    /* don't need to do anything */
	} else if ((exist_action & PFTP_DLG_OVERWRITE) || (exist_action == 0))
	{
	    filesize = 0;
	}
    }
    
    ftp = _getFTP(item->ftp_to);
    if (pftp_fxp(_getFTP(item->ftp_from), item->filename, ftp, target, 
		 'B', filesize, filesize, item->size)) {
	util_error("Error: Unable to FXP.");
	err = 1;        
    } else {
	if (_finish_que_item(item, clear_cache)) {
	    err = 1;
	}
	add_curdir_to_clear_cache(clear_cache, ftp);
    }
    
    if (fh) {
	fclose(fh);
    }
    if (err) {
	que_failed_trans(item, errorque);
    } else {
        pftp_free_queitem(&item);
    }
}

static void non_checking_trans(intptr_t *skipped ,
			       clear_cache_t *clear_cache, 
			       pftp_item_t item, unsigned int exist_action, 
			       uint64_t filesize, pftp_que_t errorque)
{
    if (item->ftp_from == NULL) {
	/* if the file is located on localhost */
	non_checking_trans_from_local(skipped, clear_cache, item, 
				      exist_action, filesize, errorque);
    } else if (item->ftp_to == NULL) { 
	/* Downloading file to localhost */
	non_checking_trans_to_local(skipped, clear_cache, item, 
				    exist_action, filesize, errorque);
    } else {
	/* FXP */
	non_checking_trans_FXP(skipped, clear_cache, item, exist_action,
			       filesize, errorque);
    }
}

static void add_to_origpath(const char *ftp, char ***name, char ***path,
			    size_t *paths)
{
    size_t o;
    if (ftp) {
	for (o = 0; o < (*paths); o++) {
	    if ((*name)[o] && strcmp(ftp, (*name)[o]) == 0) {
		/* Already saved */
		return;
	    }
	}
    } else {
	for (o = 0; o < (*paths); o++) {
	    if (!(*name)[o]) {
		/* Already saved */
		return;
	    }
	}
    }
    
    (*name) = realloc((*name), ((*paths) + 1) * sizeof(char *));
    (*path) = realloc((*path), ((*paths) + 1) * sizeof(char *));
    (*name)[(*paths)] = ftp ? alloc_strcpy(ftp) : NULL;
    (*path)[(*paths)] = NULL;
    pftp_get_path_dir(ftp, &(*path)[(*paths)]);    
    (*paths)++;
}

int pftp_begin_trans(pftp_ExistingFilefunc_t dialog, 
		     pftp_que_t fileque, pftp_que_t errorque)
{
    pftp_item_t item = NULL;
    char **origpath = NULL, **origpathname = NULL;
    size_t origpaths = 0, que_pos = 0;
    intptr_t skipped = 0;
    unsigned int all = 0;
    intptr_t exist_action = 0;
    clear_cache_t clear_cache;
    uint64_t filesize;
    size_t errorque_length_at_start = pftp_queLength(errorque);
    memset(&clear_cache, 0, sizeof(clear_cache_t));

    util_comm_central(PFTPUTIL_INITQUE, NULL, NULL,
		      (pftp_param_t) pftp_queLength(fileque), NULL);
    
    while ((item = pftp_get_next(fileque))) {
	/* this should maybe be after the "is this a directory" check? */
	
	que_pos++;
	
	add_to_origpath(item->ftp_to, &origpathname, &origpath, &origpaths);
	add_to_origpath(item->ftp_from, &origpathname, &origpath, &origpaths);
	
	/* Reset filesize for each file */
	filesize = 0;
	
	if (!change_directories(item->ftp_to, item->destpath,
				item->ftp_from, item->srcpath,
				item->action != PFTP_DELETE_FROM 
				&& item->action != PFTP_DELETE_FROM_DIR)) {
	    if (item->dir) {
		directory_trans(item, fileque, errorque, &clear_cache);
		util_comm_central(PFTPUTIL_UPDATEQUELENGTH, NULL, NULL,
				  (pftp_param_t)
				  (pftp_queLength(fileque) + que_pos), NULL);
	    } else {
		if (item->action != PFTP_DELETE_FROM) {
		    /* Check file transfer if no all option has been
		       selected */
		    if (!all) {
			all = checking_trans(&item, dialog, &filesize, 
					     &exist_action, fileque, errorque,
					     &skipped);
		    } 
		    /* Is the PFTP_DLG_ABORT flag true in exist_action? */
		    if (!(exist_action & PFTP_DLG_ABORT)) {
			non_checking_trans(&skipped, &clear_cache, item, 
					   exist_action, filesize, errorque);
		    }
		} else {
		    if (_finish_que_item(item, &clear_cache)) {
			que_failed_trans(item, errorque);
		    } else {
			pftp_free_queitem(&item);
		    }
		}
	    }
	} else {
	    que_failed_trans(item, errorque);
	}
	
	util_comm_central(PFTPUTIL_UPDATEQUE, NULL, NULL,
			  (pftp_param_t) pftp_queLength(fileque),
			  (pftp_param_t) (que_pos - pftp_queLength(errorque) -
					  skipped + errorque_length_at_start));
    }
    
    util_comm_central(PFTPUTIL_DONEQUE, NULL, NULL, 
		      (pftp_param_t) (exist_action & PFTP_DLG_ABORT), 
		      (pftp_param_t) skipped);
    
    //Change dirs to original directories
    if (origpaths) {
	size_t o;
	for (o = 0; o < origpaths; o++) {
	    trans_cd(origpathname[o], origpath[o]);
	    if (origpathname[o]) {
		free(origpathname[o]);
	    }
	    if (origpath[o]){
		free(origpath[o]);
	    }
	}
	free(origpathname);
	free(origpath);
    }
    
    clear_and_free_cache(&clear_cache);
    return 0;
}

#ifdef WIN32
#include <aclapi.h>

static void _get_psid(char **name, PSID psid)
{
    DWORD n_len = 0, d_len = 0;
    char *domain = NULL;
    SID_NAME_USE use;
    
    LookupAccountSid(NULL, psid, *name, &n_len, domain, &d_len, &use);
    
    *name = realloc(*name, n_len);
    domain = realloc(domain, d_len);
    
    if (!LookupAccountSid(NULL, psid, *name, &n_len, domain, &d_len, &use)) {
        *name = realloc_strcpy(*name, "unknown");
        free(domain);
        return;
    }
    
    free(domain);
}

static void get_user_groupname(const char *filename, char **user, char **group)
{
    PSID uid, gid;
    PSECURITY_DESCRIPTOR desc = NULL;
    
    if (GetNamedSecurityInfo(filename, SE_FILE_OBJECT, 
                             GROUP_SECURITY_INFORMATION 
			     | OWNER_SECURITY_INFORMATION,
                             &uid, &gid, NULL, NULL, &desc) == ERROR_SUCCESS) {
        _get_psid(user, uid);
        _get_psid(group, gid);
        LocalFree(desc);
    } else {
        fprintf(stderr, "GetNamed...bugged out: %d\n", GetLastError());
        (*user) = realloc_strcpy((*user), "unknown");
        (*group) = realloc_strcpy((*group), "unknown");
    }
}
#else
static void get_username(uid_t uid, char **name)
{
    struct passwd *data = getpwuid(uid);
    
    if (data) {
	(*name) = realloc_strcpy((*name), data->pw_name);
    } else {
	(*name) = realloc_strcpy((*name), "unknown");
    }
}

static void get_groupname(gid_t gid, char **name)
{
    struct group *data = getgrgid(gid);
    
    if (data) {
	(*name) = realloc_strcpy((*name), data->gr_name);
    } else {
	(*name) = realloc_strcpy((*name), "unknown");
    }
}
#endif

static pftp_server_t _getFTP(const char *name)
{
    pftp_server_t ret = NULL;
    if (util_comm_central(PFTPUTIL_NEEDCURFTP, NULL, NULL, (pftp_param_t)name, 
			  (pftp_param_t)&ret))
	return NULL;
    return ret;
}

void util_error(const char *format, ...)
{
    char *buf = malloc(1024);
    va_list list;
    
    va_start(list, format);
    
    vsnprintf(buf, 1024, format, list);

    util_comm_central(PFTPUTIL_ERROR, NULL, NULL, (pftp_param_t)buf, NULL);
    free(buf);

    va_end(list);
}

__inline int is_letter(char c)
{
    return ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'));
}

__inline int is_digit(char c)
{
    return (c >= '0' && c <= '9');
}

int pftp_valid_hostname(const char *hostname)
{
    int start = 1;
    const char *h = hostname;
    
    while (*h) {
	if (start) {
	    if (strchr(h, '.')) {
		/* only allow letter | digit as first character of 
		   normal label */
		if (!is_letter(*h) && !is_digit(*h))
		    return 0;
	    } else {
		/* only allow letter as first character of normal label */
		if (!is_letter(*h)) {
		    /* but if it's an IP? */
		    struct in_addr addr;
		    if (inet_aton(hostname, &addr)) {
			/* Valid IPv4 */
			return 1; 
		    }
		    
		    return 0;
		}
	    }
	    start = 0;
	} else if (*h == '.') {
	    /* only allow letter | digit as last character of label
	       as letter | digit | hypen is the only allowe generally so... */
	    if (*(h - 1) == '-')
		return 0;
	    start = 1;
	} else {
	    if (!is_letter(*h) && !is_digit(*h) && *h != '-')
		return 0;
	}
	
	h++;
    }
    
    /* Valid hostname */
    return 1; 
}

static void _pftp_remove_fft(pftp_directory_t *dir, size_t item)
{
    pftp_free_fft(dir->files[item]);
    dir->length--;
    memmove(dir->files + item, dir->files + item + 1, 
	    (dir->length - item) * sizeof(pftp_file_t));
}

static int _pftp_mirror_dir(const char *srcftp, const char *dstftp,
			    pftp_que_t fileque,
			    int delete_dst, int recursive, int only_new, 
			    int ign_time)
{
    pftp_directory_t *srcdir, *dstdir;
    size_t f, ff;
    char *oldsrcpath = NULL, *olddstpath = NULL;
    
    srcdir = pftp_trans_ls(srcftp, "*", &oldsrcpath, 1, 0); 
    dstdir = pftp_trans_ls(dstftp, "*", &olddstpath, 1, 0);
    
    if (!srcdir || !dstdir) {
	if (srcdir) {
	    pftp_free_fdt((*srcdir));
	    free(srcdir);
	}
	if (dstdir) {
	    pftp_free_fdt((*dstdir));
	    free(dstdir);
	}
	if (oldsrcpath) free(oldsrcpath);
	if (olddstpath) free(olddstpath);
	return -1;
    }
    
    for (f = 0; f < srcdir->length; f++) {
	int found = 0, update = 0;
	
	for (ff = 0; ff < dstdir->length; ff++) {
	    
	    if (srcdir->files[f].type != pft_directory &&
		dstdir->files[ff].type != pft_directory &&
		strcmp(srcdir->files[f].name, dstdir->files[ff].name) == 0) {
		/* Same filename */
		found = 1;
		
		if (ign_time) {
		    if (srcdir->files[f].size != dstdir->files[ff].size) {
			update = 1;
		    } else {
			update = 0;
		    }
		} else {
		    struct tm srcdate, dstdate;
		    time_t srctime, dsttime;
		    double diff;
		    
		    pftp_get_fft_date(srcdir->files + f, &srcdate);
		    pftp_get_fft_date(dstdir->files + ff, &dstdate);
		    
		    srctime = timegm(&srcdate);
		    dsttime = timegm(&dstdate);
		    
		    diff = difftime(dsttime, srctime);
		    
		    if (only_new) {
			if (diff > 0.0) {
			    update = 1;
			} else {
			    update = 0;
			}
		    } else {
			if (srcdir->files[f].size != dstdir->files[ff].size 
			    || diff != 0.0) {
			    update = 1;
			} else {
			    update = 0;
			}
		    }
		}
	    } else if (srcdir->files[f].type == pft_directory &&
		       dstdir->files[ff].type == pft_directory &&
		       strcmp(srcdir->files[f].name, 
			      dstdir->files[ff].name) == 0) {
		/* Same direcory name */
		found = 1;
		update = 0; 
		
		if (recursive) {
		    int ok = 1, fatal = 0;
		    
		    /* Change current path */
		    if (ok && trans_cd(srcftp, srcdir->files[f].name)) {
			ok = 0;
		    }
		    
		    if (ok && trans_cd(dstftp, dstdir->files[ff].name)) {
			if (trans_cd(srcftp, oldsrcpath)) {
			    fatal = 1;
			}
			ok = 0;
		    }
		    
		    /* Wander into subdirs */
		    
		    if (ok) {
			_pftp_mirror_dir(srcftp, dstftp, fileque, delete_dst, 
					 recursive, only_new, ign_time);
		    }
		    
		    /* Restore old path */
		    
		    if (ok && (trans_cd(srcftp, oldsrcpath) ||
			       trans_cd(dstftp, olddstpath))) {
			fatal = 1;
		    }
		    
		    if (fatal) {
			util_error("Unreparable error during mirror.");
			pftp_free_fdt(*srcdir);
			pftp_free_fdt(*dstdir);
			free(srcdir);
			free(dstdir);
			free(oldsrcpath);
			free(olddstpath);
			return -1;
		    }
		}
	    }
	    
	    if (found) {
		/* Remove found file from dstdir */
		_pftp_remove_fft(dstdir, ff);        
		if (!update) {
		    /* Remove file from srcdir if no need to update */
		    _pftp_remove_fft(srcdir, f);
		    f--;
		}
		break;
	    } else {
		if (!recursive && dstdir->files[ff].type == pft_directory) {
		    /* Remove directories from list if !recursive */
		    _pftp_remove_fft(dstdir, ff);        
		    ff--;
		}
	    }
	}
	
	if (!found) {
	    if (!recursive && srcdir->files[f].type == pft_directory) {
		/* Remove directories from list if !recursive */
		_pftp_remove_fft(srcdir, f);
		f--;
	    }
	}
    }
    
    if (pftp_get_files_ordered(srcftp, dstftp, oldsrcpath, srcdir, fileque, 
			       1, 
			       PFTP_PRESERVE_ACCESS |
			       (ign_time ? 0 : PFTP_PRESERVE_TIME) |
			       PFTP_PRESERVE_FORCE, 
			       PFTP_COPY,
			       PFTP_MODE_OVERWRITE_OK)) {
	pftp_free_fdt(*srcdir);
	pftp_free_fdt(*dstdir);
	free(srcdir);
	free(dstdir);
	free(oldsrcpath);
	free(olddstpath);
	return -1;
    }
    
    if (delete_dst) {
	for (f = 0; f < dstdir->length; f++) {
	    /* Remove old file or directory from destination directory */
	    if (dstdir->files[f].type == pft_directory && !recursive) {
		/* Ignore */
	    } else {
		pftp_item_t item;
		item = malloc(sizeof(struct pftp_item_s));
		memset(item, 0, sizeof(struct pftp_item_s));
		item->ftp_from = dstftp ? strdup(dstftp) : NULL;
		item->dir = (dstdir->files[f].type == pft_directory);
		item->srcpath = strdup(olddstpath);
		item->filename = strdup(dstdir->files[f].name);
		item->action = PFTP_DELETE_FROM;
		item->preserve = PFTP_PRESERVE_NONE;
		item->mode = PFTP_MODE_NORMAL;
		pftp_que_add_last(item, fileque);    
	    }
	}
    }
    
    pftp_free_fdt(*srcdir);
    pftp_free_fdt(*dstdir);
    free(srcdir);
    free(dstdir);
    free(oldsrcpath);
    free(olddstpath);
    
    return 0;
}

int pftp_mirror(const char *srcftp, const char *srcpath,
		const char *dstftp, const char *dstpath,
		pftp_que_t fileque, 
		int delete_dst, int recursive, int only_new, int ign_time)
{
    char *oldsrcpath, *olddstpath;
    int ret;
    
    ret = -1;
    oldsrcpath = olddstpath = NULL;
    
    if (srcpath) {
	if (pftp_get_path_dir(srcftp, &oldsrcpath))
	    return -1;
	
	if (trans_cd(srcftp, srcpath)) {
	    free(oldsrcpath);
	    return -1;
	}
    }
    if (dstpath) {
	if (pftp_get_path_dir(dstftp, &olddstpath))
	    if (oldsrcpath) free(oldsrcpath);
        return -1;
	
	if (trans_cd(dstftp, dstpath)) {
	    if (oldsrcpath) free(oldsrcpath);
	    free(olddstpath);
	    return -1;
	}
    }
    
    ret = _pftp_mirror_dir(srcftp, dstftp, fileque, delete_dst, recursive, 
			   only_new, ign_time);
    
    if (oldsrcpath) {
	trans_cd(srcftp, oldsrcpath);
	free(oldsrcpath);    
    }
    if (olddstpath) {
	trans_cd(dstftp, olddstpath);
	free(olddstpath);    
    }
    
    return ret;
}
