/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_CTYPE_H
# include <ctype.h>
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif /* DMALLOC */

#if !HAVE_GETLINE
# include "getline.h"
#endif
#include "str.h"
#include "util.h"
#include "editconf.h"

struct edit_conf_s {
    char *filename, **line;
    size_t lines, *linelen;
};

edit_conf_t open_conffile(const char *filepath, const char *firstline)
{
    edit_conf_t ret = malloc(sizeof(struct edit_conf_s));
    char *buffer = NULL, *pos;
    size_t buflen = 0, l, firstline_len = strlen(firstline);
    ssize_t ret2;
    int first = 1;
    FILE *fh;

    memset(ret, 0, sizeof(struct edit_conf_s));

    if (!(fh = fopen(filepath, "rb"))) {
	if (!(fh = fopen(filepath, "wb"))) {
	    util_error("ERR: Unable to create file. %s", strerror(errno));
	    free(ret);
	    return NULL;
	}

	fputs(firstline, fh);
	fputs("\n\n", fh);
	fclose(fh);
	
	if (!(fh = fopen(filepath, "rb"))) {
	    util_error("ERR: Unable to open file. %s", strerror(errno));
	    free(ret);
	    return NULL;
	}
    }

    for (;;) {
	ret2 = getline(&buffer, &buflen, fh);
	if (ret2 < 0) {
	    if (feof(fh) && (ret->lines == 0 || first)) {
		/* Empty file */
		if (buffer) free(buffer);
		for (l = 0; l < ret->lines; l++)
		    free(ret->line[l]);
		if (ret->line) free(ret->line);
		if (ret->linelen) free(ret->linelen);
		free(ret);
		fclose(fh);
		if (!(fh = fopen(filepath, "wb"))) {
		    util_error("ERR: Unable to open file for writing. %s",
			       strerror(errno));
		    return NULL;
		}
		
		fputs(firstline, fh);
		fputs("\n\n", fh);
		fclose(fh);

		return open_conffile(filepath, firstline);
	    } else if (feof(fh)) {
		/* EOF */
		break;
	    } else {
		if (buffer) free(buffer);
		for (l = 0; l < ret->lines; l++)
		    free(ret->line[l]);
		if (ret->line) free(ret->line);
		if (ret->linelen) free(ret->linelen);
		free(ret);
		util_error("ERR: Error reading file. %s", strerror(errno));
		fclose(fh);
		return NULL;
	    }
	}
	l = ret->lines++;
	ret->line = realloc(ret->line, ret->lines * sizeof(char *));
	ret->linelen = realloc(ret->linelen, ret->lines * sizeof(size_t));

	ret->linelen[l] = ret2;
	ret->line[l] = malloc(ret2 + 1);
	memcpy(ret->line[l], buffer, ret2 + 1);

	if (ret->linelen[l] == 0 || ret->line[l][ret->linelen[l]-1] != '\n') {
	    /* EOF */
	    break;
	} else {
	    ret->line[l][--ret->linelen[l]] = '\0';
	}

	if (first) {
	    int ok = 0;
	    pos = ret->line[l];
	    while (xisspace(pos[0]))
		pos[0]++;
	    if (pos[0] != '#' && pos[0]) {
		if (strncmp(pos, firstline, firstline_len) == 0) {
		    pos += firstline_len;
		    ok = 1;
		    while (pos[0]) {
			if (!xisspace(pos[0])) {
			    ok = 0;
			    break;
			}
			pos++;
		    }
		}

		if (!ok) {
		    if (buffer) free(buffer);
		    for (l = 0; l < ret->lines; l++)
			free(ret->line[l]);
		    if (ret->line) free(ret->line);
		    if (ret->linelen) free(ret->linelen);
		    free(ret);
		    util_error("ERR: Invalid file, abort.");
		    fclose(fh);
		    return NULL;
		}

		first = 0;
	    }
	}
    }

    if (fh)
	fclose(fh);
    free(buffer);
    ret->filename = alloc_strcpy(filepath);

    return ret;
}

int close_conffile(edit_conf_t fh)
{
    if (fh) {
	size_t l;
	FILE *f;
	int ret = 0;

	if (!(f = fopen(fh->filename, "wb"))) {
	    util_error("ERR: Unable to open file for writing. %s",
		       strerror(errno));
	    ret = -1;
	}

	if (!ret) {
	    for (l = 0; l < fh->lines; l++) {
		if (fputs(fh->line[l], f) == EOF ||
		    fputs("\n", f) == EOF) {
		    util_error("ERR: Error writing to file. %s",
			       strerror(errno));
		    ret = -1;
		    break;
		}
	    }

	    fclose(f);
	}

	free(fh->filename);
	for (l = 0; l < fh->lines; l++)
	    free(fh->line[l]);
	if (fh->line) free(fh->line);
	if (fh->linelen) free(fh->linelen);
	free(fh);

	return ret;
    } 

    util_error("ERR: Invalid file handle");
    return -1;
}

static void insert_line(edit_conf_t fh, size_t *cur, const char *newline);
static void replace_line(edit_conf_t fh, size_t cur, const char *newline);

int update_confvalue(edit_conf_t fh, const char *site, 
		     const char *var, const char *val)
{
    char *line = NULL, *newline = NULL;
    size_t l = 0, sitelen = 0, linelen = 0;
    char *test = NULL;

    if (!fh) {
	util_error("ERR: Invalid file handle");
	return -1;
    }

    if (val == NULL)
	val = "";

    /* Find site */
    if (site) {
	sitelen = strlen(site);
	test = malloc(sitelen + 3);
	test[0] = '[';
	memcpy(test + 1, site, sitelen);
	sitelen += 2;
	memcpy(test + sitelen - 1, "]", 2);

	for (;;) {
	    if (l == fh->lines) {
		/* End of file, add it */
		if (l > 0 && fh->linelen[l-1] > 0)
		    insert_line(fh, &l, "");
		insert_line(fh, &l, test);
		break;
	    } else {
		line = realloc_strcpy(line, fh->line[l++]);
		trim_right(line);
		linelen = strlen(line);
		if (linelen >= sitelen && strcmp(line, test) == 0) {
		    /* Found it */
		    break;
		}
	    }
	}

	free(test);
    }

    /* Find var */
    sitelen = strlen(var) + 1;
    linelen = strlen(val);
    test = malloc(sitelen + 1);
    newline = malloc(sitelen + linelen + 1);
    memcpy(test, var, sitelen - 1);
    memcpy(newline, var, sitelen - 1);
    memcpy(test + sitelen - 1, "=", 2);
    newline[sitelen - 1] = '=';
    memcpy(newline + sitelen, val, linelen + 1);   
    
    for (;;) {
	if (l == fh->lines) {
	    /* End of file, add it */
	    insert_line(fh, &l, newline);
	    break;
	} else {
	    line = realloc_strcpy(line, fh->line[l++]);	    
	    trim(line);
	    linelen = strlen(line);
	    if (linelen >= sitelen && strncmp(line, test, sitelen) == 0) {
		/* Found it */
		replace_line(fh, --l, newline);
		break;
	    } else if (site && linelen && line[0] == '[') {
                /* End of site, add it */
		l--;
		while (l > 0 && fh->linelen[l-1] == 0)
		    l--;
		insert_line(fh, &l, newline);
		break;
	    }	    
	}
    }

    if (line)
	free(line);
    free(test);
    free(newline);

    return 0;
}

void insert_line(edit_conf_t fh, size_t *cur, const char *newline)
{
    size_t l;

    if ((*cur) == fh->lines) {
	l = fh->lines++;
	fh->line = realloc(fh->line, fh->lines * sizeof(char *));
	fh->linelen = realloc(fh->linelen, fh->lines * sizeof(size_t));
    } else {
	l = (*cur);
	fh->lines++;
	fh->line = realloc(fh->line, fh->lines * sizeof(char *));
	fh->linelen = realloc(fh->linelen, fh->lines * sizeof(size_t));
	memmove(fh->line + l + 1, fh->line + l, 
		(fh->lines - (l + 1)) * sizeof(char *));
	memmove(fh->linelen + l + 1, fh->linelen + l, 
		(fh->lines - (l + 1)) * sizeof(size_t));
    }

    fh->linelen[l] = strlen(newline);
    fh->line[l] = malloc(fh->linelen[l] + 1);
    memcpy(fh->line[l], newline, fh->linelen[l] + 1);

    (*cur)++;
}

void replace_line(edit_conf_t fh, size_t cur, const char *newline)
{
    char *start = fh->line[cur], *end = fh->line[cur] + fh->linelen[cur];
    char *tmp = NULL;

    while (xisspace(start[0]))
	start++;
    while (end > start && xisspace(end[-1]))
	end--;
    
    tmp = realloc_strncpy(tmp, fh->line[cur], start - fh->line[cur]);
    tmp = realloc_strcat(tmp, newline);
    tmp = realloc_strcat(tmp, end);

    free(fh->line[cur]);
    fh->line[cur] = tmp;
    fh->linelen[cur] = strlen(tmp);
}
