/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <unistd.h>
#include <inttypes.h>

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#if !HAVE_GETLINE
# include "getline.h"
#endif
#include "str.h"
#include "parse_utils.h"
#include "gethosts.h"
#include "hosts_settings.h"
#include "util.h"

/* Set default configuration settings and return.
 * Secure connections will be used if possible as
 * default.
 */
static pftp_settings_t *host_default;

/* Init default settings. */
static void init_default2(void);
static pftp_settings_t *parse_hosts_file(FILE *file, char *conf, 
					 int *nr_settings);

/* Get hosts for the client. First look at ~/.pftphosts then /etc/pftphosts and
 * as last resort configuration defaults.
 */
pftp_settings_t *get_hosts(int *nr_settings)
{
    const char *home = getenv("HOME");
    char *conf = NULL;
    pftp_settings_t *ret = NULL;
    FILE *file = NULL;
    (*nr_settings) = 0;
    init_default2();
    if (home != NULL) {
        conf = alloc_strcpy(home);
	conf = realloc_strcat(conf, "/.pftphosts");
	file = fopen(conf, "r");
    }
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, checking /etc/pftphosts.\n", 
		   conf, strerror(errno));
	free(conf);
	conf = alloc_strcpy("/etc/pftphosts");
	file = fopen(conf, "r");
    }
    if (file == NULL) {
	util_error("Couldn't open file %s: %s, using defaults.\n", conf,
		   strerror(errno));
	free(conf);
	return host_default;
    }
    ret = parse_hosts_file(file, conf, nr_settings);
    free(conf);
    return ret;
}

void init_default2(void)
{
    host_default = NULL;
}

static int parse_host_row(char *line, pftp_settings_t conf, int line_count,
			  const char *file_name)
{
    int valid = 0;
    int len = strlen(line);

    if (len > 0 && line[len - 1] == '\n')
	line[len - 1] = '\0';

    if (!strncmp(line, "PASV=", 5)) {
	if (!(set_yes_or_no(line + 5, &(conf->pasv))))
	    return 0;
    } else if (!strncmp(line, "PORT_START=", 11)) {
	long port_start	= parse_port(line + 11, &valid);
	if (valid) {
	    conf->port_start = port_start;
	    return 0;
	}
    } else if (!strncmp(line, "PORT_STOP=", 10)) {
	long port_stop = parse_port(line + 10, &valid);
	if (valid) {
	    conf->port_stop = port_stop;
	    return 0;
	}
    } else if (!strncmp(line, "IF=", 3)) {
	rm_head_ws(line + 3, &conf->bind_to);
	if (check_interface(conf->bind_to)) {
	    free(conf->bind_to);
	    conf->bind_to = NULL;
	    util_error("%s:%d: Invalid interface `%s'", file_name, line_count, 
		       line + 3);   
	}
	return 0;
    } else if (!strncmp(line, "OUTWARD_IP=", 11)) {
	rm_head_ws(line + 11, &conf->myaddr);
	if (check_ip(conf->myaddr)) {
	    free(conf->myaddr);
	    conf->myaddr = NULL;
	    util_error("%s:%d: Invalid outward ip `%s'", file_name, line_count,
		       line + 11);   
	}
	return 0;
    } else if (!strncmp(line, "PRIO_LIST=", 10)) {
	conf->prio_list_len = split(line + 10, " ", &conf->prio_list, NULL);
	if (check_list((const char **)conf->prio_list, conf->prio_list_len)) {
	    free_split(&conf->prio_list, &conf->prio_list_len);
	    util_error("%s:%d: Invalid priority list `%s'", file_name, 
		       line_count, line + 10);
	}
	return 0;
    } else if (!strncmp(line, "IGNORE_LIST=", 12)) {
	conf->ignore_list_len = split(line + 12, " ", &conf->ignore_list,NULL);
	if (check_list((const char **)conf->ignore_list, 
		       conf->ignore_list_len)) {
	    free_split(&conf->ignore_list, &conf->ignore_list_len);
	    util_error("%s:%d: Invalid ignore list `%s'", file_name, 
		       line_count, line + 10);
	}
	return 0;
    } else if (!strncmp(line, "USE_SECURE=", 11)) {
	if (!(set_yes_or_no(line + 11, &(conf->use_secure))))
	    return 0;
    } else if (!strncmp(line, "SECURE_LIST=", 12)) {
	if (!(set_yes_or_no(line + 12, &(conf->secure_list))))
	    return 0;
    } else if (!strncmp(line, "SECURE_DATA=", 12)) {
	if (!(set_yes_or_no(line + 12, &(conf->secure_data))))
	    return 0;
    } else if (!strncmp(line, "IMPLICIT_SSL=", 13)) {
	if (!(set_yes_or_no(line + 13, &(conf->implicid_ssl))))
	    return 0;
    } else if (!strncmp(line, "USERNAME=", 9)) {
	trim(line + 9);
	conf->username = realloc_strcpy(conf->username, line + 9);
	return 0;
    } else if (!strncmp(line, "ACCOUNT=", 8)) {
	trim(line + 8);
	conf->account = realloc_strcpy(conf->account, line + 8);
	return 0;
    } else if (!strncmp(line, "INCLUDE_GLOBAL_PRIO=", 20)) {
	if (!(set_yes_or_no(line + 20, &(conf->include_global_prio))))
	    return 0;
    } else if (!strncmp(line, "INCLUDE_GLOBAL_IGNORE=", 22)) {
	if (!(set_yes_or_no(line + 22, &(conf->include_global_ignore))))
	    return 0;
    } else if (!strncmp(line, "SERVERS=", 8)) {
	if (!set_servers(line + 8, &(conf->hostname), &(conf->port),
			 &(conf->hosts))) {
	    return 0;
	} else {
	    util_error("%s:%d: No servers for host, skipping...", file_name,
		       line_count);
	    return -1;
	}
    } else if (line[0] == '[') {
	return 0;
    } else if (line[0] == '\0') {
	return 0;
    }
    util_error("%s:%d: Error parsing %s", file_name, line_count, line);
    return 0;
}

static void jump_comments(FILE *file, int *line_count)
{
    char* check = NULL;
    size_t buf = 0;
    ssize_t line_len = 0, read_len = 0;
    do {
	line_len = read_len = getline(&check, &buf, file);
	if (line_len == -1) break;
	trim(check);
	line_len = strlen(check);
	(*line_count)++;
    } while (line_len == 0 || (line_len > 0 && check[0] == '#'));
    if (check)
	free(check);
    (*line_count)--;
    if (line_len == -1)
	return;
    fseek(file, -read_len, SEEK_CUR);
}

static void set_host_defaults(pftp_settings_t host)
{
    memset(host, 0, sizeof (struct pftp_settings_s));
    host->username = alloc_strcpy("anonymous");
    host->pasv = 1;
    host->include_global_prio = 1;
    host->include_global_ignore = 1;
    host->use_secure = 1;
    host->secure_list = 1;
    host->secure_data = 1;
}

static void set_host_name(pftp_settings_t host, char *name, int name_len)
{
    host->name = malloc(name_len + 1);
    strcpy(host->name, name + 1);
    trim_right(host->name);
    host->name[strlen(host->name) - 1] = '\0';
}

static int parse_host(pftp_settings_t host, char *conf, int *line_count, 
		      FILE *file)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    do {
	jump_comments(file, line_count);
	line_len = getline(&line, &buf, file);
	(*line_count)++;
	if (parse_host_row(line, host, (*line_count), conf) == -1) {
	    free(line);
	    return -1;
	}
    } while (line_len == 0 || (line_len > 0 && !(line[0] == '[')));
    if (line_len > 0) {
	fseek(file, -line_len, SEEK_CUR);
	(*line_count)--;
    }
    free(line);
    return 0;
}

static int add_host(pftp_settings_t **hosts, int *nr_settings, char *conf,
		    int *line_count, FILE *file)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    pftp_settings_t new_host = malloc(sizeof(struct pftp_settings_s));
    do {
	line_len = getline(&line, &buf, file);
	(*line_count)++;
    } while (line_len == 0 || (line_len > 0 && line[0] != '['));
    set_host_defaults(new_host);
    set_host_name(new_host, line, line_len);
    if (line_len != -1) {
	if (!parse_host(new_host, conf, line_count, file)) {
	    (*hosts) = realloc((*hosts), 
			       ((*nr_settings) + 1) * sizeof(pftp_settings_t));
	    (*hosts)[(*nr_settings)++] = new_host;
	    free(line);
	    return 0;
	}
    }
    freeFTPSettings(&new_host);
    free(line);
    return -1;
}

static pftp_settings_t *parse_hosts_file(FILE *file, char *conf, 
					 int *nr_settings)
{
    char *line = NULL;
    size_t buf = 0;
    int line_len = 0;
    pftp_settings_t *ret = NULL;
    int line_count = 0;
    jump_comments(file, &line_count);
    line_len = getline(&line, &buf, file);
    trim_right(line);
    line_count++;
    if (strcmp(line, "[pftp-hosts]") != 0) {
	util_error("%s: Not a valid hosts file.", conf);
	free(line);
	return ret;
    }
    while (!feof(file)) {
	jump_comments(file, &line_count);
	add_host(&ret, nr_settings, conf, &line_count, file);
    }
    free(line);
    return ret;
}
