/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_SYS_TIME_H
# include <sys/time.h>
#endif

#ifndef NO_SSL
# include <openssl/ssl.h>
# include <openssl/rand.h>
# include <openssl/err.h>
#endif /* NO_SSL */

#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t; 
#endif

#include "pftp_default.h"
#include "pftp_settings.h"
#include "pftp.h"
#include "pftp_sftp.h"
#include "pftp_speed.h"
#include "pftp_internal.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

/* Static lib vars */
#ifndef NO_SSL
static SSL_CTX *ssl_ctx = NULL;
static X509_STORE *x509_store = NULL;
static char ssl_rand_file[256];
static int ftp_server_index = 0;

static int pftp_verify_SSL(int ok, X509_STORE_CTX *ctx);

int pftp_init_SSL(pftp_server_t ftp, SSL **ssl, socket_t sock)
{
    int retdata;
    
    pftp_status_message(ftp, "SSL: Init secure connection");
    
    (*ssl) = SSL_new(ssl_ctx);
    
    if (!(*ssl)) {
	pftp_status_message(ftp, "ERR: Unable to init SSL");
	return -2;
    }
    
    SSL_set_ex_data((*ssl), ftp_server_index, ftp);
    
    if (!SSL_set_fd((*ssl), (int) sock)) {
	pftp_status_message(ftp, "ERR: Unable to add socket to SSL");
	SSL_free((*ssl));
	(*ssl) = NULL;
	return -2;
    }
    
    SSL_ctrl((*ssl), SSL_CTRL_MODE, SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER, 0);
    
    if (ftp->cmdssl && (*ssl) != ftp->cmdssl)
	SSL_copy_session_id((*ssl), ftp->cmdssl);
    
    pftp_start_wait(ftp, SSL_HANDSHAKE, 0);
    
    for (;;) {
	retdata = SSL_connect((*ssl));
	
	if (retdata == 1) {
	    break;
	} else if (retdata == 0) {
	    ERR_print_errors_fp(stderr);
	    pftp_status_message(ftp, "WARN: SSL handshake was shutdown (%s)", 
				ERR_error_string(SSL_get_error((*ssl), 
							       retdata), 
						 NULL));
	    SSL_free((*ssl));
	    (*ssl) = NULL;
	    pftp_end_wait(ftp, SSL_HANDSHAKE, 1);
	    return -1;
	} else {
	    fd_set set;
	    struct timeval timeout;
	    
	    timeout.tv_sec = 0;
	    timeout.tv_usec = DEFAULT_TIMEOUT;
	    FD_ZERO(&set); 
	    FD_SET(sock, &set);
	    
	    if (BIO_sock_should_retry(retdata)) {
		if (SSL_want_read((*ssl))) {
		    for (;;) {
			retdata = select(FD_SETSIZE, &set, NULL, NULL, 
					 &timeout);
			if (retdata == -1 && errno == EINTR)
			    continue;
			break;
		    }
		} else if (SSL_want_write((*ssl))) {
		    for (;;) {
			retdata = select(FD_SETSIZE, NULL, &set, NULL, 
					 &timeout);
			if (retdata == -1 && errno == EINTR)
			    continue;
			break;
		    }
		} else {
		    if (SSL_want_x509_lookup((*ssl))) {
			puts("wants x509 lookup?");
		    } else {
			puts("What?!");
		    }	  
		}
	    } else {
		ERR_print_errors_fp(stderr);
		retdata = SSL_get_error((*ssl), retdata);
		pftp_status_message(ftp, 
				    "ERR: SSL returned connect error (%d)", 
				    retdata);
		SSL_free((*ssl));
		(*ssl) = NULL;
		pftp_end_wait(ftp, SSL_HANDSHAKE, 1);
		return -2;
	    }
	    
	    if (retdata == -1) {
		pftp_status_message(ftp, "ERR: Select returned error: %s", 
				    strerror(errno));
		SSL_free((*ssl));
		(*ssl) = NULL;
		pftp_end_wait(ftp, SSL_HANDSHAKE, 1);
		return -2;
	    } else if (retdata == 0) {
		if (pftp_do_wait(ftp, NULL, 0, "SSL_HANDSHAKE") == -1) {
		    SSL_free((*ssl));
		    (*ssl) = NULL;
		    pftp_end_wait(ftp, SSL_HANDSHAKE, 1);
		    return -1;
		}
	    } else {
		if (pftp_do_wait(ftp, NULL, 1, "SSL_HANDSHAKE") == -1) {
		    SSL_free((*ssl));
		    (*ssl) = NULL;
		    pftp_end_wait(ftp, SSL_HANDSHAKE, 1);
		    return -1;
		}
	    }      
	}
    }
    
    pftp_end_wait(ftp, SSL_HANDSHAKE, 0);
    
    return 0;
}

void pftp_init_libSSL(void)
{    
    SSL_load_error_strings();                /* readable error messages */
    SSL_library_init();                      /* initialize library */
    
    RAND_file_name(ssl_rand_file, sizeof(ssl_rand_file));
    
    if (RAND_egd(ssl_rand_file) < 0) {
	if (!RAND_load_file(ssl_rand_file, -1) || !RAND_status())
	    strcpy(ssl_rand_file, "");
    } else {
	strcpy(ssl_rand_file, "");
    }
    
#  if SSLEAY_VERSION_NUMBER < 0x0800
    ssl_ctx = SSL_CTX_new();
    
    if (ssl_ctx) {
	X509_set_default_verify_paths(ssl_ctx->cert);
    }
#  else
    //  SSLeay_add_ssl_algorithmns();
    ssl_ctx = SSL_CTX_new(SSLv23_client_method());
    
    if (ssl_ctx) {
	ftp_server_index = SSL_get_ex_new_index(0, "pftp server index", 
						NULL, NULL, NULL);

	SSL_CTX_set_options(ssl_ctx, SSL_OP_ALL);
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_PEER, pftp_verify_SSL);
	SSL_CTX_set_default_verify_paths(ssl_ctx);    
    }
#  endif /* SSLEAY_VERSION_NUMBER */ 
}

void pftp_free_libSSL(void)
{
    if (ssl_ctx) {
	SSL_CTX_free(ssl_ctx);
	ssl_ctx = NULL;
    }
    
    if (strlen(ssl_rand_file) > 0)
	RAND_write_file(ssl_rand_file);
}

/* Most of this code is based on source from lftp */
static int ssl_verify_crl(pftp_server_t ftp, X509_STORE_CTX *ctx)
{
    X509_OBJECT obj;
    X509_NAME *subject;
    X509_NAME *issuer;
    X509 *xs;
    X509_CRL *crl;
    X509_REVOKED *revoked;
    X509_STORE_CTX store_ctx;
    long serial;
    int i, n, rc;
    char *cp;

    pftp_status_message(ftp, "SSL: Verifying CERT...");
    
    if (!x509_store) {
	pftp_status_message(ftp, "OK (Nothing to check agains)");
	return 1;
    }

    xs      = X509_STORE_CTX_get_current_cert(ctx);
    subject = X509_get_subject_name(xs);
    issuer  = X509_get_issuer_name(xs);

    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, x509_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, subject, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    
    if (rc > 0 && crl != NULL) {
	if (X509_CRL_verify(crl, X509_get_pubkey(xs)) <= 0) {
	    pftp_status_message(ftp, "Invalid signature on CRL!");
	    X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_SIGNATURE_FAILURE);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	
	i = X509_cmp_current_time(X509_CRL_get_nextUpdate(crl));
	if (i == 0) {
	    pftp_status_message(ftp, 
			  "Found CRL has invalid nextUpdate field!");
	    X509_STORE_CTX_set_error(ctx, 
				     X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	if (i < 0) {
	    pftp_status_message(ftp, "Found CRL is expired!");
	    X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_HAS_EXPIRED);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	X509_OBJECT_free_contents(&obj);
    }
    
    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, x509_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, issuer, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    
    if (rc > 0 && crl != NULL) {
	n = sk_X509_REVOKED_num(X509_CRL_get_REVOKED(crl));
	for (i = 0; i < n; i++) {
	    revoked = sk_X509_REVOKED_value(X509_CRL_get_REVOKED(crl), i);
	    if (ASN1_INTEGER_cmp(revoked->serialNumber, 
				 X509_get_serialNumber(xs)) == 0) {
		serial = ASN1_INTEGER_get(revoked->serialNumber);
		cp = X509_NAME_oneline(issuer, NULL, 0);
		pftp_status_message(ftp, "Certificate with serial %ld (0x%lX) "
				    "revoked per CRL from issuer %s!", 
				    serial, serial, cp ? cp : "(ERROR)");
		free(cp);
		X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED);
		X509_OBJECT_free_contents(&obj);
		return 0;
	    }
	}
	X509_OBJECT_free_contents(&obj);
    }
    
    pftp_status_message(ftp, "OK");
    
    return 1;
}

int pftp_verify_SSL(int ok, X509_STORE_CTX *ctx)
{
    pftp_server_t ftp;
    SSL *ssl;

    ssl = X509_STORE_CTX_get_ex_data(ctx, SSL_get_ex_data_X509_STORE_CTX_idx());
    ftp = SSL_get_ex_data(ssl, ftp_server_index);   

    pftp_status_message(ftp, "SSL: Incoming CERT");
    
    if (ok) {
	if (!ssl_verify_crl(ftp, ctx))
	    ok = 0;
    } else {
	pftp_status_message(ftp, "SSL: CERT Accepted (ha-ha)");
	ok = 1;
    }
    
    return ok;
}

int pftp_ok_libSSL(void)
{
    return (ssl_ctx != NULL);
}
#endif /* NO_SSL */
