/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_SYS_TIME_H
# include <sys/time.h>
#endif

#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t;
#endif

#ifndef NO_SSL

#include <openssl/bn.h>
#include <openssl/ssl.h>

#include "pftp_default.h"
#include "pftp_settings.h"
#include "pftp.h"
#include "pftp_speed.h"
#include "pftp_sftp.h"
#include "pftp_internal.h"
#include "pftp_ssh_packet.h"
#include "pftp_ssh.h"
#include "pftp_ssh_channel.h"

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#define DEFAULT_SSH_TIMEOUT 5000 // in ms

#define WINDOW_SIZE_LOW 32767 //4096
#define WINDOW_SIZE 131072

#define SSH_OPEN_ADMINISTRATIVELY_PROHIBITED 1
#define SSH_OPEN_CONNECT_FAILED 2
#define SSH_OPEN_UNKNOWN_CHANNEL_TYPE 3
#define SSH_OPEN_RESOURCE_SHORTAGE 4

#define SSH_EXTENDED_DATA_STDERR 1

#define SSH_MSG_GLOBAL_REQUEST 80
#define SSH_MSG_REQUEST_SUCCESS 81
#define SSH_MSG_REQUEST_FAILURE 82
#define SSH_MSG_CHANNEL_OPEN 90
#define SSH_MSG_CHANNEL_OPEN_CONFIRMATION 91
#define SSH_MSG_CHANNEL_OPEN_FAILURE 92
#define SSH_MSG_CHANNEL_WINDOW_ADJUST 93
#define SSH_MSG_CHANNEL_DATA 94
#define SSH_MSG_CHANNEL_EXTENDED_DATA 95
#define SSH_MSG_CHANNEL_EOF 96
#define SSH_MSG_CHANNEL_CLOSE 97
#define SSH_MSG_CHANNEL_REQUEST 98
#define SSH_MSG_CHANNEL_SUCCESS 99
#define SSH_MSG_CHANNEL_FAILURE 100

#ifndef min
# define min(__x, __y) (((__x) < (__y)) ? (__x) : (__y))
#endif

struct pftp_ssh_channel_master_s
{
    pftp_ssh_t ssh;
    pftp_server_t ftp;
    pftp_ssh_channel_t *channel;
    size_t channels;
};

typedef enum { OPEN_REQ, OPEN_OK, OPEN_FAIL } channel_state_t;

typedef enum { REQ_WAITING, REQ_SUCCESS, REQ_FAILURE } request_status_t;

struct pftp_ssh_channel_s 
{
    pftp_ssh_channel_master_t master;
    size_t out_window_size, in_window_size, max_size;
    char *in_buf, *out_buf;
    size_t in_buf_len, out_buf_len;
    uint32_t send_id, recv_id;
    channel_state_t state;
    request_status_t request_status;
};

static void _close_channel(pftp_ssh_channel_master_t master, 
			   pftp_ssh_channel_t channel);
static pftp_ssh_channel_t _add_channel(pftp_ssh_channel_master_t master, 
				       size_t max_size);
static int _handle_channel_open(pftp_ssh_channel_master_t master,
				pftp_ssh_pkt_t pkg);
static int _handle_channel_request(pftp_ssh_channel_master_t master,
				   pftp_ssh_pkt_t pkg);
static int _handle_channel_open_confirmation(pftp_ssh_channel_master_t master,
					     pftp_ssh_pkt_t pkg);
static int _handle_channel_open_failure(pftp_ssh_channel_master_t master,
					pftp_ssh_pkt_t pkg);
static int _handle_channel_close(pftp_ssh_channel_master_t master,
				 pftp_ssh_pkt_t pkg);
static int _handle_channel_data(pftp_ssh_channel_master_t master,
				pftp_ssh_pkt_t pkg);
static int _handle_channel_success(pftp_ssh_channel_master_t master,
				   pftp_ssh_pkt_t pkg);
static int _handle_channel_failure(pftp_ssh_channel_master_t master,
				   pftp_ssh_pkt_t pkg);
static int _handle_channel_window_adjust(pftp_ssh_channel_master_t master,
					 pftp_ssh_pkt_t pkg);
static int _handle_channel_extended_data(pftp_ssh_channel_master_t master,
					 pftp_ssh_pkt_t pkg);
static int _handle_channel_eof(pftp_ssh_channel_master_t master,
			       pftp_ssh_pkt_t pkg);
static int _send_channel_data(pftp_ssh_channel_master_t master,
			      pftp_ssh_channel_t channel);
static pftp_ssh_channel_t _open_status_channel(pftp_ssh_channel_master_t master,
					       pftp_ssh_channel_t channel);

pftp_ssh_channel_master_t pftp_ssh_init_channel_master(pftp_ssh_t ssh, 
						       pftp_server_t ftp)
{
    pftp_ssh_channel_master_t ret;

    ret = malloc(sizeof(struct pftp_ssh_channel_master_s));
    memset(ret, 0, sizeof(struct pftp_ssh_channel_master_s));

    ret->ssh = ssh;
    ret->ftp = ftp;
    return ret;
}

void pftp_ssh_free_channel_master(pftp_ssh_channel_master_t master)
{
    if (master) {
	while (master->channels > 0)
	    _close_channel(master, master->channel[0]);
	
	free(master);
    }
}

int pftp_ssh_channel_master_touch(pftp_ssh_channel_master_t master, 
				  unsigned long timeout)
{
    pftp_ssh_pkt_t pkg;
    
    pkg = pftp_ssh_get(master->ssh, timeout);    

    if (!pkg) {
	return -1;
    }

    switch (pftp_ssh_pkt_msg(pkg)) {
    case SSH_MSG_CHANNEL_OPEN:
	if (_handle_channel_open(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_OPEN_CONFIRMATION:
	if (_handle_channel_open_confirmation(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}	
	break;  
    case SSH_MSG_CHANNEL_OPEN_FAILURE:
	if (_handle_channel_open_failure(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_DATA:
	if (_handle_channel_data(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_WINDOW_ADJUST:
	if (_handle_channel_window_adjust(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_EXTENDED_DATA:
	if (_handle_channel_extended_data(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}	
	break;   
    case SSH_MSG_CHANNEL_EOF:
	if (_handle_channel_eof(master, pkg)) {
       	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_CLOSE:
	if (_handle_channel_close(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_REQUEST:
	if (_handle_channel_request(master, pkg)) {
     	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_SUCCESS:
	if (_handle_channel_success(master, pkg)) {
       	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    case SSH_MSG_CHANNEL_FAILURE:
	if (_handle_channel_failure(master, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    return -1;
	}
	break;
    default:
#ifdef DEBUG
	fprintf(stderr, "SSH: Channels: Unhandled message `%u'.\n",
		pftp_ssh_pkt_msg(pkg));
#endif
	break;
    }

    pftp_ssh_pkt_free(pkg);
    return 0;
}

pftp_ssh_channel_t pftp_ssh_open_channel_session(
    pftp_ssh_channel_master_t master, size_t max_size)
{
    pftp_ssh_channel_t ret;
    ret = _add_channel(master, max_size);
    if (ret) {
	pftp_ssh_pkt_t pkg;
	pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_OPEN);
	pftp_ssh_pkt_put_string(pkg, "session");
	pftp_ssh_pkt_put_uint32(pkg, ret->send_id);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->in_window_size);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->max_size);
	
	if (pftp_ssh_send(master->ssh, pkg, 0)) {
	    pftp_ssh_pkt_free(pkg);
	    _close_channel(master, ret);
	    return NULL;
	}
	pftp_ssh_pkt_free(pkg);

	return _open_status_channel(master, ret);
    } else {
	return NULL;
    }
}

pftp_ssh_channel_t pftp_ssh_open_channel_x11(
    pftp_ssh_channel_master_t master, size_t max_size,
    const char *orig_addr, uint16_t orig_port)
{
    pftp_ssh_channel_t ret;
    ret = _add_channel(master, max_size);
    if (ret) {
	pftp_ssh_pkt_t pkg;
	pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_OPEN);
	pftp_ssh_pkt_put_string(pkg, "x11");
	pftp_ssh_pkt_put_uint32(pkg, ret->send_id);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->in_window_size);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->max_size);
	pftp_ssh_pkt_put_string(pkg, orig_addr);
	pftp_ssh_pkt_put_uint32(pkg, orig_port);
	
	if (pftp_ssh_send(master->ssh, pkg, 0)) {
	    pftp_ssh_pkt_free(pkg);
	    _close_channel(master, ret);
	    return NULL;
	}
	pftp_ssh_pkt_free(pkg);

	return _open_status_channel(master, ret);
    } else {
	return NULL;
    }
}

pftp_ssh_channel_t pftp_ssh_open_channel_tcpip(
    pftp_ssh_channel_master_t master, size_t max_size,
    int forward,
    const char *dst_addr, uint16_t dst_port,
    const char *src_addr, uint16_t src_port)
{
    pftp_ssh_channel_t ret;
    ret = _add_channel(master, max_size);
    if (ret) {
	pftp_ssh_pkt_t pkg;
	pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_OPEN);
	pftp_ssh_pkt_put_string(pkg, 
				forward ? "forwarded-tcpip" : "direct-tcpip");
	pftp_ssh_pkt_put_uint32(pkg, ret->send_id);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->in_window_size);
	pftp_ssh_pkt_put_uint32(pkg, (uint32_t) ret->max_size);
	pftp_ssh_pkt_put_string(pkg, dst_addr);
	pftp_ssh_pkt_put_uint32(pkg, dst_port);
	pftp_ssh_pkt_put_string(pkg, src_addr);
	pftp_ssh_pkt_put_uint32(pkg, src_port);
	
	if (pftp_ssh_send(master->ssh, pkg, 0)) {
	    pftp_ssh_pkt_free(pkg);
	    _close_channel(master, ret);
	    return NULL;
	}
	pftp_ssh_pkt_free(pkg);

	return _open_status_channel(master, ret);
    } else {
	return NULL;
    }
}

void pftp_ssh_channel_close(pftp_ssh_channel_t channel)
{
    _close_channel(channel->master, channel);
}

int pftp_ssh_channel_request_subsystem(pftp_ssh_channel_t channel,
				       int want_reply, const char *subsystem)
{
    pftp_ssh_pkt_t pkg;
    assert(channel->state == OPEN_OK);

    pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_REQUEST);
    pftp_ssh_pkt_put_uint32(pkg, channel->recv_id);
    pftp_ssh_pkt_put_string(pkg, "subsystem");
    if (want_reply) {
	channel->request_status = REQ_WAITING;
	pftp_ssh_pkt_put_char(pkg, '\1');
    } else {
	pftp_ssh_pkt_put_char(pkg, '\0');
    }
    pftp_ssh_pkt_put_string(pkg, subsystem);
    
    if (pftp_ssh_send(channel->master->ssh, pkg, 0)) {
	pftp_ssh_pkt_free(pkg);
	return -1;
    }

    pftp_ssh_pkt_free(pkg);

    if (want_reply) {
	while (channel->request_status == REQ_WAITING) {
	    if (pftp_ssh_channel_master_touch(channel->master, 0)) {
		return -1;
	    }	    
	}

	if (channel->request_status == REQ_SUCCESS) { 
	    return 0;
	} else {
	    pftp_status_message(channel->master->ftp,
				"SSH: Channel: "
				"Request for subsystem `%s' failed.",
				subsystem);
	    return -1;
	}
    }

    return 0;
}

ssize_t pftp_ssh_channel_write(pftp_ssh_channel_t channel, const void *data,
			       size_t size, speed_data_t speed)
{
    channel->out_buf = realloc(channel->out_buf, channel->out_buf_len + size);
    memcpy(channel->out_buf + channel->out_buf_len, data, size);
    channel->out_buf_len += size;

    if (_send_channel_data(channel->master, channel))
	return -1;

    if (speed) {
	if (pftp_do_wait(channel->master->ftp, speed, 1, "SFTP-UPLOAD") == -1) {
	    return -1;
	}
    }

    return 0;
}

ssize_t pftp_ssh_channel_read(pftp_ssh_channel_t channel, void *data, 
			      size_t size,speed_data_t speed)
{
    while (channel->in_buf_len < size) {
	size_t need, old;
	need = size - channel->in_buf_len;

	if (need > channel->in_window_size 
	    || channel->in_window_size < WINDOW_SIZE_LOW) {
	    pftp_ssh_pkt_t pkg;
	    uint32_t add;
	    if (need < WINDOW_SIZE)
		add = (uint32_t) (WINDOW_SIZE - channel->in_window_size);
	    else
		add = (uint32_t) (need - channel->in_window_size);
	    pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_WINDOW_ADJUST);
	    pftp_ssh_pkt_put_uint32(pkg, channel->recv_id);
	    pftp_ssh_pkt_put_uint32(pkg, add);
	    if (pftp_ssh_send(channel->master->ssh, pkg, 0)) {
		pftp_ssh_pkt_free(pkg);
		return -1;
	    }
	    channel->in_window_size += add;
	    pftp_ssh_pkt_free(pkg);
	}
	
	old = channel->in_buf_len;

	if (pftp_ssh_channel_master_touch(channel->master, 
					  DEFAULT_SSH_TIMEOUT))
	    return -1;	
	if (speed) {
	    if (pftp_do_wait(channel->master->ftp, speed, 
			     1, "SFTP-DOWNLOAD") == -1) {
		return -1;
	    }
	}
    }

    memcpy(data, channel->in_buf, size);
    channel->in_buf_len -= size;
    memmove(channel->in_buf, channel->in_buf + size, channel->in_buf_len);

    return 0;
}

void _close_channel(pftp_ssh_channel_master_t master, 
		    pftp_ssh_channel_t channel)
{
    size_t c;

    if (channel->state == OPEN_OK) {
	pftp_ssh_pkt_t pkg;
	pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_CLOSE);
	pftp_ssh_pkt_put_uint32(pkg, channel->recv_id);
	pftp_ssh_send(master->ssh, pkg, 0);
	pftp_ssh_pkt_free(pkg);
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c] == channel) {
	    break;
	}
    }

    if (c < master->channels) {
	master->channels--;
	memmove(master->channel + c, master->channel + c + 1, 
		(master->channels - c) * sizeof(pftp_ssh_channel_t));
    }

    if (channel->in_buf)
	free(channel->in_buf);
    if (channel->out_buf)
	free(channel->out_buf);
    free(channel);
}

pftp_ssh_channel_t _add_channel(pftp_ssh_channel_master_t master, 
				size_t max_size)
{
    pftp_ssh_channel_t ret;
    size_t c;
    ret = malloc(sizeof(struct pftp_ssh_channel_s));
    memset(ret, 0, sizeof(struct pftp_ssh_channel_s));

    ret->master = master;
    ret->in_window_size = WINDOW_SIZE;
    ret->out_window_size = 0;
    ret->max_size = max_size;

    ret->send_id = 1;

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id >= ret->send_id) {
	    ret->send_id = master->channel[c]->send_id + 1;
	}
    }

    master->channel = realloc(master->channel, (master->channels + 1) 
			      * sizeof(pftp_ssh_channel_t));
    master->channel[master->channels++] = ret;

    return ret;
}

int _handle_channel_open(pftp_ssh_channel_master_t master,
			 pftp_ssh_pkt_t pkg)
{
    char *type;
    uint32_t id;
    int valid;
    pftp_ssh_pkt_t resp;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_OPEN);
    pftp_ssh_pkt_reset(pkg);

    type = pftp_ssh_pkt_get_string(pkg);
    id = pftp_ssh_pkt_get_uint32(pkg, &valid);

    if (!type || !valid) {
	pftp_status_message(master->ftp, "SSH: Channel: Invalid OPEN message.");
	if (type) free(type);
	return -1;
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: Server trying to open channel for `%s'.\n",
	    type);
#endif
    free(type);

    resp = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_OPEN_FAILURE);
    pftp_ssh_pkt_put_uint32(resp, id);
    pftp_ssh_pkt_put_uint32(resp, SSH_OPEN_ADMINISTRATIVELY_PROHIBITED);
    pftp_ssh_pkt_put_string(resp, "Only a client, nothing more.");
    pftp_ssh_pkt_put_string(resp, "");

    if (pftp_ssh_send(master->ssh, resp, 0)) {
	pftp_ssh_pkt_free(resp);
	return -1;
    }

    pftp_ssh_pkt_free(resp);

    return 0;
}

int _handle_channel_request(pftp_ssh_channel_master_t master,
			    pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    char *type, reply;
    int valid_id, valid_char;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_REQUEST);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);
    type = pftp_ssh_pkt_get_string(pkg);
    reply = pftp_ssh_pkt_get_char(pkg, &valid_char);

    if (!valid_id || !type || !valid_char) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid REQUEST message.");
	if (type) free(type);
	return -1;
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: Server trying to request `%s'.\n",
	    type);
#endif
    free(type);

    if (reply) {
	pftp_ssh_pkt_t resp;

	resp = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_FAILURE);
	pftp_ssh_pkt_put_uint32(resp, id);
	
	if (pftp_ssh_send(master->ssh, resp, 0)) {
	    pftp_ssh_pkt_free(resp);
	    return -1;
	}

	pftp_ssh_pkt_free(resp);
    }

    return 0;
}

int _handle_channel_open_confirmation(pftp_ssh_channel_master_t master,
				      pftp_ssh_pkt_t pkg)
{
    uint32_t recv_id, send_id, wnd_size, max_size;
    int valid_rid, valid_sid, valid_wnd_size, valid_max_size;	
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_OPEN_CONFIRMATION);
    pftp_ssh_pkt_reset(pkg);
    
    recv_id = pftp_ssh_pkt_get_uint32(pkg, &valid_rid);
    send_id = pftp_ssh_pkt_get_uint32(pkg, &valid_sid);
    wnd_size = pftp_ssh_pkt_get_uint32(pkg, &valid_wnd_size);
    max_size = pftp_ssh_pkt_get_uint32(pkg, &valid_max_size);

    if (!valid_rid || !valid_sid || !valid_wnd_size || !valid_max_size) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid OPEN_CONFIRMATION message.");
	return -1;
    }

    for (c = 0; c < master->channels; c++) {
	if (recv_id == master->channel[c]->send_id) {
	    if (master->channel[c]->state == OPEN_REQ) {
		master->channel[c]->out_window_size = wnd_size;
		if (master->channel[c]->max_size > max_size)
		    master->channel[c]->max_size = max_size;
		master->channel[c]->recv_id = send_id;
		master->channel[c]->state = OPEN_OK;

		return 0;
	    } else {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "Getting confirmation on a already open"
				    " channel.");
		return -1;
	    }
	}
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got OPEN_CONFIRMATION for nonexistent channel.");
#endif

    return 0;
}

int _handle_channel_open_failure(pftp_ssh_channel_master_t master,
				 pftp_ssh_pkt_t pkg)
{
    uint32_t recv_id, reason;
    int valid_id, valid_reason;
    char *desc, *lang;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_OPEN_FAILURE);
    pftp_ssh_pkt_reset(pkg);

    recv_id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);
    reason = pftp_ssh_pkt_get_uint32(pkg, &valid_reason);
    desc = pftp_ssh_pkt_get_string(pkg);
    lang = pftp_ssh_pkt_get_string(pkg);

    if (!valid_id || !valid_reason || !desc || !lang) {
	if (desc) free(desc);
	if (lang) free(lang);
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid OPEN_FAILURE message.");
	return -1;
    }

    for (c = 0; c < master->channels; c++) {
	if (recv_id == master->channel[c]->send_id) {
	    if (master->channel[c]->state == OPEN_REQ) {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "Open failed `%s' (%lu).",
				    desc, reason);
		master->channel[c]->state = OPEN_FAIL;
		free(desc);
		free(lang);
		return 0;
	    } else {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "Getting failure on a already open "
				    "channel.");
		free(desc);
		free(lang);
		return -1;
	    }
	}
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got OPEN_FAILURE for nonexistent channel.");
#endif

    free(desc);
    free(lang);
    return 0;
}

int _handle_channel_close(pftp_ssh_channel_master_t master,
			  pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    int valid_id;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_CLOSE);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);

    if (!valid_id) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid CLOSE message.");
	return -1;	
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    _close_channel(master, master->channel[c]);
	    return 0;
	}
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got CLOSE for nonexistent channel.");
#endif

    return 0;    
}

int _handle_channel_data(pftp_ssh_channel_master_t master,
			 pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    int valid_id;
    char *data;
    size_t data_len, c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_DATA);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);
    data = pftp_ssh_pkt_get_string2(pkg, &data_len);

    if (!valid_id || !data) {
	if (data) free(data);
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid DATA message.");
	return -1;	
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    if (master->channel[c]->max_size < data_len) {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "DATA is bigger than MAX (%lu > %lu).",
				    data_len, master->channel[c]->max_size);	
	    } else if (master->channel[c]->in_window_size < data_len) {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "DATA is bigger than WINDOW (%lu > %lu).",
				    data_len, 
				    master->channel[c]->in_window_size);	
	    } else {	
		master->channel[c]->in_buf 
		    = realloc(master->channel[c]->in_buf,
			      master->channel[c]->in_buf_len + data_len);
		memcpy(master->channel[c]->in_buf 
		       + master->channel[c]->in_buf_len,
		       data, data_len);
		master->channel[c]->in_buf_len += data_len;
		master->channel[c]->in_window_size -= data_len;
	    }

	    free(data);

	    return 0;
	}
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got DATA for nonexistent channel.");
#endif 

    free(data);

    return 0;    
}

int _handle_channel_success(pftp_ssh_channel_master_t master,
			    pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    int valid_id;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_SUCCESS);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);

    if (!valid_id) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid SUCCESS message.");
	return -1;	
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    if (master->channel[c]->request_status == REQ_WAITING) {
		master->channel[c]->request_status = REQ_SUCCESS;
		return 0;
	    } else {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "Getting SUCCES on non-waiting channel.");
		return -1;
	    }
	}
    }
    
#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got SUCCESS for nonexistent channel.");
#endif 

    return 0;    
}

int _handle_channel_failure(pftp_ssh_channel_master_t master,
			    pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    int valid_id;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_FAILURE);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);

    if (!valid_id) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid FAILURE message.");
	return -1;	
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    if (master->channel[c]->request_status == REQ_WAITING) {
		master->channel[c]->request_status = REQ_FAILURE;
		return 0;
	    } else {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "Getting FAILURE on non-waiting channel.");
		return -1;
	    }
	}
    }
    
#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got FAILURE for nonexistent channel.");
#endif 

    return 0;
}

int _handle_channel_window_adjust(pftp_ssh_channel_master_t master,
				  pftp_ssh_pkt_t pkg)
{
    uint32_t id, bytes;
    int valid_id, valid_bytes;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_WINDOW_ADJUST);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);
    bytes = pftp_ssh_pkt_get_uint32(pkg, &valid_bytes);

    if (!valid_id || !valid_bytes) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid WINDOW_ADJUST message.");
	return -1;	
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    master->channel[c]->out_window_size += bytes;
	    if (_send_channel_data(master, master->channel[c])) {
		_close_channel(master, master->channel[c]);
		return -1;
	    }
	    return 0;
	}
    }
    
#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got WINDOW_ADJUST for nonexistent channel.");
#endif 

    return 0;
}

int _handle_channel_extended_data(pftp_ssh_channel_master_t master,
				  pftp_ssh_pkt_t pkg)
{
    uint32_t id, type;
    int valid_id, valid_type;
    char *data;
    size_t data_len, c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_EXTENDED_DATA);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);
    type = pftp_ssh_pkt_get_uint32(pkg, &valid_type);
    data = pftp_ssh_pkt_get_string2(pkg, &data_len);

    if (!valid_id || !valid_type || !data) {
	if (data) free(data);
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid EXTENDED_DATA message.");
	return -1;	
    }

    if (type != SSH_EXTENDED_DATA_STDERR) {
#ifdef DEBUG
	fprintf(stderr, "SSH: Channel: "
		"Got unknown type of EXTENDED_DATA (%lu).", 
		(unsigned long)type);
#endif 
	free(data);
	return 0;
    }
    
    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    if (master->channel[c]->max_size < data_len) {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "EXT_DATA is bigger than MAX (%lu > %lu).",
				    data_len, master->channel[c]->max_size);	
	    } else if (master->channel[c]->in_window_size < data_len) {
		pftp_status_message(master->ftp, "SSH: Channel: "
				    "EXT_DATA is bigger than WND (%lu > %lu).",
				    data_len, 
				    master->channel[c]->in_window_size);    	
	    } else {	
		pftp_status_message(master->ftp, "SSH: STDERR: `%s'.",
				    data);
		master->channel[c]->in_window_size -= data_len;
	    }
	    free(data);
	    return 0;
	}
    }
    
#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got EXTENDED_DATA for nonexistent channel.");
#endif 

    free(data);
    return 0;
}

int _handle_channel_eof(pftp_ssh_channel_master_t master,
			pftp_ssh_pkt_t pkg)
{
    uint32_t id;
    int valid_id;
    size_t c;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_CHANNEL_EOF);
    pftp_ssh_pkt_reset(pkg);

    id = pftp_ssh_pkt_get_uint32(pkg, &valid_id);

    if (!valid_id) {
	pftp_status_message(master->ftp, "SSH: Channel: "
			    "Invalid EOF message.");
	return -1;
    }

    for (c = 0; c < master->channels; c++) {
	if (master->channel[c]->send_id == id) {
	    /* EOF... */
	    return 0;
	}
    }

#ifdef DEBUG
    fprintf(stderr, "SSH: Channel: "
	    "Got EOF for nonexistent channel.");
#endif 

    return 0;
}

int _send_channel_data(pftp_ssh_channel_master_t master,
		       pftp_ssh_channel_t channel)
{
    size_t size;
    pftp_ssh_pkt_t pkg;
    
    assert(channel->state == OPEN_OK);

    size = min(channel->out_buf_len, min(channel->out_window_size,
					 channel->max_size));

    if (size == 0) {
	return 0;
    }
	
    pkg = pftp_ssh_create_pkt(SSH_MSG_CHANNEL_DATA);
    pftp_ssh_pkt_put_uint32(pkg, channel->recv_id);
    pftp_ssh_pkt_put_string2(pkg, channel->out_buf, size);
    if (pftp_ssh_send(master->ssh, pkg, DEFAULT_SSH_TIMEOUT)) {
	pftp_ssh_pkt_free(pkg);
	return -1;
    }

    pftp_ssh_pkt_free(pkg);
    channel->out_window_size -= size;
    channel->out_buf_len -= size;
    memmove(channel->out_buf, channel->out_buf + size, channel->out_buf_len);

    return 0;
}

pftp_ssh_channel_t _open_status_channel(pftp_ssh_channel_master_t master,
					pftp_ssh_channel_t channel)
{
    while (channel->state == OPEN_REQ) {
	if (pftp_ssh_channel_master_touch(master, 0)) {
	    _close_channel(master, channel);
	    return NULL;
	}	
    }

    if (channel->state == OPEN_OK) {
	return channel;
    } else {
	_close_channel(master, channel);
	return NULL;
    }
}

#endif /* NO_SSL */
