/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#if HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif
#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif

#ifndef NO_SSL

#include <openssl/ssl.h>
#include <openssl/hmac.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/bn.h>
#include <openssl/dh.h>


#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t;
#define closesocket close
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#include "pftp_settings.h"
#include "pftp.h"
#include "pftp_ssh_packet.h"
#include "pftp_ssh.h"
#include "pftp_ssl.h"
#include "pftp_sftp.h"
#include "pftp_speed.h"
#include "pftp_internal.h"
#include "pftp_utf8.h"
#include "pftp_arc4random.h"
#include "pftp_dh.h"
#include "pftp_ssh_buf.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#define SSH_MSG_DISCONNECT 1
#define SSH_MSG_IGNORE 2
#define SSH_MSG_UNIMPLEMENTED 3
#define SSH_MSG_DEBUG 4
#define SSH_MSG_SERVICE_REQUEST 5
#define SSH_MSG_SERVICE_ACCEPT 6
#define SSH_MSG_KEXINIT 20
#define SSH_MSG_NEWKEYS 21
#define SSH_MSG_KEXDH_INIT 30
#define SSH_MSG_KEXDH_REPLY 31

#define CLIENT_VERSION_STR "SSH-2.0-pftp_0.6"
#define CLIENT_KEX_LIST "diffie-hellman-group14-sha1,diffie-hellman-group1-sha1"
#define CLIENT_HOSTKEY_LIST "ssh-rsa,ssh-dss"
#define CLIENT_ENC_CTOS_LIST "aes128-cbc,3des-cbc,blowfish-cbc"
#define CLIENT_ENC_STOC_LIST "aes128-cbc,3des-cbc,blowfish-cbc"
#define CLIENT_MAC_CTOS_LIST "hmac-md5,hmac-sha1,hmac-ripemd160"
#define CLIENT_MAC_STOC_LIST "hmac-md5,hmac-sha1,hmac-ripemd160"
#define CLIENT_COMP_CTOS_LIST "none"
#define CLIENT_COMP_STOC_LIST "none"

typedef enum { MAC_NONE = 0, MAC_SHA1, MAC_MD5, MAC_RIPEMD160 } mac_algo_t;
typedef enum { COMP_NONE = 0 } comp_algo_t;
typedef enum { KEX_NONE = 0, 
	       KEX_DIFF_HELL_GRP1_SHA1,
	       KEX_DIFF_HELL_GRP14_SHA1 } kex_algo_t;
typedef enum { KEY_NONE = 0, KEY_SSH_DSS, KEY_SSH_RSA } key_algo_t;
typedef enum { ENC_NONE = 0, ENC_3DES, ENC_AES128, ENC_BLOWFISH } enc_algo_t;
typedef enum { STATE_INIT = 0, STATE_CONNECTED, STATE_DEAD } state_t;
typedef enum { INIT_KEXINIT = 0, INIT_KEYEXCH, INIT_NEWKEYS } init_state_t;

typedef struct {
    key_algo_t algo;
    RSA *rsa;
    DSA *dsa;
} hostkey_t, *phostkey_t;

typedef struct {
    unsigned char *data;
    size_t size;
} raw_data_t;

typedef struct {
    EVP_CIPHER_CTX evp;
    const raw_data_t *key, *iv;
} enc_data_t, *penc_data_t;

struct pftp_ssh_s {
    pftp_server_t ftp;    
    mac_algo_t my_mac_algo, srv_mac_algo, new_my_mac_algo, new_srv_mac_algo;
    kex_algo_t kex_algo, new_kex_algo;
    key_algo_t key_algo, new_key_algo;
    phostkey_t my_key, srv_key;
    raw_data_t my_mac_key, srv_mac_key;
    raw_data_t my_enc_key, srv_enc_key;
    raw_data_t my_enc_iv, srv_enc_iv;
    raw_data_t my_kexinit, srv_kexinit;
    comp_algo_t my_comp_algo, new_my_comp_algo;
    comp_algo_t srv_comp_algo, new_srv_comp_algo;
    enc_algo_t my_enc_algo, new_my_enc_algo, srv_enc_algo, new_srv_enc_algo;

    size_t need_size;
    DH *my_dh;
    
    char *server, *server_version_str;
    uint32_t my_sequence_number, srv_sequence_number;

    int srv_sent_keyinit;

    penc_data_t srv_enc, my_enc;

    int service_ok;
    char *service_waiting;

    raw_data_t session_id;

    state_t state;
    init_state_t init_state;
};

static void _free_hostkey(phostkey_t key);
static void _free_enc_data(penc_data_t enc);
static void _enc_data_init(penc_data_t *enc, enc_algo_t enc_algo,
			   const raw_data_t *key, const raw_data_t *iv,
			   int do_encrypt);
static void _crypt_data(penc_data_t enc, enc_algo_t algo, unsigned char *data, 
			size_t size);
static phostkey_t _hostkey_from_blob(const char *data, size_t len);
static __inline size_t _enc_blocklen(enc_algo_t enc);
static __inline size_t _enc_keylen(enc_algo_t enc);
static __inline size_t _mac_keylen(mac_algo_t mac);
static int _send_hello(pftp_ssh_t ssh);
static int _get_hello(pftp_ssh_t ssh);
static int _safe_write(pftp_ssh_t ssh, const void *data, size_t len,
		       unsigned long timeout);
static int _safe_read(pftp_ssh_t ssh, void *data, size_t len, 
		      unsigned long timeout);
static __inline int _safe_writestr(pftp_ssh_t ssh, const char *data,
				   unsigned long timeout);
static int _send_keyinit(pftp_ssh_t ssh);
static int _send_keyrequest(pftp_ssh_t ssh);
#if 0
static int _send_unimplemented(pftp_ssh_t ssh, uint32_t sequence_number);
#endif
static int _send_disconnect(pftp_ssh_t ssh);
static int _send_newkeys(pftp_ssh_t ssh);
static void _handle_kexinit(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_kexdh_reply(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_newkeys(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_disconnect(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_ignore(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_debug(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_unimplemented(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _handle_service_accept(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static int _send_service_request(pftp_ssh_t ssh, const char *service);
static int _generate_public_key(pftp_ssh_t ssh);
static size_t _calc_mac_len(pftp_ssh_t ssh, mac_algo_t mac_algo);
static void _calc_mac(pftp_ssh_t ssh, 
		      mac_algo_t mac_algo, 
		      raw_data_t key,
		      uint32_t sequence_number, 
		      const unsigned char *packet, uint32_t packet_length, 
		      unsigned char **mac, size_t mac_len);
static int _get_matching_namelist(const char *client_list,
				  const char *server_list, char **algo);
static pftp_ssh_pkt_t _ssh_read(pftp_ssh_t ssh, unsigned long timeout);
static int _ssh_write(pftp_ssh_t ssh, const pftp_ssh_pkt_t pkg,
		      unsigned long timeout);
static int _handle_ssh_packet(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg);
static void _kill_ssh(pftp_ssh_t ssh);

pftp_ssh_t pftp_ssh_connect(pftp_server_t ftp)
{
    pftp_ssh_t ret;
    ret = malloc(sizeof(struct pftp_ssh_s));
    memset(ret, 0, sizeof(struct pftp_ssh_s));

    ret->ftp = ftp;
    
    if (_send_hello(ret)) {
	closesocket(ret->ftp->cmdstream);
	ret->ftp->cmdstream = -1;
	free(ret);
	return NULL;
    }
    if (_get_hello(ret)) {
	closesocket(ret->ftp->cmdstream);
	ret->ftp->cmdstream = -1;
	free(ret);
	return NULL;
    }
    if (_send_keyinit(ret)) {
	closesocket(ret->ftp->cmdstream);
	ret->ftp->cmdstream = -1;
	free(ret);
	return NULL;
    }

    while (ret->state == STATE_INIT) {
	pftp_ssh_pkt_t pkg = NULL;

	if (!(pkg = _ssh_read(ret, 0))) {
	    pftp_status_message(ret->ftp, 
				"SSH: Connection aborted during init.");
	    closesocket(ret->ftp->cmdstream);
	    ret->ftp->cmdstream = -1;
	    free(ret);
	    return NULL;
	}

	if (_handle_ssh_packet(ret, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	} else {
	    pftp_ssh_pkt_free(pkg);
	    pftp_status_message(ret->ftp,
				"SSH: Invalid packet during init found.");
	    closesocket(ret->ftp->cmdstream);
	    ret->ftp->cmdstream = -1;
	    free(ret);
	    return NULL;
	}
    }

    if (ret->state == STATE_DEAD) {
	pftp_status_message(ret->ftp,
			    "SSH: Connection closed during init.");
	closesocket(ret->ftp->cmdstream);
	ret->ftp->cmdstream = -1;
	free(ret);
	return NULL;
    }

    return ret;
}

void pftp_ssh_close(pftp_ssh_t *ssh)
{
    if (*ssh) {
	if ((*ssh)->server_version_str) {
	    free((*ssh)->server_version_str);
	}
	if ((*ssh)->server) {
	    free((*ssh)->server);
	}
	if ((*ssh)->my_dh) {
	    DH_free((*ssh)->my_dh);
	}
	if ((*ssh)->srv_key) {
	    _free_hostkey((*ssh)->srv_key);
	}
	if ((*ssh)->my_key) {
	    _free_hostkey((*ssh)->my_key);
	}
	if ((*ssh)->session_id.data) {
	    free((*ssh)->session_id.data);
	}
	if ((*ssh)->srv_kexinit.data) {
	    free((*ssh)->srv_kexinit.data);
	}
	if ((*ssh)->my_kexinit.data) {
	    free((*ssh)->my_kexinit.data);
	}
	if ((*ssh)->srv_mac_key.data) {
	    free((*ssh)->srv_mac_key.data);
	}
	if ((*ssh)->my_mac_key.data) {
	    free((*ssh)->my_mac_key.data);
	}
	if ((*ssh)->srv_enc) {
	    _free_enc_data((*ssh)->srv_enc);
	}
	if ((*ssh)->my_enc) {
	    _free_enc_data((*ssh)->my_enc);
	}
	if ((*ssh)->srv_enc_key.data) {
	    free((*ssh)->srv_enc_key.data);
	}
	if ((*ssh)->my_enc_key.data) {
	    free((*ssh)->my_enc_key.data);
	}
	if ((*ssh)->srv_enc_iv.data) {
	    free((*ssh)->srv_enc_iv.data);
	}
	if ((*ssh)->my_enc_iv.data) {
	    free((*ssh)->my_enc_iv.data);
	}
	if ((*ssh)->service_waiting) {
	    free((*ssh)->service_waiting);
	}

	closesocket((*ssh)->ftp->cmdstream);
	(*ssh)->ftp->cmdstream = -1;

	free(*ssh);
	*ssh = NULL;
    }
}

int _send_hello(pftp_ssh_t ssh)
{
    return _safe_writestr(ssh, CLIENT_VERSION_STR "\r\n", 0);
}

int _get_hello(pftp_ssh_t ssh)
{
    char *buf;
    size_t buflen, pos;
    ssize_t ret;

    pos = 0;
    buflen = 100;
    buf = malloc(buflen);
    
    for (;;) {
	ret = recv(ssh->ftp->cmdstream, buf + pos, 1, 0);
	if (ret < 0) {
#ifdef WIN32
	    if (WSAGetLastError() == WSAEWOULDBLOCK || WSAGetLastError() == WSAEINTR) {
		continue;
	    } else {
		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to read from socket: `%s'",
				    strerror(errno));
		free(buf);
		return -1;				    
	    }
#else
	    if (errno == EAGAIN || errno == EINTR || errno == EWOULDBLOCK) {
		continue;
	    } else {
		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to read from socket: `%s'",
				    strerror(errno));
		free(buf);
		return -1;				    
	    }
#endif
	} else if (ret == 0) {
	    pftp_status_message(ssh->ftp, 
				"ERR: Unable to read from socket: "
				"`Socket closed'");
	    free(buf);
	    return -1;
	} else {
	    if (buf[pos] == '\n') {
		if (buf[pos - 1] == '\r') {
		    pos--;
		}

		buf[pos] = '\0';

		if (strncmp(buf, "SSH-", 4) == 0) {		   	
		    size_t start;
		    if ((strncmp(buf + 4, "2.0-", 4) == 0 && (start = 8)) ||
			(strncmp(buf + 4, "1.99-", 5) == 0 && (start = 9))) {
			/* OK */
			char *sp;
			ssh->server_version_str = strdup(buf);

			if ((sp = strchr(buf + start, ' '))) {
			    ssh->server = strndup(buf + start, 
						  sp - (buf + start));
			} else {
			    ssh->server = strdup(buf + start);
			}
			pftp_status_message(ssh->ftp,
					    "SSH: Talking to %s...",
					    ssh->server);
			free(buf);
			return 0;
		    } else {
			pftp_status_message(ssh->ftp, 
					    "ERR: Not a SSH 2.0 server: `%s'.", 
					    buf);
			free(buf);
			return -1;
		    }
		} else {
		    /* Allowed to display text first. */
		    pftp_parse_unicode_utf8(buf);
		    pftp_status_message(ssh->ftp, "SSH: %s\n", buf);
		    pos = 0;
		    continue;
		}
	    }

	    if ((++pos) == buflen) {
		buflen *= 2;
		if (buflen > 1000) {
		    pftp_status_message(ssh->ftp, 
					"ERR: To much identification data "
					"(%lu bytes)", (buflen / 2));
		    free(buf);
		    return -1;
		}
		buf = realloc(buf, buflen);
	    }
	}
    }
}

int _safe_write(pftp_ssh_t ssh, const void *data, size_t len, 
		unsigned long timeout)
{
    size_t pos;
    ssize_t ret;
    fd_set write_set;
    struct timeval *to, _to;
    pos = 0;
    while (pos < len) {
	ret = send(ssh->ftp->cmdstream, 
		   (const char *) data + pos, (int) (len - pos), 0);
	if (ret < 0) {
	    int nonfatal = 0;
#ifdef WIN32
	    if (WSAGetLastError() == WSAEINTR) {
		continue;	    
	    } else if (WSAGetLastError() == WSAEWOULDBLOCK) {	
		nonfatal = 1;
	    }
#else
	    if (errno == EINTR) {
		continue;	    
	    } else if (errno == EAGAIN || errno == EWOULDBLOCK) {	
		nonfatal = 1;
	    }
#endif
	    if (nonfatal) {
		FD_ZERO(&write_set);
		FD_SET(ssh->ftp->cmdstream, &write_set);
		if (timeout > 0) {
		    to = &_to;
		    to->tv_sec = timeout / 1000;
		    to->tv_usec = (timeout % 1000) * 1000;
		} else {
		    to = NULL;
		}
		for (;;) {
		    ret = select(FD_SETSIZE, NULL, &write_set, NULL, to);
		    if (ret < 0) {
			if (errno == EINTR) {
			    continue;
			}
			    
			pftp_status_message(ssh->ftp, 
					    "ERR: Unable to select from"
					    " socket: `%s'", 
					    strerror(errno));	
			return -1;
		    } else if (ret == 0) {
			if (timeout > 0) {
			    /* Timeout */
			    pftp_status_message(ssh->ftp, 
						"ERR: Socket timeout "
						"during write.");
			    return -1;
			} else {
			    /* Should never happen? */
			    continue;
			}
		    } else {
			break;
		    }
		}				
	    } else {
		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to write to socket: `%s'",
				    strerror(errno));
		return -1;				    
	    }
	} if (ret == 0) {
	    /* Closed */
	    pftp_status_message(ssh->ftp, 
				"ERR: Unable to read from socket: "
				"`Socket closed'");	    
	    return -1;
	} else {
	    pos += ret;
	}
    }

    return 0;
}

__inline int _safe_writestr(pftp_ssh_t ssh, const char *data, 
			    unsigned long timeout)
{
    return _safe_write(ssh, data, strlen(data), timeout);
}

pftp_ssh_pkt_t _ssh_read(pftp_ssh_t ssh, unsigned long timeout)
{
    uint32_t packet_length = 0;
    uint8_t padding_length = 0;
    size_t cbc_size = 0;
    unsigned char *packet = NULL, *mac = NULL;
    size_t mac_len = 0;
    pftp_ssh_pkt_t ret = NULL;

    cbc_size = _enc_blocklen(ssh->srv_enc_algo);
    packet = malloc(cbc_size);

    if (_safe_read(ssh, packet, cbc_size, timeout)) {
	free(packet);
	return ret;
    }
	
    _crypt_data(ssh->srv_enc, ssh->srv_enc_algo, packet, cbc_size);

    memcpy(&packet_length, packet, 4);
    packet_length = ntohl(packet_length);
    
    padding_length = packet[4];

    if (((4 + packet_length) % cbc_size) > 0) {
	pftp_status_message(ssh->ftp, 
			    "SSH: Invalid packet found (%lu % %lu > 0)",
			    4 + packet_length, cbc_size);
	free(packet);
	return ret;
    }

    packet = realloc(packet, 4 + packet_length);
    
    if (_safe_read(ssh, packet + cbc_size, (4 + packet_length) - cbc_size,
		   timeout)) {
	free(packet);
	return ret;
    }

    _crypt_data(ssh->srv_enc, ssh->srv_enc_algo, packet + cbc_size, 
		(4 + packet_length) - cbc_size);

    mac_len = _calc_mac_len(ssh, ssh->srv_mac_algo);

    _calc_mac(ssh, ssh->srv_mac_algo, ssh->srv_mac_key,
	      ssh->srv_sequence_number, packet, packet_length + 4, 
	      &mac, mac_len);
    
    if (mac && mac_len) {
	char *_mac;

	_mac = malloc(mac_len);

	if (_safe_read(ssh, _mac, mac_len, timeout)) {
	    free(packet);
	    free(mac);
	    free(_mac);
	    return ret;
	}		

	if (memcmp(_mac, mac, mac_len) != 0) {
	    pftp_status_message(ssh->ftp, "SSH: MAC missmatch.");
	    free(packet);
	    free(mac);
	    free(_mac);
	    return ret;	    
	}

	free(_mac);
    }

    if (mac)
	free(mac);

    /* !!! Need to redo this when supporting compression. */

    ret = pftp_ssh_load_pkt(packet + 4 + 1, 
			    packet_length - (1 + padding_length));
    free(packet);
    
    switch (ssh->srv_comp_algo) {
    case COMP_NONE:
	break;
    }    
	
    ssh->srv_sequence_number++;
	
    return ret;
}

int _ssh_write(pftp_ssh_t ssh, const pftp_ssh_pkt_t pkt, unsigned long timeout)
{
    uint32_t packet_length = 0;
    uint8_t padding_length = 0;
    size_t cbc_size = 0, tmp_size = 0;
    unsigned char *packet = NULL, *mac = NULL;
    size_t payload_len = 0, mac_len = 0;

    switch (ssh->my_comp_algo) {
    case COMP_NONE: {
	const char *data;
	data = pftp_ssh_pkt_save(pkt, &payload_len);
	packet = malloc(4 + 1 + payload_len);
	memcpy(packet + 4 + 1, data, payload_len);
    }; break;
    }    

    cbc_size = _enc_blocklen(ssh->my_enc_algo);
    /* uint32 + byte + payload  */
    tmp_size = (4 + 1 + payload_len);
    padding_length = (uint8_t) (cbc_size - (tmp_size % cbc_size));    
    /* adding "random padding" */
    padding_length += (uint8_t) (cbc_size * (size_t)(rand() % 5));
    /* padding min is 4 */
    if (padding_length < 4)
	padding_length += (uint8_t) cbc_size;

    packet_length = (uint32_t) (1 + payload_len + padding_length);

    packet_length = htonl(packet_length);
    memcpy(packet, &packet_length, 4);
    packet_length = ntohl(packet_length);

    packet[4] = padding_length;

    packet = realloc(packet, 4 + packet_length);

    if (ssh->my_enc_algo != ENC_NONE) {
	for (tmp_size = 4 + packet_length - padding_length; 
	     tmp_size < 4 + packet_length; tmp_size++) {
	    packet[tmp_size] = (char)(rand() % 256);
	}
    } else {
	memset(packet + (4 + packet_length - padding_length), 0,
	       padding_length);
    }

    mac_len = _calc_mac_len(ssh, ssh->my_mac_algo);

    _calc_mac(ssh, ssh->my_mac_algo, ssh->my_mac_key,
	      ssh->my_sequence_number, packet, packet_length + 4, 
	      &mac, mac_len);

    _crypt_data(ssh->my_enc, ssh->my_enc_algo, packet, 4 + packet_length);
    
    if (_safe_write(ssh, packet, 4 + packet_length, timeout)) {
	free(packet);
	if (mac) free(mac);
	return -1;
    }

    free(packet);

    if (mac && mac_len) {
	if (_safe_write(ssh, mac, mac_len, timeout)) {
	    free(mac);
	    return -1;
	}
    }

    if (mac)
	free(mac);

    ssh->my_sequence_number++;

    return 0;
}

int _send_keyinit(pftp_ssh_t ssh)
{
    pftp_ssh_pkt_t pkg;
    int ret;
    size_t s;
    uint32_t rnd = 0;
	
    pkg = pftp_ssh_create_pkt(SSH_MSG_KEXINIT);

    for (s = 0; s < 16; ++s) {
	if (s % 4 == 0)
	    rnd = pftp_arc4random();
	pftp_ssh_pkt_put_char(pkg, rnd);
	rnd >>= 8;
    }
    /* key exchange */
    pftp_ssh_pkt_put_string(pkg, CLIENT_KEX_LIST);
    /* server hostkey */
    pftp_ssh_pkt_put_string(pkg, CLIENT_HOSTKEY_LIST);
    /* encryption client -> server */
    pftp_ssh_pkt_put_string(pkg, CLIENT_ENC_CTOS_LIST);
    /* encryption server -> client */
    pftp_ssh_pkt_put_string(pkg, CLIENT_ENC_STOC_LIST);
    /* mac client -> server */
    pftp_ssh_pkt_put_string(pkg, CLIENT_MAC_CTOS_LIST);
    /* mac server -> client */
    pftp_ssh_pkt_put_string(pkg, CLIENT_MAC_STOC_LIST);
    /* compression client -> server */
    pftp_ssh_pkt_put_string(pkg, CLIENT_COMP_CTOS_LIST);
    /* compression server -> client */
    pftp_ssh_pkt_put_string(pkg, CLIENT_COMP_STOC_LIST);
    /* languages client -> server */
    pftp_ssh_pkt_put_string(pkg, "");
    /* languages server -> client */
    pftp_ssh_pkt_put_string(pkg, "");
    /* first_kex_packet_follows */
    pftp_ssh_pkt_put_char(pkg, '\0');
    pftp_ssh_pkt_put_uint32(pkg, 0);

    ret = _ssh_write(ssh, pkg, 0);

    if (ret == 0) {
	if (ssh->my_kexinit.data) 
	    free(ssh->my_kexinit.data);	
		
	ssh->my_kexinit.size = pftp_ssh_pkt_size(pkg) - 1;
	ssh->my_kexinit.data = malloc(ssh->my_kexinit.size);
	memcpy(ssh->my_kexinit.data, 
	       pftp_ssh_pkt_get_raw(pkg, ssh->my_kexinit.size), 
	       ssh->my_kexinit.size);
    }

    pftp_ssh_pkt_free(pkg);

    return ret;
}

int _send_keyrequest(pftp_ssh_t ssh)
{
    pftp_ssh_pkt_t pkg;
    int ret;
    pkg = pftp_ssh_create_pkt(SSH_MSG_KEXDH_INIT);

    if (!ssh->my_dh) {
	if (_generate_public_key(ssh)) {
	    pftp_status_message(ssh->ftp, 
				"SSH: Unable to generate Public Key.");
	}
    }

    pftp_ssh_pkt_put_bignum(pkg, ssh->my_dh->pub_key);
    ret = _ssh_write(ssh, pkg, 0);
    pftp_ssh_pkt_free(pkg);
    return ret;
}

static kex_algo_t _parse_kex_algo(const char *algo)
{
    if (strcmp(algo, "diffie-hellman-group14-sha1") == 0) {
	return KEX_DIFF_HELL_GRP14_SHA1;
    } else if (strcmp(algo, "diffie-hellman-group1-sha1") == 0) {
	return KEX_DIFF_HELL_GRP1_SHA1;
    } else {
	assert(0);
	return KEX_NONE;
    }
}

static key_algo_t _parse_key_algo(const char *algo)
{
    if (strcmp(algo, "ssh-rsa") == 0) {
	return KEY_SSH_RSA;
    } else if (strcmp(algo, "ssh-dss") == 0) {
	return KEY_SSH_DSS;
    } else {
	assert(0);
	return KEY_NONE;
    }
}

static enc_algo_t _parse_enc_algo(const char *algo)
{
    if (strcmp(algo, "none") == 0) {
	return ENC_NONE;
    } else if (strcmp(algo, "aes128-cbc") == 0) {
	return ENC_AES128;
    } else if (strcmp(algo, "3des-cbc") == 0) {
	return ENC_3DES;
    } else if (strcmp(algo, "blowfish-cbc") == 0) {
	return ENC_BLOWFISH;
    } else {
	assert(0);
	return ENC_NONE;
    }
}

static mac_algo_t _parse_mac_algo(const char *algo)
{
    if (strcmp(algo, "hmac-md5") == 0) {
	return MAC_MD5;
    } else if (strcmp(algo, "hmac-sha1") == 0) {
	return MAC_SHA1;
    } else if (strcmp(algo, "hmac-ripemd160") == 0) {
	return MAC_RIPEMD160;
    } else {
	assert(0);
	return MAC_NONE;
    }
}

static comp_algo_t _parse_comp_algo(const char *algo)
{
    if (strcmp(algo, "none") == 0) {
	return COMP_NONE;
    } else {
	assert(0);
	return COMP_NONE;
    }
}

void _handle_kexinit(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    char *algo = NULL, *tmpstr = NULL;
    int guessed_ok = 0, valid;	
    int kexinit_sent;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_KEXINIT);
    pftp_ssh_pkt_reset(pkg);

    if (ssh->state != STATE_INIT) {
	pftp_status_message(ssh->ftp, "SSH: Exchange of keys started.");
	ssh->state = STATE_INIT;
	ssh->init_state = INIT_KEXINIT;
    }

    if (ssh->init_state != INIT_KEXINIT) {
	pftp_status_message(ssh->ftp, "SSH: KEXINIT when not expected.");
	_kill_ssh(ssh);
	return;
    }

    /* Cookie */
    if (!pftp_ssh_pkt_get_raw(pkg, 16)) 
	goto kex_packet_to_small;    
	
    /* Kex algos */    
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    
	
    if (_get_matching_namelist(CLIENT_KEX_LIST, tmpstr, &algo)) {
	guessed_ok = 1;
    } else {
	guessed_ok = 0;
    }
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common kex_algorithm found. `%s'",
			    tmpstr);
	goto kex_common_fail;
    }

    ssh->new_kex_algo = _parse_kex_algo(algo);
    free(tmpstr);

    /* Server hostkey algos */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)    
	goto kex_packet_to_small;    
	
    _get_matching_namelist(CLIENT_HOSTKEY_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common hostkey format found. `%s'",
			    tmpstr);
	goto kex_common_fail;
    }

    ssh->new_key_algo = _parse_key_algo(algo);
    free(tmpstr);

    /* Encryption algos Client -> Server */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_ENC_CTOS_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common encryption client -> server"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }

    ssh->new_my_enc_algo = _parse_enc_algo(algo);
    free(tmpstr);

    if (ssh->new_my_enc_algo == ENC_NONE) {
	pftp_status_message(ssh->ftp,
			    "SSH: Only suiatable encryption client -> server "
			    "was NONE which we won\'t allow.");
	goto kex_common_fail;
    }

    /* Encryption algos Server -> Client */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_ENC_STOC_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common encryption server -> client"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }

    ssh->new_srv_enc_algo = _parse_enc_algo(algo);
    free(tmpstr);

    /* MAC algos Client -> Server */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_MAC_CTOS_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common MAC hash function client -> server"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }

    ssh->new_my_mac_algo = _parse_mac_algo(algo);
    free(tmpstr);

    /* MAC algos Server -> Client */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_MAC_STOC_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common MAC hash function server -> client"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }

    ssh->new_srv_mac_algo = _parse_mac_algo(algo);
    free(tmpstr);

    /* Compression algos Client -> Server */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_COMP_CTOS_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common compression client -> server"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }

    ssh->new_my_comp_algo = _parse_comp_algo(algo);
    free(tmpstr);

    /* Compression algos Server -> Client */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;    

    _get_matching_namelist(CLIENT_COMP_STOC_LIST, tmpstr, &algo);
    if (algo == NULL) {
	pftp_status_message(ssh->ftp, 
			    "SSH: No common compression server -> client"
			    " found. `%s'", tmpstr);
	goto kex_common_fail;
    }
    
    ssh->new_srv_comp_algo = _parse_comp_algo(algo);
    free(tmpstr);

    if (algo) {
	free(algo);	
	algo = NULL;
    }

    /* Languages Client -> Server */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;

    free(tmpstr);

    /* Languages Server -> Client */
    tmpstr = pftp_ssh_pkt_get_string(pkg);
    if (!tmpstr)
	goto kex_packet_to_small;

    free(tmpstr);

    /* Was KeyInit sent? */
    kexinit_sent = (pftp_ssh_pkt_get_char(pkg, &valid) == '\1');
    if (!valid)
	goto kex_packet_to_small;

    if (kexinit_sent && !guessed_ok)
	ssh->srv_sent_keyinit++;

    pftp_ssh_pkt_get_uint32(pkg, &valid);
    if (!valid)
	goto kex_packet_to_small;

    if (ssh->srv_kexinit.data) 
	free(ssh->srv_kexinit.data);	

    pftp_ssh_pkt_reset(pkg);
    ssh->srv_kexinit.size = pftp_ssh_pkt_size(pkg) - 1;
    ssh->srv_kexinit.data = malloc(ssh->srv_kexinit.size);
    memcpy(ssh->srv_kexinit.data, 
	   pftp_ssh_pkt_get_raw(pkg, ssh->srv_kexinit.size), 
	   ssh->srv_kexinit.size);
	
    ssh->init_state = INIT_KEYEXCH;
    if (_send_keyrequest(ssh)) {
	_kill_ssh(ssh);
    }

    return;

kex_packet_to_small:
    pftp_status_message(ssh->ftp, 
			"SSH: KEXINIT packet to small or has invalid "
			"content.");	
kex_common_fail:
    if (algo) free(algo);	
    _kill_ssh(ssh);
}

static int _mangle_keys(pftp_ssh_t ssh, const BIGNUM *dh_server_pub, 
			const char *srv_host_key, size_t srv_key_len,
			const char *signature, size_t sig_len);

void _handle_kexdh_reply(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    char *signature = NULL, *srv_key = NULL;
    size_t sig_len = 0, srv_key_len = 0;
    BIGNUM *dh_server_pub = NULL;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_KEXDH_REPLY);
    pftp_ssh_pkt_reset(pkg);

    if (ssh->state != STATE_INIT) {
	pftp_status_message(ssh->ftp, 
			    "SSH: KEYExchange Reply while not in INIT.");
	_kill_ssh(ssh);
	return;
    }
	
    if (ssh->init_state != INIT_KEYEXCH) {
	pftp_status_message(ssh->ftp, 
			    "SSH: KEYExchange Reply while not waiting for it.");
	_kill_ssh(ssh);
	return;
    }
	
    /* server public host key and certificates (K_S) */    
    srv_key = pftp_ssh_pkt_get_string2(pkg, &srv_key_len);
    if (!srv_key)
	goto keyexchange_packet_to_small;

    if (ssh->srv_key) {
	_free_hostkey(ssh->srv_key);
	ssh->srv_key = NULL;
    }

    ssh->srv_key = _hostkey_from_blob(srv_key, srv_key_len);

    if (!ssh->srv_key) {
	pftp_status_message(ssh->ftp,
			    "SSH: Unable to read server hostkey.");
	_kill_ssh(ssh);
	free(srv_key);
	return;
    }

    if (ssh->srv_key->algo != ssh->new_key_algo) {
	pftp_status_message(ssh->ftp,
			    "SSH: Wrong type of hostkey from server.");
	_kill_ssh(ssh);
	free(srv_key);
	return;
    }

    /* !!! Verify hostkey anyone? */

    /* f (dh_server_pub) */
    dh_server_pub = pftp_ssh_pkt_get_bignum(pkg);
    if (!dh_server_pub)
	goto keyexchange_packet_to_small;

    if (!pftp_dh_pub_is_valid(ssh->my_dh, dh_server_pub)) {
	pftp_status_message(ssh->ftp,
			    "SSH: Bad server public DH value.");
	_kill_ssh(ssh);
	free(srv_key);;
	BN_free(dh_server_pub);
	return;
    }

    /* signature of H */
    signature = pftp_ssh_pkt_get_string2(pkg, &sig_len);
    if (!signature)
	goto keyexchange_packet_to_small;

    /* do the wacky */
    if (_mangle_keys(ssh, dh_server_pub, srv_key, srv_key_len, 
		     signature, sig_len)) {
	_kill_ssh(ssh);
	free(srv_key);
	free(signature);
	BN_free(dh_server_pub);
	return;
    }

    free(srv_key);
    free(signature);
    BN_free(dh_server_pub);
	
    ssh->init_state = INIT_NEWKEYS;

    if (_send_newkeys(ssh)) {
	_kill_ssh(ssh);
	return;
    }
    
    return;

keyexchange_packet_to_small:
    pftp_status_message(ssh->ftp, 
			"SSH: KEYExchange packet to small or has invalid "
			"content.");	
    if (srv_key) free(srv_key);
    if (signature) free(signature);
    if (dh_server_pub) BN_free(dh_server_pub);
    _kill_ssh(ssh);
}

static const u_char *_kex_dh_hash(const char *client_version_string,
				  const char *server_version_string,
				  const unsigned char *ckexinit, 
				  size_t ckexinitlen,
				  const unsigned char *skexinit, 
				  size_t skexinitlen,
				  const char *serverhostkeyblob, 
				  size_t sbloblen,
				  const BIGNUM *client_dh_pub,
				  const BIGNUM *server_dh_pub,
				  const BIGNUM *shared_secret)
{
    pftp_ssh_buf_t buf;
    static unsigned char digest[EVP_MAX_MD_SIZE];
    const EVP_MD *evp_md = EVP_sha1();
    EVP_MD_CTX md;

    buf = pftp_ssh_create_buf(); 

    pftp_ssh_buf_put_string(buf, client_version_string);
    pftp_ssh_buf_put_string(buf, server_version_string);

    /* kexinit messages: fake header: len+SSH_MSG_KEXINIT */
    pftp_ssh_buf_put_uint32(buf, (uint32_t) (ckexinitlen + 1));
    pftp_ssh_buf_put_char(buf, SSH_MSG_KEXINIT);
    pftp_ssh_buf_put_raw(buf, ckexinit, ckexinitlen);
    pftp_ssh_buf_put_uint32(buf, (uint32_t) (skexinitlen + 1));
    pftp_ssh_buf_put_char(buf, SSH_MSG_KEXINIT);
    pftp_ssh_buf_put_raw(buf, skexinit, skexinitlen);

    pftp_ssh_buf_put_string2(buf, (const char *) serverhostkeyblob, sbloblen);
    pftp_ssh_buf_put_bignum(buf, client_dh_pub);
    pftp_ssh_buf_put_bignum(buf, server_dh_pub);
    pftp_ssh_buf_put_bignum(buf, shared_secret);

    EVP_DigestInit(&md, evp_md);
    EVP_DigestUpdate(&md, pftp_ssh_buf_raw(buf), pftp_ssh_buf_size(buf));
    EVP_DigestFinal(&md, digest, NULL);

    pftp_ssh_buf_free(buf);

    return digest;
}

static raw_data_t _derive_key(pftp_ssh_t ssh, char id, size_t need, 
			      const char *hash, const BIGNUM *shared_secret)
{
    pftp_ssh_buf_t buf;
    const EVP_MD *evp_md = EVP_sha1();
    EVP_MD_CTX md;
    char c = id;
    size_t have;
    size_t mdsz = EVP_MD_size(evp_md);
    raw_data_t digest;

    assert(mdsz >= 0);
    assert(ssh->session_id.data);

    digest.size = roundup(need, mdsz);
    digest.data = malloc(digest.size);

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_bignum(buf, shared_secret);

    /* K1 = HASH(K || H || "A" || session_id) */
    EVP_DigestInit(&md, evp_md);
    EVP_DigestUpdate(&md, pftp_ssh_buf_raw(buf), pftp_ssh_buf_size(buf));
    EVP_DigestUpdate(&md, hash, mdsz);
    EVP_DigestUpdate(&md, &c, 1);
    EVP_DigestUpdate(&md, ssh->session_id.data, ssh->session_id.size);
    EVP_DigestFinal(&md, digest.data, NULL);

    /*
     * expand key:
     * Kn = HASH(K || H || K1 || K2 || ... || Kn-1)
     * Key = K1 || K2 || ... || Kn
     */
    for (have = mdsz; need > have; have += mdsz) {
	EVP_DigestInit(&md, evp_md);
	EVP_DigestUpdate(&md, pftp_ssh_buf_raw(buf), pftp_ssh_buf_size(buf));
	EVP_DigestUpdate(&md, hash, mdsz);
	EVP_DigestUpdate(&md, digest.data, have);
	EVP_DigestFinal(&md, digest.data + have, NULL);
    }

    pftp_ssh_buf_free(buf);

    return digest;
}

static void _kex_derive_keys(pftp_ssh_t ssh, const char *hash, 
			     const BIGNUM *shared_secret)
{
    if (ssh->my_enc_iv.data)
	free(ssh->my_enc_iv.data);
    if (ssh->srv_enc_iv.data)
	free(ssh->srv_enc_iv.data);
    if (ssh->my_enc_key.data)
	free(ssh->my_enc_key.data);
    if (ssh->srv_enc_key.data)
	free(ssh->srv_enc_key.data);
    if (ssh->my_mac_key.data)
	free(ssh->my_mac_key.data);
    if (ssh->srv_mac_key.data)
	free(ssh->srv_mac_key.data);

    assert(ssh->need_size > 0);

    ssh->my_enc_iv = _derive_key(ssh, 'A', ssh->need_size, hash, 
				 shared_secret);
    ssh->srv_enc_iv = _derive_key(ssh, 'B', ssh->need_size, hash, 
				  shared_secret);
    ssh->my_enc_key = _derive_key(ssh, 'C', ssh->need_size, hash, 
				  shared_secret);
    ssh->srv_enc_key = _derive_key(ssh, 'D', ssh->need_size, hash, 
				   shared_secret);
    ssh->my_mac_key = _derive_key(ssh, 'E', ssh->need_size, hash, 
				  shared_secret);
    ssh->srv_mac_key = _derive_key(ssh, 'F', ssh->need_size, hash, 
				   shared_secret);
}

#define INTBLOB_LEN	20
#define SIGBLOB_LEN	(2*INTBLOB_LEN)

static int _ssh_dss_verify(const phostkey_t key, 
			   const char *signature, size_t sig_len,
			   const char *data, size_t datalen)
{
    pftp_ssh_buf_t buf;
    DSA_SIG *sig;
    const EVP_MD *evp_md = EVP_sha1();
    EVP_MD_CTX md;
    unsigned char digest[EVP_MAX_MD_SIZE];
    unsigned char *sigblob = NULL;
    size_t len;
    unsigned int dlen;
    int ret;
    char *ktype = NULL;

    if (key == NULL || key->algo != KEY_SSH_DSS || key->dsa == NULL) {
	assert(0);
	return -1;
    }

    buf = pftp_ssh_attach_buf(signature, sig_len);

    /* ietf-drafts */
    ktype = pftp_ssh_buf_get_string(buf);
    if (!ktype) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (strcmp("ssh-dss", ktype)) {
#ifdef DEBUG
	fprintf(stderr, "ssh_dss_verify: cannot handle type `%s'.\n", ktype);
#endif
	free(ktype);
	pftp_ssh_buf_free(buf);
	return -1;
    }
    free(ktype);
    sigblob = (unsigned char *) pftp_ssh_buf_get_string2(buf, &len);
    pftp_ssh_buf_free(buf);

    if (!sigblob) {
	return -1;
    }
    if (pftp_ssh_buf_data_left(buf)) {
#ifdef DEBUG
	fprintf(stderr, 
		"ssh_dss_verify: remaining bytes in signature.\n");
#endif
	free(sigblob);
	return -1;
    }	

    if (len != SIGBLOB_LEN) {
	assert(0);
	free(sigblob);
	return -1;
    }

    /* parse signature */
    sig = DSA_SIG_new();
    sig->r = BN_new();
    sig->s = BN_new();
    BN_bin2bn(sigblob, INTBLOB_LEN, sig->r);
    BN_bin2bn(sigblob + INTBLOB_LEN, INTBLOB_LEN, sig->s);
	
    /* clean up */
    memset(sigblob, 0, len);
    free(sigblob);

    /* sha1 the data */
    EVP_DigestInit(&md, evp_md);
    EVP_DigestUpdate(&md, data, datalen);
    EVP_DigestFinal(&md, digest, &dlen);

    ret = DSA_do_verify(digest, dlen, sig, key->dsa);
    memset(digest, 'd', sizeof(digest));

    DSA_SIG_free(sig);

    return ret;
}

/*
 * See:
 * http://www.rsasecurity.com/rsalabs/pkcs/pkcs-1/
 * ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1.asn
 */
/*
 * id-sha1 OBJECT IDENTIFIER ::= { iso(1) identified-organization(3)
 *	oiw(14) secsig(3) algorithms(2) 26 }
 */
static const u_char id_sha1[] = {
    0x30, 0x21, /* type Sequence, length 0x21 (33) */
    0x30, 0x09, /* type Sequence, length 0x09 */
    0x06, 0x05, /* type OID, length 0x05 */
    0x2b, 0x0e, 0x03, 0x02, 0x1a, /* id-sha1 OID */
    0x05, 0x00, /* NULL */
    0x04, 0x14  /* Octet string, length 0x14 (20), followed by sha1 hash */
};

static int _openssh_RSA_verify(const unsigned char *hash, size_t hashlen,
			       const unsigned char *sigbuf, size_t siglen, 
			       RSA *rsa)
{
    size_t ret, rsasize, oidlen = 0, hlen = 0;
    int len;
    const u_char *oid = NULL;
    unsigned char *decrypted = NULL;

    ret = 0;
    oid = id_sha1;
    oidlen = sizeof(id_sha1);
    hlen = 20;
    if (hashlen != hlen) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: bad hashlen.\n");
#endif
	goto done;
    }
    rsasize = RSA_size(rsa);
    if (siglen == 0 || siglen > rsasize) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: bad siglen.\n");
#endif		
	goto done;
    }
    decrypted = malloc(rsasize);
    if ((len = RSA_public_decrypt((int) siglen, sigbuf, decrypted, rsa,
				  RSA_PKCS1_PADDING)) < 0) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: RSA_public_decrypt failed: %s.\n",
		ERR_error_string(ERR_get_error(), NULL));
#endif
	goto done;
    }
    if (len < 0 || (u_int)len != hlen + oidlen) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: bad decrypted len: %d != %lu + %lu.\n",
		len, (unsigned long) hlen, (unsigned long) oidlen);
#endif
	goto done;
    }
    if (memcmp(decrypted, oid, oidlen) != 0) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: oid mismatch.\n");
#endif		
	goto done;
    }
    if (memcmp(decrypted + oidlen, hash, hlen) != 0) {
#ifdef DEBUG
	fprintf(stderr, "RSA_verify: hash mismatch.\n");
#endif	   
	goto done;
    }
    ret = 1;
done:
    if (decrypted)
	free(decrypted);
    return (int) ret;
}

#define SSH_RSA_MINIMUM_MODULUS_SIZE  768

static int _ssh_rsa_verify(const phostkey_t key, const char *signature, 
			   size_t sig_len, const char *data, size_t datalen)
{
    pftp_ssh_buf_t buf;
    const EVP_MD *evp_md = EVP_sha1();
    EVP_MD_CTX md;
    char *ktype = NULL;
    unsigned char digest[EVP_MAX_MD_SIZE], *sigblob = NULL;
    size_t len, modlen;
    unsigned int dlen;
    int ret;

    if (key == NULL || key->algo != KEY_SSH_RSA || key->rsa == NULL) {
	assert(0);
	return -1;
    }

    if (BN_num_bits(key->rsa->n) < SSH_RSA_MINIMUM_MODULUS_SIZE) {
	assert(0);
	return -1;
    }

    buf = pftp_ssh_attach_buf(signature, sig_len);

    ktype = pftp_ssh_buf_get_string(buf);
    if (!ktype) {
	pftp_ssh_buf_free(buf);
	return -1;	
    }

    if (strcmp("ssh-rsa", ktype) != 0) {
	free(ktype);
	pftp_ssh_buf_free(buf);
	return -1;
    }
    free(ktype);

    sigblob = (unsigned char *) pftp_ssh_buf_get_string2(buf, &len);
    pftp_ssh_buf_free(buf);
	
    if (!sigblob)
	return -1;

    if (pftp_ssh_buf_data_left(buf)) {
#ifdef DEBUG
	fprintf(stderr, 
		"ssh_rsa_verify: remaining bytes in signature.\n");
#endif
	free(sigblob);
	return -1;
    }
    /* RSA_verify expects a signature of RSA_size */
    modlen = RSA_size(key->rsa);
    if (len > modlen) {
#ifdef DEBUG
	fprintf(stderr, "ssh_rsa_verify: len %u > modlen %u.\n", 
		(unsigned int) len, (unsigned int) modlen);
#endif
	free(sigblob);
	return -1;
    } else if (len < modlen) {
	size_t diff = modlen - len;
	sigblob = realloc(sigblob, modlen);
	memmove(sigblob + diff, sigblob, len);
	memset(sigblob, 0, diff);
	len = modlen;
    }

    EVP_DigestInit(&md, evp_md);
    EVP_DigestUpdate(&md, data, datalen);
    EVP_DigestFinal(&md, digest, &dlen);
	
    ret = _openssh_RSA_verify(digest, dlen, sigblob, len, key->rsa);
    memset(digest, 'd', sizeof(digest));
    memset(sigblob, 's', len);
    free(sigblob);

    return ret;
}

/*
 * key_verify returns 1 for a correct signature, 0 for an incorrect signature
 * and -1 on error.
 */
static int _key_verify(const phostkey_t key, 
		       const char *signature, size_t sig_len,
		       const char *data, size_t datalen)
{
    if (sig_len == 0)
	return -1;

    switch (key->algo) {
    case KEY_SSH_DSS:
	return _ssh_dss_verify(key, signature, sig_len, data, datalen);
    case KEY_SSH_RSA:
	return _ssh_rsa_verify(key, signature, sig_len, data, datalen);
    default:
    case KEY_NONE:
	assert(0);
	return -1;
    }
}

int _mangle_keys(pftp_ssh_t ssh, const BIGNUM *dh_server_pub, 
		 const char *srv_host_key, size_t srv_key_len,
		 const char *signature, size_t sig_len)
{
    size_t klen, kout;
    unsigned char *kbuf;
    const unsigned char *hash;
    BIGNUM *shared_secret;

    klen = DH_size(ssh->my_dh);
    kbuf = malloc(klen);
    kout = DH_compute_key(kbuf, dh_server_pub, ssh->my_dh);
    shared_secret = BN_new();
    BN_bin2bn(kbuf, (int) kout, shared_secret);
    memset(kbuf, 0, klen);
    free(kbuf);

    hash = _kex_dh_hash(CLIENT_VERSION_STR, ssh->server_version_str,
			ssh->my_kexinit.data, ssh->my_kexinit.size,
			ssh->srv_kexinit.data, ssh->srv_kexinit.size,
			srv_host_key, srv_key_len, ssh->my_dh->pub_key,
			dh_server_pub, shared_secret);

    if (_key_verify(ssh->srv_key, signature, sig_len, 
		    (const char *) hash, 20) != 1) {
	pftp_status_message(ssh->ftp, "SSH: Invalid hostkey from server.");
	return -1;
    }

    if (ssh->session_id.data == NULL) {
	ssh->session_id.size = 20;
	ssh->session_id.data = malloc(ssh->session_id.size);
	memcpy(ssh->session_id.data, hash, ssh->session_id.size);
    }
	
    _kex_derive_keys(ssh, (const char *) hash, shared_secret);

    return 0;
}

int _safe_read(pftp_ssh_t ssh, void *data, size_t len, unsigned long timeout)
{
    size_t pos;
    ssize_t ret;
    struct timeval *to, _to;
    fd_set read_set;
    pos = 0;

    while (pos < len) {
	if (timeout > 0) {
	    to = &_to;
	    to->tv_sec = timeout / 1000;
	    to->tv_usec = (timeout % 1000) * 1000;
	} else {
	    to = NULL;
	}

	FD_ZERO(&read_set);
	FD_SET(ssh->ftp->cmdstream, &read_set);
	
	for (;;) {
	    ret = select(FD_SETSIZE, &read_set, NULL, NULL, to);
	    if (ret < 0) {
		if (errno == EINTR) {
		    continue;
		}

		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to select from socket: `%s'",
				    strerror(errno));	
		return -1;
	    } else if (ret == 0) {
		if (timeout > 0) {
		    /* Timeout */
		    pftp_status_message(ssh->ftp, 
					"ERR: Socket timeout during read.");
		    return -1;
		} else {
		    /* Should never happen? */
		    continue;
		}
	    } else {
		break;
	    }
	}

	ret = recv(ssh->ftp->cmdstream, ((char *) data) + pos, (int) (len - pos), 0);
	if (ret < 0) {
#ifdef WIN32
	    if (WSAGetLastError() == WSAEINTR || WSAGetLastError() == WSAEWOULDBLOCK) {
		continue;
	    } else {
		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to read from socket: `%lu'",
				    WSAGetLastError());
		return -1;				    
	    }
#else
	    if (errno == EAGAIN || errno == EINTR || errno == EWOULDBLOCK) {
		continue;
	    } else {
		pftp_status_message(ssh->ftp, 
				    "ERR: Unable to read from socket: `%s'",
				    strerror(errno));
		return -1;				    
	    }
#endif
	} if (ret == 0) {
	    /* Closed */
	    pftp_status_message(ssh->ftp, 
				"ERR: Unable to read from socket: "
				"`Socket closed'");
	    return -1;
	} else {
	    pos += ret;
	}
    }
    
    return 0;    
}

static const EVP_MD *_get_mac_algo(mac_algo_t mac)
{
    switch (mac) {
    case MAC_NONE:
	assert(0);
	return NULL;
    case MAC_SHA1:
	return EVP_sha1();	    
    case MAC_RIPEMD160:
	return EVP_ripemd160();
    case MAC_MD5:
	return EVP_md5();
    default:
	assert(0);
	return NULL;
    }
}

__inline size_t _mac_keylen(mac_algo_t mac)
{
    return EVP_MD_size(_get_mac_algo(mac));
}

size_t _calc_mac_len(pftp_ssh_t ssh, mac_algo_t mac_algo)
{
    if (mac_algo == MAC_NONE) {
	return 0;
    }

    return _mac_keylen(mac_algo);
}

void _calc_mac(pftp_ssh_t ssh, mac_algo_t mac_algo, 
	       raw_data_t key, uint32_t sequence_number, 
	       const unsigned char *packet, uint32_t packet_length, 
	       unsigned char **mac, size_t mac_len)
{
    uint32_t nbr;
    HMAC_CTX ctx;

    assert(*mac == NULL);

    if (mac_algo == MAC_NONE) {
	*mac = NULL;
	return;
    }

    HMAC_Init(&ctx, key.data, (int) mac_len, _get_mac_algo(mac_algo));

    nbr = htonl(sequence_number);

    HMAC_Update(&ctx, (const void *)&nbr, 4);
    HMAC_Update(&ctx, packet, packet_length);

    *mac = malloc(EVP_MAX_MD_SIZE);
    HMAC_Final(&ctx, *mac, NULL);
    *mac = realloc(*mac, mac_len);
    HMAC_cleanup(&ctx);
}

static __inline void _get_next_name(const char **start, const char **stop)
{
    const char *ret;
    (*stop)++;
    if ((ret = strchr(*stop, ','))) {
	*start = *stop;
	*stop = ret;
    } else {
	if ((*((*stop) - 1)) == '\0') {
	    *start = NULL;
	    *stop = NULL;
	} else {
	    *start = *stop;
	    *stop = *start + strlen(*start);
	}
    }
}

int _get_matching_namelist(const char *client_list, const char *server_list, 
			   char **algo)
{
    const char *client_s, *client_e, *server_s, *server_e;

    if (*algo) {
	free(*algo);
	*algo = NULL;
    }

    client_s = client_list;
    client_e = strchr(client_s, ',');
    if (!client_e)
	client_e = client_s + strlen(client_s);
    while (client_s) {
	server_s = server_list;
	server_e = strchr(server_s, ',');
	if (!server_e)
	    server_e = server_s + strlen(server_s);
	while (server_s) {
	    if (((client_e - client_s) == (server_e - server_s)) &&
		strncmp(client_s, server_s, client_e - client_s) == 0) {
		*algo = strndup(client_s, client_e - client_s);
		if (client_s == client_list && server_s == server_list)
		    return 1;
		return 0;
	    }

	    _get_next_name(&server_s, &server_e);
	}
	
	_get_next_name(&client_s, &client_e);
    }

    return 0;
}

int _generate_public_key(pftp_ssh_t ssh)
{
    size_t need_size = 0, _need;

    if (need_size < (_need = _enc_keylen(ssh->new_my_enc_algo)))
	need_size = _need;
    if (need_size < (_need = _enc_keylen(ssh->new_srv_enc_algo)))
	need_size = _need;

    if (need_size < (_need = _enc_blocklen(ssh->new_my_enc_algo)))
	need_size = _need;
    if (need_size < (_need = _enc_blocklen(ssh->new_srv_enc_algo)))
	need_size = _need;

    if (need_size < (_need = _mac_keylen(ssh->new_my_mac_algo)))
	need_size = _need;
    if (need_size < (_need = _mac_keylen(ssh->new_srv_mac_algo)))
	need_size = _need;

    ssh->need_size = need_size;

    assert(ssh->my_dh == NULL);

    switch (ssh->new_kex_algo) {
    case KEX_NONE:
	assert(0);
	return -1;
    case KEX_DIFF_HELL_GRP1_SHA1:
	if (!(ssh->my_dh = pftp_dh_new_group1()))
	    return -1;
	break;
    case KEX_DIFF_HELL_GRP14_SHA1:
	if (!(ssh->my_dh = pftp_dh_new_group14()))
	    return -1;
	break;
    }

    pftp_dh_gen_key(ssh->my_dh, (int) (need_size * 8));

    return 0;
}

void _handle_newkeys(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_NEWKEYS);
    pftp_ssh_pkt_reset(pkg);

    if (ssh->state != STATE_INIT) {
	pftp_status_message(ssh->ftp, "SSH: NewKeys outside INIT.");
	_kill_ssh(ssh);
	return;
    }

    if (ssh->init_state != INIT_NEWKEYS) {
	pftp_status_message(ssh->ftp, "SSH: NewKeys without keyexchange.");
	_kill_ssh(ssh);
	return;
    }

    ssh->my_mac_algo = ssh->new_my_mac_algo;
    ssh->srv_mac_algo = ssh->new_srv_mac_algo;
    ssh->kex_algo = ssh->new_kex_algo;
    ssh->key_algo = ssh->new_key_algo;
    ssh->my_comp_algo = ssh->new_my_comp_algo;
    ssh->srv_comp_algo = ssh->new_srv_comp_algo;
    ssh->my_enc_algo = ssh->new_my_enc_algo;
    ssh->srv_enc_algo = ssh->new_srv_enc_algo;

    _enc_data_init(&ssh->srv_enc, ssh->srv_enc_algo,
		   &ssh->srv_enc_key, &ssh->srv_enc_iv, 0);
    _enc_data_init(&ssh->my_enc, ssh->my_enc_algo,
		   &ssh->my_enc_key, &ssh->my_enc_iv, 1);

    if (ssh->srv_enc == NULL && ssh->srv_enc_algo != ENC_NONE) {
	pftp_status_message(ssh->ftp," SSH: Failed to init decryption.");
	_kill_ssh(ssh);
	return;
    }
    if (ssh->my_enc == NULL && ssh->my_enc_algo != ENC_NONE) {
	pftp_status_message(ssh->ftp," SSH: Failed to init encryption.");
	_kill_ssh(ssh);
	return;
    }

    /* Free data not needed after kex */

    if (ssh->my_dh) {
	DH_free(ssh->my_dh);
	ssh->my_dh = NULL;
    }
    if (ssh->my_key) {
	_free_hostkey(ssh->my_key);
	ssh->my_key = NULL;
    }
    if (ssh->srv_key) {
	_free_hostkey(ssh->srv_key);
	ssh->srv_key = NULL;
    }
    if (ssh->srv_kexinit.data) {
	free(ssh->srv_kexinit.data);
	ssh->srv_kexinit.data = NULL;
    }
    if (ssh->my_kexinit.data) {
	free(ssh->my_kexinit.data);
	ssh->my_kexinit.data = NULL;
    }

    ssh->state = STATE_CONNECTED;
}

void _handle_disconnect(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    int valid;
    uint32_t reason = 0;
    char *desc = NULL;
	
    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_DISCONNECT);
    pftp_ssh_pkt_reset(pkg);

    reason = pftp_ssh_pkt_get_uint32(pkg, &valid);

    if (valid) {
	desc = pftp_ssh_pkt_get_string(pkg);
	if (desc) {
	    pftp_status_message(ssh->ftp,
				"SSH: DISCONNECT `%s' (%lu).", 
				desc, reason);
	    free(desc);
	} else {
	    pftp_status_message(ssh->ftp,
				"SSH: DISCONNECT (reason %lu).", 
				reason);
	}
    } else {
	pftp_status_message(ssh->ftp,
			    "SSH: Invalid DISCONNECT message.");
    }	

    _kill_ssh(ssh);
}

pftp_ssh_pkt_t pftp_ssh_get(pftp_ssh_t ssh, unsigned long timeout)
{
    pftp_ssh_pkt_t pkg = NULL;

    for (;;) {
	pkg = _ssh_read(ssh, timeout);

	if (!pkg) 
	    return NULL;

	if (_handle_ssh_packet(ssh, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	    pkg = NULL;
	} else {
	    return pkg;
	}

	if (ssh->state == STATE_DEAD) {
	    return NULL;
	}
    }
}

int pftp_ssh_send(pftp_ssh_t ssh, const pftp_ssh_pkt_t pkg, 
		  unsigned long timeout)
{
    if (ssh->state == STATE_DEAD) {
	return -1;
    } else {
	return _ssh_write(ssh, pkg, timeout);
    }
}

int _handle_ssh_packet(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    int msg;

    msg = pftp_ssh_pkt_msg(pkg);
    if (msg == -1) {
	assert(0);
	return 1;
    }

    if (ssh->srv_sent_keyinit > 0 && (msg >= 30 && msg <= 49)) {
	ssh->srv_sent_keyinit--;
	return 1;
    }

    switch (msg) {
    case SSH_MSG_KEXINIT:
	_handle_kexinit(ssh, pkg);
	return 1;
    case SSH_MSG_NEWKEYS:
	_handle_newkeys(ssh, pkg);
	return 1;
    case SSH_MSG_DISCONNECT:
	_handle_disconnect(ssh, pkg);
	return 1;
    case SSH_MSG_KEXDH_REPLY:
	_handle_kexdh_reply(ssh, pkg);
	return 1;
    case SSH_MSG_SERVICE_ACCEPT:
	_handle_service_accept(ssh, pkg);
	return 1;
    case SSH_MSG_IGNORE:
	_handle_ignore(ssh, pkg);
	return 1;
    case SSH_MSG_DEBUG:
	_handle_debug(ssh, pkg);
	return 1;
    case SSH_MSG_UNIMPLEMENTED:
	_handle_unimplemented(ssh, pkg);
	return 1;
    default:
	return 0;
    }
}

void _kill_ssh(pftp_ssh_t ssh)
{
    _send_disconnect(ssh);

    pftp_status_message(ssh->ftp, "SSH: Connection killed.");
    ssh->state = STATE_DEAD;

    if (ssh->ftp->cmdstream > -1) {
	closesocket(ssh->ftp->cmdstream);
	ssh->ftp->cmdstream = -1;
    }
}

void _handle_ignore(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_IGNORE);
	
    /* Ignore data */
}

void _handle_debug(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    int valid;
    int always_display = 0;
    char *message = NULL, *language = NULL;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_DEBUG);
    pftp_ssh_pkt_reset(pkg);
	
    always_display = (unsigned char)pftp_ssh_pkt_get_char(pkg, &valid);
    if (!valid)
	goto debug_packet_to_small;
	
    message = pftp_ssh_pkt_get_string(pkg);
    if (!message)
	goto debug_packet_to_small;
	
    language = pftp_ssh_pkt_get_string(pkg);
    if (!language)
	goto debug_packet_to_small;

    if (always_display) {
	pftp_status_message(ssh->ftp, "SSH: DEBUG: %s.", message);
    } else {
#ifdef DEBUG
	pftp_status_message(ssh->ftp, "SSH: DEBUG: %s.", message);
#endif
    }
	
    free(message);
    free(language);
	
    return;

debug_packet_to_small:
    pftp_status_message(ssh->ftp, "SSH: Debug packet to small or invalid.");
    if (message) free(message);
    if (language) free(language);	
}

#if 0
int _send_unimplemented(pftp_ssh_t ssh, uint32_t sequence_number)
{
    pftp_ssh_pkt_t pkg;
    int ret;

    pkg = pftp_ssh_create_pkt(SSH_MSG_UNIMPLEMENTED);

    pftp_ssh_pkt_put_uint32(pkg, sequence_number);
    ret = _ssh_write(ssh, pkg, 0);
    pftp_ssh_pkt_free(pkg);
    return ret;
}
#endif

int _send_disconnect(pftp_ssh_t ssh)
{
    pftp_ssh_pkt_t pkg;
    int ret;

    pkg = pftp_ssh_create_pkt(SSH_MSG_DISCONNECT);

    ret = _ssh_write(ssh, pkg, 5000);
    pftp_ssh_pkt_free(pkg);
    return ret;
}

void _handle_unimplemented(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    int valid;
    uint32_t seqnum;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_UNIMPLEMENTED);
    pftp_ssh_pkt_reset(pkg);

    seqnum = pftp_ssh_pkt_get_uint32(pkg, &valid);

    if (!valid)
	goto unimp_packet_to_small;
   
    pftp_status_message(ssh->ftp,
			"SSH: Unimplemented response (to packet %lu).",
			seqnum);
	
    return;

unimp_packet_to_small:
    pftp_status_message(ssh->ftp,
			"SSH: UNIMPLEMENTED packet to small or invalid.");
    _kill_ssh(ssh);
}

void _handle_service_accept(pftp_ssh_t ssh, pftp_ssh_pkt_t pkg)
{
    char *service = NULL;

    assert(pftp_ssh_pkt_msg(pkg) == SSH_MSG_SERVICE_ACCEPT);
    pftp_ssh_pkt_reset(pkg);

    service = pftp_ssh_pkt_get_string(pkg);
    if (!service)
	goto serviceaccept_packet_to_small;

    assert(ssh->service_waiting);

    if (strcmp(service, ssh->service_waiting) == 0) {
	ssh->service_ok = 1;
    } else {
	pftp_status_message(ssh->ftp,
			    "SSH: Service Accept: `%s' ???.",
			    service);
    }
	
    free(service);

    return;
	
serviceaccept_packet_to_small:
    pftp_status_message(ssh->ftp,
			"SSH: SERVICE_ACCEPT packet to small or invalid.");
    if (service) free(service);
    _kill_ssh(ssh);
}

phostkey_t _hostkey_from_blob(const char *data, size_t len)
{
    char *tmpstr = NULL;
    phostkey_t ret;
    pftp_ssh_buf_t buf;

    buf = pftp_ssh_attach_buf(data, len);

    ret = malloc(sizeof(hostkey_t));
    memset(ret, 0, sizeof(hostkey_t));

    tmpstr = pftp_ssh_buf_get_string(buf);
    if (!tmpstr) {
#ifdef DEBUG
	fprintf(stderr, "hostkey_from_blob: can\'t read key type.\n");
#endif
	free(ret);
	pftp_ssh_buf_free(buf);
	return NULL;
    }
	
    ret->algo = _parse_key_algo(tmpstr);

    switch (ret->algo) {
    case KEY_NONE:
#ifdef DEBUG
	fprintf(stderr, "hostkey_from_blob: unknown type `%s'.\n",
		tmpstr);
#endif
	free(tmpstr);
	pftp_ssh_buf_free(buf);
	free(ret);
	return NULL;
    case KEY_SSH_RSA: 
	ret->rsa = RSA_new();
	ret->rsa->e = pftp_ssh_buf_get_bignum(buf);
	ret->rsa->n = pftp_ssh_buf_get_bignum(buf);
	if (ret->rsa->e == NULL || ret->rsa->n == NULL) {
#ifdef DEBUG
	    fprintf(stderr, "hostkey_from_blob: can't read rsa key");
#endif
	    _free_hostkey(ret);
	    pftp_ssh_buf_free(buf);
	    return NULL;
	}
	break;
    case KEY_SSH_DSS:
	ret->dsa = DSA_new();
	ret->dsa->p = pftp_ssh_buf_get_bignum(buf);
	ret->dsa->q = pftp_ssh_buf_get_bignum(buf);
	ret->dsa->g = pftp_ssh_buf_get_bignum(buf);
	ret->dsa->pub_key = pftp_ssh_buf_get_bignum(buf);
	if (ret->dsa->p == NULL || ret->dsa->q == NULL || 
	    ret->dsa->g == NULL || ret->dsa->pub_key == NULL) {
#ifdef DEBUG
	    fprintf(stderr, "hostkey_from_blob: can't read dsa key.\n");
#endif
	    _free_hostkey(ret);
	    pftp_ssh_buf_free(buf);
	    return NULL;
	}
	break;
    }
	
    if (pftp_ssh_buf_data_left(buf)) {
#ifdef DEBUG
	fprintf(stderr, "hostkey_from_blob: remaining bytes in blob %lu.\n",
		(unsigned long) pftp_ssh_buf_data_left(buf));
#endif
	_free_hostkey(ret);
	pftp_ssh_buf_free(buf);
	return NULL;
    }

    pftp_ssh_buf_free(buf);
		
    return ret;
}

void _free_hostkey(phostkey_t key)
{
    if (key) {
	if (key->rsa) {
	    RSA_free(key->rsa);
	}

	if (key->dsa) {
	    DSA_free(key->dsa);
	}

	free(key);
    }
}

int _send_service_request(pftp_ssh_t ssh, const char *service)
{
    pftp_ssh_pkt_t pkg;
    int ret;
    pkg = pftp_ssh_create_pkt(SSH_MSG_SERVICE_REQUEST);

    assert(ssh->service_ok == 0);
    ssh->service_waiting = strdup(service);

    pftp_ssh_pkt_put_string(pkg, service);
    ret = _ssh_write(ssh, pkg, 0);
    pftp_ssh_pkt_free(pkg);
    return ret;	
}

int pftp_ssh_request_service(pftp_ssh_t ssh, const char *service)
{
    if (_send_service_request(ssh, service))
	return -1;

    for (;;) {
	pftp_ssh_pkt_t pkg;
		
	if (!(pkg = _ssh_read(ssh, 0))) {
	    _kill_ssh(ssh);
	    return -1;
	}

	if (_handle_ssh_packet(ssh, pkg)) {
	    pftp_ssh_pkt_free(pkg);
	} else {
	    pftp_ssh_pkt_free(pkg);
	    pftp_status_message(ssh->ftp, 
				"SSH: Unhandled message while waiting for "
				"service accept.");
	    _kill_ssh(ssh);
	    return -1;
	}

	if (ssh->service_ok) {			
	    break;
	}
    }

    free(ssh->service_waiting);
    ssh->service_waiting = NULL;
    ssh->service_ok = 0;

    return 0;
}

int _send_newkeys(pftp_ssh_t ssh)
{
    pftp_ssh_pkt_t pkg;
    int ret;
    pkg = pftp_ssh_create_pkt(SSH_MSG_NEWKEYS);

    ret = _ssh_write(ssh, pkg, 0);
    pftp_ssh_pkt_free(pkg);
    return ret;		
}

static __inline const EVP_CIPHER *_get_enc_algo(enc_algo_t enc)
{
    switch (enc) {
    case ENC_NONE:
	assert(0);
	return NULL;
    case ENC_3DES:
	return EVP_des_ede3_cbc();
    case ENC_BLOWFISH:
	return EVP_bf_cbc();
    case ENC_AES128:
	return EVP_aes_128_cbc();
    default:
	assert(0);
	return NULL;
    }
}

__inline size_t _enc_blocklen(enc_algo_t enc)
{
    switch (enc) {
    case ENC_NONE:
	return 8;
    case ENC_3DES:
	return 8;
    case ENC_AES128:
	return 16;
    case ENC_BLOWFISH:
	return 8;
    default:
	assert(0);
	return 0;
    }
}

__inline size_t _enc_keylen(enc_algo_t enc)
{
    switch (enc) {
    case ENC_NONE:
	return 0;
    case ENC_3DES:
	return 24;
    case ENC_AES128:
	return 16;
    case ENC_BLOWFISH:
	return 16;
    default:
	assert(0);
	return 0;
    }
}

void _enc_data_init(penc_data_t *enc, enc_algo_t enc_algo,
		    const raw_data_t *key, const raw_data_t *iv,
		    int do_encrypt)
{
    size_t klen;

    if (*enc) {
	_free_enc_data(*enc);
	*enc = NULL;
    }

    if (enc_algo == ENC_NONE) {
	return;
    }

    *enc = malloc(sizeof(enc_data_t));
    memset(*enc, 0, sizeof(enc_data_t));

    assert(key->size >= _enc_keylen(enc_algo));
    assert(iv->size >= _enc_blocklen(enc_algo));
	
    EVP_CIPHER_CTX_init(&(*enc)->evp);
    if (EVP_CipherInit(&(*enc)->evp, _get_enc_algo(enc_algo), NULL, 
		       iv->data, do_encrypt) == 0) {
#ifdef DEBUG
	fprintf(stderr, "cipher_init: EVP_CipherInit failed.\n");
#endif
	assert(0);
	_free_enc_data(*enc);
	*enc = NULL;
	return;
    }
    klen = EVP_CIPHER_CTX_key_length(&(*enc)->evp);
    if (klen > 0 && key->size != klen) {
	if (EVP_CIPHER_CTX_set_key_length(&(*enc)->evp, (int) key->size) == 0) {
	    /* Does seam to work perfectly without this returning OK */
#ifdef DEBUG
	    /*
	      fprintf(stderr, "cipher_init: set keylen failed (%d -> %lu).\n",
	      klen, (unsigned long)key->size);
	    */
#endif
/*			assert(0);
			_free_enc_data(*enc);
			*enc = NULL;
			return;
*/
	}
    }
    if (EVP_CipherInit(&(*enc)->evp, NULL, key->data, NULL, -1) == 0) {
#ifdef DEBUG
	fprintf(stderr, "cipher_init: EVP_CipherInit: set key failed.\n");
#endif
	assert(0);
	_free_enc_data(*enc);
	*enc = NULL;
	return;
    }
}

void _free_enc_data(penc_data_t enc)
{
    if (enc) {
	EVP_CIPHER_CTX_cleanup(&enc->evp);

	free(enc);
    }
}

void _crypt_data(penc_data_t enc, enc_algo_t algo, unsigned char *data, 
		 size_t size)
{
    unsigned char *dest;

    if (algo == ENC_NONE) {
	return;
    }

    assert((size % _enc_blocklen(algo)) == 0);

    dest = malloc(size);
    EVP_Cipher(&enc->evp, dest, data, (unsigned int) size);
    memcpy(data, dest, size);
    free(dest);
}

const char *pftp_ssh_get_server_str(pftp_ssh_t ssh)
{
    return ssh->server;
}

#endif /* NO_SSL */
