/* -*- pftp-c -*- */
#ifndef PFTP_H
#define PFTP_H

typedef struct pftp_server_s *pftp_server_t;

#include "pftp_settings.h"
#include "pftp_directory.h"
#include "pftp_comm.h"

/* Init lib, must be run before any other calls */
void pftp_initlib(void);

/* Cleans up after lib */
void pftp_donelib(void);

/* Login, make the connection to the ftp. Takes a settings structure pointer
   that must exist until logout is called. 
   Returns NULL if error. Any error messages are sent to status */
pftp_server_t pftp_login(pftp_settings_t settings, 
			 pftp_comm_centralFunc_t comm_central);

/* Logout, disconnect from a ftp and free memory of pointer */
void pftp_logout(pftp_server_t *ftp);

/* Change current directory, 
   If newdir is not null:
     The point newdir points at must be NULL.
     If cd knows (by server telling it) by the response which directory
     where in, newdir is set to that.
     If cd doesn't know, newdir is left alone.
   returns 0 on success, -1 on error */
int pftp_cd(pftp_server_t ftp, const char *dir, char **newdir);

/* Remove directory, returns 0 on success, -1 on error */
int pftp_rmdir(pftp_server_t ftp, const char *dir);

/* Create directory, returns 0 on success, -1 on error */
int pftp_mkdir(pftp_server_t ftp, const char *dir);

/* Get current directory, The pointer currentDirectory must point to NULL and
   will be left to NULL if any errors accur. Otherwise it will be set to 
   current directory. User must free the string themself. 
   force - if set, cache will not be used */
void pftp_curdir(pftp_server_t ftp, char **currentDirectory, int force);

/* List current directory. Will fill the structure dir with info. 
   If force is set, cache will not be used.
   If cache is not null, and cache is used it's set to 1 otherwise 0.
   Returns 0 on no errors, -1 on errors (and then dir is not touched) */
int pftp_ls(pftp_server_t ftp, pftp_directory_t *dir, int force, int *cached);

/* Download a file from current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then download is in binary mode otherwise ascii mode.
   fh - must point to a writeable stream.
   resume_from - Tells from where we should start reading from ftp 
   (if supported by server). If it's supported, get will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.
   crc - If not NULL, after file download this will be the crc32 code of 
   downloaded data. (So, dont rely on this when using resume).

   Returns 0 on OK and -1 on error.
 */
int pftp_get(pftp_server_t ftp, const char *filename, char ascii, FILE *fh, 
	     uint64_t resume_from, uint64_t total_size, uint32_t *crc);

/* Upload a file to the current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then upload is in binary mode otherwise ascii mode.
   fh - must point to a readable stream.
   resume_from - Tells from where we should start writing to the file on the
   ftp (if supported by server). If it's supported, put will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.

   Returns 0 on OK and -1 on error.
 */
int pftp_put(pftp_server_t ftp, const char *filename, char ascii, FILE *fh,
	     uint64_t resume_from, uint64_t total_size);

/* Remove a file in current directory 
   Return 0 on OK and -1 on error. */
int pftp_rm(pftp_server_t ftp, const char *filename);

/* Rename a file from filename to new in current directory
   Return 0 on OK and -1 on error. */
int pftp_rename(pftp_server_t ftp, const char *filename, const char *new);

/* Send a RAW cmd to server.
   FTP server response will be put in data (that must point on NULL).
   Returns -1 on error and the return code if OK (100 -> 600). */
int pftp_rawcmd(pftp_server_t ftp, const char *cmd, char **data);

/* Transfer file from FTP server A to FTP server B. 
   ascii - If 'B' binary mode is used, otherwise ascii.
   Returns -1 on error and 0 on OK. */
int pftp_fxp(pftp_server_t srcftp, const char *srcfile, pftp_server_t destftp, 
	     const char *destfile, char ascii, uint64_t resume_from,
	     uint64_t append_to, uint64_t total_size);

/* Clear directory cache for directory.
   If directory is NULL, then whole cache is cleared. */
void pftp_clear_dir_cache(pftp_server_t ftp, const char *directory);

/* A little helper function. "Removes" . and .. from paths if it can.
   Returns 0 when error. 
   May reallocate path, but only if OK. */
int pftp_compactPath(char **path);

#endif /* PFTP_H */
