
/*
 * Copyright (c) 2001-2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@freebsd.org>
 */

#include "tmpl_internal.h"

/* Internal functions */
static struct	tmpl *tmpl_create_internal(FILE *input, void *addr,
			size_t len, int *num_errors, const char *mtype);

/*
 * Create a new template object.
 */
struct tmpl *
tmpl_create(FILE *input, int *num_errors, const char *mtype)
{
	return (tmpl_create_internal(input, NULL, 0, num_errors, mtype));
}

/*
 * Create a new template object using mmap().
 */
struct tmpl *
tmpl_create_mmap(const char *path, int *num_errors, const char *mtype)
{
	struct tmpl *tmpl;
	struct stat sb;
	void *addr;
	int esave;
	FILE *fp;
	int fd;

	/* Open and memory map file */
	if ((fd = open(path, O_RDONLY)) == -1)
		return (NULL);
	if (fstat(fd, &sb) == -1) {
		close(fd);
		return (NULL);
	}
	if ((addr = mmap(NULL, sb.st_size,
	    PROT_READ, MAP_SHARED, fd, 0)) == NULL) {
		close(fd);
		return (NULL);
	}
	close(fd);

	/* Parse file */
	if ((fp = fopen(path, "r")) == NULL)
		return (NULL);
	if ((tmpl = tmpl_create_internal(fp, addr,
	    sb.st_size, num_errors, mtype)) == NULL) {
		esave = errno;
		fclose(fp);
		errno = esave;
		return (NULL);
	}

	/* Done */
	return (tmpl);
}

/*
 * Create a new template object.
 */
static struct tmpl *
tmpl_create_internal(FILE *input, void *addr, size_t len,
	int *num_errors, const char *mtype)
{
	struct tmpl *tmpl;
	int dummy;
	int r;

	/* Sanity check */
	if (input == NULL) {
		errno = EINVAL;
		return (NULL);
	}
	if (num_errors == NULL)
		num_errors = &dummy;

	/* Initialize template object */
	if ((tmpl = MALLOC(mtype, sizeof(*tmpl))) == NULL)
		return (NULL);
	memset(tmpl, 0, sizeof(*tmpl));
	tmpl->mmap_addr = addr;
	tmpl->mmap_len = len;
	if (mtype != NULL) {
		strlcpy(tmpl->mtype_buf, mtype, sizeof(tmpl->mtype_buf));
		tmpl->mtype = tmpl->mtype_buf;
	}

	/* Parse template; cleanup if thread is canceled */
	pthread_cleanup_push((void (*)(void *))tmpl_destroy, &tmpl);
	r = _tmpl_parse(tmpl, input, num_errors);
	pthread_cleanup_pop(0);

	/* Check for error */
	if (r == -1) {
		tmpl_destroy(&tmpl);
		return (NULL);
	}

	/* Done */
	return (tmpl);
}

/*
 * Destroy a template object.
 */
void
tmpl_destroy(struct tmpl **tmplp)
{
	struct tmpl *const tmpl = *tmplp;

	if (tmpl == NULL)
		return;
	*tmplp = NULL;
	_tmpl_free_elems(tmpl->mtype,
	    tmpl->eblock, tmpl->elems, tmpl->num_elems);
	if (tmpl->mmap_addr != NULL)
		munmap(tmpl->mmap_addr, tmpl->mmap_len);
	FREE(tmpl->mtype, tmpl);
}

