
/*
 * Copyright (c) 2001-2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@freebsd.org>
 */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>
#include <errno.h>

#include "structs/structs.h"
#include "structs/type/array.h"
#include "structs/type/id.h"
#include "util/typed_mem.h"

/*********************************************************************
			IDENTIFIER TYPES
*********************************************************************/

int
structs_id_init(const struct structs_type *type, void *data)
{
	const struct structs_id *const ids = type->args[0].v;

	switch (type->size) {
	case 1:
	    {
		const u_int8_t value = (u_int8_t)ids[0].value;

		memcpy(data, &value, type->size);
		break;
	    }
	case 2:
	    {
		const u_int16_t value = (u_int16_t)ids[0].value;

		memcpy(data, &value, type->size);
		break;
	    }
	case 4:
	    {
		const u_int32_t value = (u_int32_t)ids[0].value;

		memcpy(data, &value, type->size);
		break;
	    }
	default:
		assert(0);
	}
	return (0);
}

char *
structs_id_ascify(const struct structs_type *type,
	const char *mtype, const void *data)
{
	const struct structs_id *id;
	u_int32_t value = 0;

	switch (type->size) {
	case 1:
		value = *((u_int8_t *)data);
		break;
	case 2:
		value = *((u_int16_t *)data);
		break;
	case 4:
		value = *((u_int32_t *)data);
		break;
	default:
		assert(0);
	}
	for (id = type->args[0].v; id->id != NULL; id++) {
		if (value == id->value)
			return (STRDUP(mtype, id->id));
	}
	return (STRDUP(mtype, "INVALID"));
}

int
structs_id_binify(const struct structs_type *type,
	const char *ascii, void *data, char *ebuf, size_t emax)
{
	const struct structs_id *id;

	for (id = type->args[0].v; id->id != NULL; id++) {
		int (*const cmp)(const char *, const char *)
		    = id->imatch ? strcasecmp : strcmp;

		if ((*cmp)(ascii, id->id) == 0) {
			switch (type->size) {
			case 1:
				*((u_int8_t *)data) = id->value;
				break;
			case 2:
				*((u_int16_t *)data) = id->value;
				break;
			case 4:
				*((u_int32_t *)data) = id->value;
				break;
			default:
				assert(0);
			}
			return (0);
		}
	}
	snprintf(ebuf, emax, "invalid value \"%s\"", ascii);
	errno = EINVAL;
	return (-1);
}

