
/*
 * Copyright (c) 2001-2002 Packet Design, LLC.
 * All rights reserved.
 * 
 * Subject to the following obligations and disclaimer of warranty,
 * use and redistribution of this software, in source or object code
 * forms, with or without modifications are expressly permitted by
 * Packet Design; provided, however, that:
 * 
 *    (i)  Any and all reproductions of the source or object code
 *         must include the copyright notice above and the following
 *         disclaimer of warranties; and
 *    (ii) No rights are granted, in any manner or form, to use
 *         Packet Design trademarks, including the mark "PACKET DESIGN"
 *         on advertising, endorsements, or otherwise except as such
 *         appears in the above copyright notice or in the software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED BY PACKET DESIGN "AS IS", AND
 * TO THE MAXIMUM EXTENT PERMITTED BY LAW, PACKET DESIGN MAKES NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, REGARDING
 * THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY AND ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT.  PACKET DESIGN DOES NOT WARRANT, GUARANTEE,
 * OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF, OR THE RESULTS
 * OF THE USE OF THIS SOFTWARE IN TERMS OF ITS CORRECTNESS, ACCURACY,
 * RELIABILITY OR OTHERWISE.  IN NO EVENT SHALL PACKET DESIGN BE
 * LIABLE FOR ANY DAMAGES RESULTING FROM OR ARISING OUT OF ANY USE
 * OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE, OR CONSEQUENTIAL
 * DAMAGES, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, LOSS OF
 * USE, DATA OR PROFITS, HOWEVER CAUSED AND UNDER ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF PACKET DESIGN IS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Author: Archie Cobbs <archie@freebsd.org>
 */

#include <sys/types.h>

#include <netinet/in.h>

#include <stdio.h>
#include <assert.h>
#include <syslog.h>
#include <stdarg.h>

#include "structs/structs.h"
#include "structs/type/array.h"
#include "util/typed_mem.h"
#include "config/app_config.h"
#include "sys/alog.h"

/************************************************************************
			ALOG LOGGING SUBSYSTEM
************************************************************************/

static app_ss_startup_t		app_alog_start;
static app_ss_shutdown_t	app_alog_shutdown;
static app_ss_willrun_t		app_alog_willrun;
static app_ss_changed_t		app_alog_changed;

const struct app_subsystem	app_config_alog_subsystem = {
	"alog",
	NULL,
	app_alog_start,
	app_alog_shutdown,
	app_alog_willrun,
	app_alog_changed,
	NULL
};

/*
 * Alog startup
 */
static int
app_alog_start(struct app_config_ctx *ctx,
	const struct app_subsystem *ss, const void *config)
{
	const struct app_config_alog_info *const info = ss->arg;
	struct alog_config aconf;

	/* Get alog config */
	if (structs_get(app_config_get_type(ctx),
	    info->name, config, &aconf) == -1) {
		alogf(LOG_ERR, "%s: %m", "structs_get");
		return (-1);
	}

	/* Start up alog channel */
	if (alog_configure(info->channel, &aconf) == -1) {
		alog(LOG_ERR, "error configuring logging: %m");
		structs_free(&alog_config_type, NULL, &aconf);
		return (-1);
	}
	structs_free(&alog_config_type, NULL, &aconf);
	return (0);
}

/*
 * Alog shutdown
 */
static void
app_alog_shutdown(struct app_config_ctx *ctx,
	const struct app_subsystem *ss, const void *config)
{
	const struct app_config_alog_info *const info = ss->arg;

	alog_shutdown(info->channel);
}

/*
 * Alog necessity check
 */
static int
app_alog_willrun(struct app_config_ctx *ctx,
	const struct app_subsystem *ss, const void *config)
{
	const struct app_config_alog_info *const info = ss->arg;

	(void)info;				/* avoid compiler warning */
	assert(info->name != NULL);
	return (1);
}

/*
 * Alog configuration changed check
 */
static int
app_alog_changed(struct app_config_ctx *ctx,
	const struct app_subsystem *ss,
	const void *config1, const void *config2)
{
	const struct app_config_alog_info *const info = ss->arg;
	int ret;

	/* Compare configs */
	if ((ret = structs_equal(app_config_get_type(ctx),
	    info->name, config1, config2)) == -1) {
		alogf(LOG_ERR, "%s: %m", "structs_equal");
		return (-1);
	}
	return (!ret);
}

