/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <dirent.h>

#include <libnd_misc.h>
#include <libnd_prefs.h>

struct lnd_plugin
{
  const char *(*name) (void);
  const char *(*author) (void);
  const char *(*version) (void);

  /* Plugin initialization hook. Must return TRUE on success, FALSE otherwise. */
  gboolean    (*init) (void);

  /* If the plugin provides functionality applicable to a trace, you can
   * implement its run() hook. The implementation needs to return an error
   * code of your choosing. When using lndtool to invoke the plugin, this
   * error code will be returned to the shell.
   */
  int         (*run) (LND_Trace *trace, void *user_data);

  char        *filename;
  lt_dlhandle  lt;
  gboolean     initialized;
};

static GList *plugins = NULL;  /* list<LND_Plugin*> */

static LND_Plugin  *plugin_new(char *filename);
static void         plugin_scan(const char *dirname);
static gboolean     plugin_hook_up(LND_Plugin *p);
static void         plugin_load(void);
static int          plugin_cmp(LND_Plugin *p1, char *filename);

static const char  *plugin_dummy_name(void) { return "No name specified."; }
static const char  *plugin_dummy_author(void) { return "No author(s) specified."; }
static const char  *plugin_dummy_version(void) { return "No version specified."; }
static gboolean     plugin_dummy_init(void) { return TRUE; }

static int          plugin_dummy_run(LND_Trace *trace, void *user_data)
{
  return 0;
  TOUCH(trace);
  TOUCH(user_data);
}


static LND_Plugin *
plugin_new(char *filename)
{
  LND_Plugin *plugin;

  plugin = (LND_Plugin*) g_new0(LND_Plugin, 1);
  D_ASSERT_PTR(plugin);
  if (!plugin)
    return NULL;

  if (filename)
    plugin->filename = g_strdup(filename);

  /* We hook in dummy callbacks for segfault protection */
  plugin->name        = plugin_dummy_name;
  plugin->author      = plugin_dummy_author;
  plugin->version     = plugin_dummy_version;
  plugin->init        = plugin_dummy_init;
  plugin->run         = plugin_dummy_run;
  
  return plugin;
}


static void
plugin_free(LND_Plugin *plugin)
{
  D_ENTER;

  if (!plugin)
    D_RETURN;
  
  lt_dlclose(plugin->lt);
  g_free(plugin->filename);
  g_free(plugin);

  D_RETURN;
}


static int    
plugin_cmp(LND_Plugin *p1, char *filename)
{
  if (!p1 || !p1->filename || !filename)
    return (-1);
  
  return strcmp(p1->filename, filename);
}


static void
plugin_scan(const char *dir)
{
  DIR             *dirp;
  struct dirent   *de;
  char            *dot;
  struct stat      st;
  char             path[MAXPATHLEN];

  D(("Scanning '%s' for plugins\n", dir));

  if (! (dirp = opendir(dir)))
    return;

  for (de = readdir(dirp); de; de = readdir(dirp))
    {
      g_snprintf(path, MAXPATHLEN, "%s/%s", dir, de->d_name);

      if (stat(path, &st) != 0)
	{
	  D(("Could not stat %s\n", path));
	  continue;
	}

      if (! S_ISREG(st.st_mode))
	{
	  D(("Skipping %s, not a regular file.\n", path));
	  continue;
	}

      /* Chop off any file extension, if found. libltdl takes
       * care of that.
       */
      if ( (dot = strrchr(path, '.')))
	*dot = '\0';

      if (!g_list_find_custom(plugins, path, (GCompareFunc)plugin_cmp))
	plugins = g_list_append(plugins, plugin_new(path));
    }
  
  closedir(dirp);
}


static gboolean
plugin_hook_up(LND_Plugin *plugin)
{
  lt_ptr sym;

  if ((sym = lt_dlsym(plugin->lt, "name")) != NULL)
    plugin->name = sym;
  if ((sym = lt_dlsym(plugin->lt, "author")) != NULL)
    plugin->author = sym;
  if ((sym = lt_dlsym(plugin->lt, "version")) != NULL)
    plugin->version = sym;

  if ((sym = lt_dlsym(plugin->lt, "init")) != NULL)
    plugin->init = sym;
  if ((sym = lt_dlsym(plugin->lt, "run")) != NULL)
    plugin->run = sym;

  D(("Plugin %s's hooks:\n  name %p/%p,\n  author %p/%p,\n  version %p/%p,\n  init %p/%p,\n  run %p/%p.\n",
     plugin->filename,
     plugin->name, plugin_dummy_name,
     plugin->author, plugin_dummy_author,
     plugin->version, plugin_dummy_version,
     plugin->init, plugin_dummy_init,
     plugin->run, plugin_dummy_run));
     
  return plugin->init();
}


static gint
plugin_strcmp(gconstpointer a, gconstpointer b)
{
  LND_Plugin *p1 = (LND_Plugin *) a;
  LND_Plugin *p2 = (LND_Plugin *) b;

  return strcmp(p1->name(), p2->name());
}

static void       
plugin_load(void)
{
  GList        *l, *plugins_new = NULL;
  LND_Plugin   *plugin;
  int           i, num_passes;

  num_passes = g_list_length(plugins);

  /* We have a list of filenames of plugins and now try to dlopen and initialize them.
   * Individual plugins may fail for several reasons when they depend on other plugins
   * that haven't been loaded yet, for example due to missing symbols or because
   * registry checks in the initialization procedure fail. For this reason, we make
   * up to as many passes as there are plugins, but bail as soon as all plugins have
   * registered successfully.
   */
  
  for (i = 0; i < num_passes; i++)
    {
      int failures = 0;
      D(("%i. pass.\n", i));

      /* Make a pass over the plugins to get the dl
       * handle, and try to initialize. Initialization may
       * fail for some plugins if they require other plugins
       * that have not been initialized yet.
       */
      for (l = g_list_first(plugins); l; l = g_list_next(l))
	{
	  plugin = (LND_Plugin*)l->data;
	  
	  D(("Trying %s\n", plugin->filename));
	  
	  if (! plugin->lt)
	    {
	      plugin->lt = libnd_misc_dlopen(plugin->filename);	      
	      if (! plugin->lt)
		{
		  D(("dlopen failed on %s: %s\n", plugin->filename, lt_dlerror()));
		  failures++;
		  continue;
		}
	    }
	  
	  if (! plugin->initialized)
	    {
	      if ( (plugin->initialized = plugin_hook_up(plugin)))
		{
		  D(("Plugin %s hooked up.\n", plugin->name()));
		}
	      else
		{
		  D(("Plugin %s not yet hooked up.\n", plugin->name()));
		  failures++;
		}
	    }
	}

      if (failures == 0)
	break;

      D(("%i failures in %i. pass.\n", failures, i));
    }
  
  /* We may have deleted some plugins above that failed to initialize.
   * Clear out those list items in a separate iteration by creating
   * a new list and replacing the old one with it.
   */
  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      plugin = (LND_Plugin*)l->data;

      if (plugin && plugin->initialized)
	plugins_new = g_list_append(plugins_new, l->data);
      else
	{
	  plugin_free(plugin);
	  l->data = NULL;
	}
    }

  g_list_free(plugins);
  plugins = plugins_new;
  
  /* Now sort the list of plugins alphanumerically,
   * so that we automatically get sorted menus etc.
   */
  plugins = g_list_sort(plugins, plugin_strcmp);
}


static void
plugin_unload(void)
{
  GList     *l;
  LND_Plugin *pl;

  if (!plugins)
    return;

  for (l = plugins; l; l = g_list_next(l))
    {
      pl = (LND_Plugin *) l->data;

      plugin_free(pl);
      l->data = NULL;
    }

  g_list_free(plugins);
  plugins = NULL;
}


void 
libnd_plugin_init(void)
{
  static gboolean deja_vu = FALSE;

  D_ENTER;

  if (deja_vu)
    plugin_unload();

  if (lt_dlinit() != 0)
    {
      fprintf(stderr, "Could not initialize libltdl -- exiting.\n");
      exit(-1);
    }

  /* We first scan the directories where plugins may be installed,
   * and create a list of all plugins found.
   */
  plugin_scan(libnd_prefs_get_plugin_dir_user());
  plugin_scan(libnd_prefs_get_plugin_dir_global());

  /* Once we have that list, we actually open the DLLs and extract
   * symbols, check dependencies etc.
   */
  plugin_load();

  deja_vu = TRUE;
  D_RETURN;
}


LND_Plugin   *
libnd_plugin_find(const char *name)
{
  GList       *l;
  LND_Plugin   *plugin;

  D_ENTER;

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      plugin = (LND_Plugin*)l->data;

      if (! plugin)
	continue;

      if (! plugin->initialized)
	continue;

      if (! g_strcasecmp(plugin->name(), name))
	D_RETURN_(plugin);
    }

  D(("No plugin with name '%s' found.\n", name));
  D_RETURN_(NULL);
}


int
libnd_plugin_run(LND_Plugin *plugin,
	      LND_Trace *trace,
	      void *user_data)
{
  return plugin->run(trace, user_data);
}



const char *
libnd_plugin_get_name(LND_Plugin *plugin)
{
  if (!plugin)
    return (NULL);
  
  return plugin->name();
}


const char   *
libnd_plugin_get_author(LND_Plugin *plugin)
{
  if (!plugin)
    return (NULL);

  return plugin->author();
}


const char   *
libnd_plugin_get_version(LND_Plugin *plugin)
{
  if (!plugin)
    return (NULL);

  return plugin->version();
}


void           
libnd_plugin_foreach(LND_PluginFunc callback,
		     void *user_data)
{
  GList *l;

  if (!callback)
    return;
  
  for (l = g_list_first(plugins); l; l = g_list_next(l))
    callback((LND_Plugin *)l->data, user_data);
}

