/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_dumper_h
#define __libnd_dumper_h

#include <pcapnav.h>
#include <libnd_types.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* LND_Dumpers are an abstraction of pcap output traces. They allow
 * transparent appending to existing traces in a safe manner, and
 * automatic roll-over to new files once a configurable per-file
 * size is hit, and also allows specifying an upper bound on how
 * much data to capture. Once a file size limit is hit, a new output
 * trace is created, with numerical extensions of the form ".%05i".
 */

#define LND_DUMPER_500MB  500000000
#define LND_DUMPER_1GB   1000000000
#define LND_DUMPER_2GB   2000000000

typedef struct lnd_dumper LND_Dumper;

typedef struct lnd_dumper_args {
  pcapnav_dumpmode_t      open_mode;
  guint64                 file_limit;
  guint64                 hard_limit;
} LND_DumperArgs;

/**
 * libnd_dumper_args_init - initializes dumper specification structure.
 * @args: the argument structure to initialize.
 *
 * The function initializes the dumper spec structure
 * to default values. Currently, these are all zeroes.
 */
void        libnd_dumper_args_init(LND_DumperArgs *args);

/**
 * libnd_dumper_new - creates a new dumper.
 * @pcap: the pcap handle for the new dumper, as used in pcap_dump_open().
 * @dumper_name: the file name of the dumper.
 * @args: the dumper specification.
 *
 * The function returns a new dumper, or %NULL on error. @args can
 * be %NULL, in which case a per-file size limit of 1GB is enforced.
 */
LND_Dumper *libnd_dumper_new(pcap_t *pcap, const char *dumper_name,
			     LND_DumperArgs *args);

/**
 * libnd_dumper_free - closes dumper.
 * @dumper: dumper to close.
 *
 * The function closes the current output file, if any,
 * and releases all memory resources of the dumper.
 */
void        libnd_dumper_free(LND_Dumper *dumper);

gboolean    libnd_dumper_write(LND_Dumper *dumper, LND_Packet *packet);

/**
 * libnd_dumper_delete - deletes all output files belonging to a dumper.
 * @dumper: the dumper whose files to delete.
 *
 * The function looks for all files the dumper might have
 * created based on its name and deletes them, until one
 * is found that does not exist. So if the dumper name is
 * "foo", it potentially deletes files foo, foo.00001,
 * foo.00002 etc, until a nonexistant file is hit.
 */
void        libnd_dumper_delete(LND_Dumper *dumper);

/**
 * libnd_dumper_get_size - returns the total size of dumper.
 * @dumper: dumper to return size of.
 *
 * The function returns the total size, in bytes, of all
 * packets ever written to this dumper (counting across
 * multiple created files).
 */
guint64     libnd_dumper_get_size(LND_Dumper *dumper);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
     
