/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd.h>
#include <libnd_bpf.h>

typedef struct lnd_bpf_filter
{
  char               *bpf_expr;
  struct bpf_program  bpf_program;

} LND_BPF_Filter;


static LND_BPF_Filter *
bpf_filter_new(const char *bpf_expr)
{
  LND_BPF_Filter *filter;

  if (!bpf_expr)
    return NULL;

  if (! (filter = g_new0(LND_BPF_Filter, 1)))
    return NULL;
  
  filter->bpf_expr = g_strdup(bpf_expr);
  return filter;
}

static void
bpf_filter_free(LND_BPF_Filter *data)
{
  D_ENTER;

  if (!data)
    D_RETURN;

  g_free(data->bpf_expr);
  g_free(data);

  D_RETURN;
}



static gboolean
bpf_filter_init(LND_Filter *filter,
		LND_Trace *trace,
		LND_BPF_Filter *filter_data)
{
  D_ENTER;

  /* Try to compile the filter program for this trace: */
  if (pcap_compile(pcapnav_pcap(trace->tpm->base->pcn), &filter_data->bpf_program,
		   (char *) filter_data->bpf_expr, TRUE, 0xFFFFFFFF))
    {
      D(("The BPF filter expression '%s' could not be compiled.\n", filter_data->bpf_expr));
      D_RETURN_(FALSE);
    }
  
  D_RETURN_(TRUE);
  TOUCH(filter);
}


static gboolean
bpf_filter_apply(LND_Filter *filter,
		 LND_Packet *packet,
		 LND_BPF_Filter *filter_data)
{
  D_ENTER;

  if (bpf_filter(filter_data->bpf_program.bf_insns, packet->data,
		 packet->ph.len, packet->ph.caplen))
    D_RETURN_(TRUE);
  
  D_RETURN_(FALSE);
  TOUCH(filter);
}


static void
bpf_filter_cleanup(LND_Filter *filter,
		LND_BPF_Filter *filter_data)
{
  D_ENTER;

  if (!filter)
    D_RETURN;

  pcap_freecode(&filter_data->bpf_program);
  D_RETURN;
}



const char *
name(void)
{
  return "BPF-Filter";
}

const char *
author(void)
{
  return "Christian Kreibich, <christian@whoop.org>";
}

const char *
version(void)
{
  return VERSION;
}


void
run(LND_Trace *trace, void *user_data)
{
  LND_Filter *filter = libnd_bpf_new(user_data);

  if (!filter)
    return;

  libnd_filter_registry_add(filter);

  return;
  TOUCH(trace);
}


LND_Filter *
libnd_bpf_new(const LND_BPF_Params *params)
{
  LND_Filter         *filter;
  LND_BPF_Filter     *bpf_filter;
  struct bpf_program  bpf_code;

  D_ENTER;

  if (!params)
    D_RETURN_(NULL);

  if (pcap_compile_nopcap(1514, DLT_EN10MB, &bpf_code,
			  (char *) params->filter_expr, TRUE, 0xFFFFFFFF))
    {
      D(("The BPF filter expression '%s' could not be compiled.\n", params->filter_expr));
      D_RETURN_(NULL);
    }

  pcap_freecode(&bpf_code);
  
  if (! (bpf_filter = bpf_filter_new(params->filter_expr)))
    D_RETURN_(NULL);

  filter = libnd_filter_new(params->filter_name,
			    (LND_FilterInitFunc) bpf_filter_init,
			    (LND_FilterFunc) bpf_filter_apply,
			    (LND_FilterCleanupFunc) bpf_filter_cleanup,
			    (LND_FilterFreeFunc) bpf_filter_free,
			    bpf_filter);
  
  D_RETURN_(filter);
}
