// Copyright (c) the JPEG XL Project Authors. All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

#include "lib/jpegli/quant.h"

#include "lib/jxl/image_ops.h"

namespace jpegli {

namespace {
static constexpr float kBaseQuantMatrixXYB[] = {
    // c = 0
    0.010745695802f,
    0.014724285860f,
    0.016765073259f,
    0.015352546818f,
    0.016849715608f,
    0.017505664513f,
    0.019171796023f,
    0.026983627671f,
    0.014724285860f,
    0.016005879113f,
    0.014807802023f,
    0.015257294568f,
    0.016239266522f,
    0.017754112611f,
    0.021007430943f,
    0.024258001854f,
    0.016765073259f,
    0.014807802023f,
    0.016266879484f,
    0.014202573480f,
    0.016155362246f,
    0.018324768181f,
    0.018883664957f,
    0.024261275157f,
    0.015352546818f,
    0.015257294568f,
    0.014202573480f,
    0.014974020066f,
    0.018844302744f,
    0.019286162437f,
    0.023009874591f,
    0.023277331489f,
    0.016849715608f,
    0.016239266522f,
    0.016155362246f,
    0.018844302744f,
    0.019491371738f,
    0.030153905190f,
    0.032131952026f,
    0.047015070993f,
    0.017505664513f,
    0.017754112611f,
    0.018324768181f,
    0.019286162437f,
    0.030153905190f,
    0.035875428738f,
    0.025324149774f,
    0.046037739693f,
    0.019171796023f,
    0.021007430943f,
    0.018883664957f,
    0.023009874591f,
    0.032131952026f,
    0.025324149774f,
    0.025619236945f,
    0.049740249957f,
    0.026983627671f,
    0.024258001854f,
    0.024261275157f,
    0.023277331489f,
    0.047015070993f,
    0.046037739693f,
    0.049740249957f,
    0.029683058303f,
    // c = 1
    0.002310547025f,
    0.002391506241f,
    0.002592377991f,
    0.002907631930f,
    0.003590107614f,
    0.003647418579f,
    0.003946607583f,
    0.004580867024f,
    0.002391506241f,
    0.002565945978f,
    0.002676532241f,
    0.003167916799f,
    0.003592423110f,
    0.003581864537f,
    0.004168190188f,
    0.004711190832f,
    0.002592377991f,
    0.002676532241f,
    0.002899922325f,
    0.003221508920f,
    0.003597377805f,
    0.004015001363f,
    0.004164168214f,
    0.004536180462f,
    0.002907631930f,
    0.003167916799f,
    0.003221508920f,
    0.003421333194f,
    0.003843692347f,
    0.004011729362f,
    0.004486022354f,
    0.005037524314f,
    0.003590107614f,
    0.003592423110f,
    0.003597377805f,
    0.003843692347f,
    0.003991982168f,
    0.004561113887f,
    0.005683994831f,
    0.005587879717f,
    0.003647418579f,
    0.003581864537f,
    0.004015001363f,
    0.004011729362f,
    0.004561113887f,
    0.004711190832f,
    0.005279489671f,
    0.005645298559f,
    0.003946607583f,
    0.004168190188f,
    0.004164168214f,
    0.004486022354f,
    0.005683994831f,
    0.005279489671f,
    0.005269099460f,
    0.005234577959f,
    0.004580867024f,
    0.004711190832f,
    0.004536180462f,
    0.005037524314f,
    0.005587879717f,
    0.005645298559f,
    0.005234577959f,
    0.005138602544f,
    // c = 2
    0.004694191704f,
    0.007478405841f,
    0.009119519544f,
    0.010846788859f,
    0.012040055008f,
    0.014283609506f,
    0.020805819128f,
    0.041346026531f,
    0.007478405841f,
    0.008473337032f,
    0.008457467755f,
    0.011507290737f,
    0.012282006381f,
    0.011077942494f,
    0.019589180487f,
    0.030348661601f,
    0.009119519544f,
    0.008457467755f,
    0.012692131754f,
    0.010360988009f,
    0.011883779193f,
    0.021216622915f,
    0.019468523508f,
    0.022375231013f,
    0.010846788859f,
    0.011507290737f,
    0.010360988009f,
    0.015688875916f,
    0.019428087454f,
    0.018982414995f,
    0.030218311113f,
    0.025108166811f,
    0.012040055008f,
    0.012282006381f,
    0.011883779193f,
    0.019428087454f,
    0.019908111501f,
    0.019428676375f,
    0.026540699320f,
    0.032446303017f,
    0.014283609506f,
    0.011077942494f,
    0.021216622915f,
    0.018982414995f,
    0.019428676375f,
    0.025654942665f,
    0.030689090332f,
    0.036234971093f,
    0.020805819128f,
    0.019589180487f,
    0.019468523508f,
    0.030218311113f,
    0.026540699320f,
    0.030689090332f,
    0.035378000966f,
    0.041109510150f,
    0.041346026531f,
    0.030348661601f,
    0.022375231013f,
    0.025108166811f,
    0.032446303017f,
    0.036234971093f,
    0.041109510150f,
    0.047241950370f,
};

static const float kBaseQuantMatrixYCbCr[] = {
    // c = 0
    0.002000000095f,
    0.001999931643f,
    0.001999980304f,
    0.002170301275f,
    0.002292827703f,
    0.002597195329f,
    0.002653788775f,
    0.002672206843f,
    0.001999931643f,
    0.001999898814f,
    0.002019875217f,
    0.002171598841f,
    0.002367980313f,
    0.002273257589f,
    0.002367967507f,
    0.002844951348f,
    0.001999980304f,
    0.002019875217f,
    0.002193420427f,
    0.002641294384f,
    0.002754618181f,
    0.002270965371f,
    0.002780993935f,
    0.003155430313f,
    0.002170301275f,
    0.002171598841f,
    0.002641294384f,
    0.002272309968f,
    0.003103095805f,
    0.003655790119f,
    0.003553370945f,
    0.003420925234f,
    0.002292827703f,
    0.002367980313f,
    0.002754618181f,
    0.003103095805f,
    0.003229002934f,
    0.004116337746f,
    0.004140008241f,
    0.004050195683f,
    0.002597195329f,
    0.002273257589f,
    0.002270965371f,
    0.003655790119f,
    0.004116337746f,
    0.004039928783f,
    0.003790788585f,
    0.004529902712f,
    0.002653788775f,
    0.002367967507f,
    0.002780993935f,
    0.003553370945f,
    0.004140008241f,
    0.003790788585f,
    0.004021393601f,
    0.004049635492f,
    0.002672206843f,
    0.002844951348f,
    0.003155430313f,
    0.003420925234f,
    0.004050195683f,
    0.004529902712f,
    0.004049635492f,
    0.003237925470f,
    // c = 1
    0.004000000190f,
    0.007746503688f,
    0.011991346255f,
    0.011417495087f,
    0.013112261891f,
    0.014193773270f,
    0.012999247760f,
    0.021923458204f,
    0.007746503688f,
    0.011694040149f,
    0.012510522269f,
    0.011379567906f,
    0.013065839186f,
    0.012536211871f,
    0.020117942244f,
    0.015852928162f,
    0.011991346255f,
    0.012510522269f,
    0.010957037099f,
    0.013425820507f,
    0.013532835059f,
    0.017240526155f,
    0.072882637382f,
    0.017531186342f,
    0.011417495087f,
    0.011379567906f,
    0.013425820507f,
    0.012843516655f,
    0.011990147643f,
    0.017859818414f,
    0.020170627162f,
    0.026238106191f,
    0.013112261891f,
    0.013065839186f,
    0.013532835059f,
    0.011990147643f,
    0.014321122319f,
    0.012101849541f,
    0.014621426351f,
    0.037948686630f,
    0.014193773270f,
    0.012536211871f,
    0.017240526155f,
    0.017859818414f,
    0.012101849541f,
    0.024001309648f,
    0.033962588757f,
    0.047998063266f,
    0.012999247760f,
    0.020117942244f,
    0.072882637382f,
    0.020170627162f,
    0.014621426351f,
    0.033962588757f,
    0.045924827456f,
    0.063177861273f,
    0.021923458204f,
    0.015852928162f,
    0.017531186342f,
    0.026238106191f,
    0.037948686630f,
    0.047998063266f,
    0.063177861273f,
    0.083749666810f,
    // c = 2
    0.004000000190f,
    0.004065354355f,
    0.005469236057f,
    0.004842195194f,
    0.006053265650f,
    0.006479793228f,
    0.006478008348f,
    0.006953275762f,
    0.004065354355f,
    0.005311556160f,
    0.005351677071f,
    0.006025474519f,
    0.006053003948f,
    0.005298251286f,
    0.005826937500f,
    0.006202675402f,
    0.005469236057f,
    0.005351677071f,
    0.005141463131f,
    0.005998719018f,
    0.006771280896f,
    0.005474951584f,
    0.005469893571f,
    0.006446144544f,
    0.004842195194f,
    0.006025474519f,
    0.005998719018f,
    0.005815117154f,
    0.005747615825f,
    0.005839910824f,
    0.005526151508f,
    0.006295263767f,
    0.006053265650f,
    0.006053003948f,
    0.006771280896f,
    0.005747615825f,
    0.005270409398f,
    0.006623993628f,
    0.006093843374f,
    0.005827767309f,
    0.006479793228f,
    0.005298251286f,
    0.005474951584f,
    0.005839910824f,
    0.006623993628f,
    0.006392144598f,
    0.007216034457f,
    0.048031840473f,
    0.006478008348f,
    0.005826937500f,
    0.005469893571f,
    0.005526151508f,
    0.006093843374f,
    0.007216034457f,
    0.046043820679f,
    0.061863519251f,
    0.006953275762f,
    0.006202675402f,
    0.006446144544f,
    0.006295263767f,
    0.005827767309f,
    0.048031840473f,
    0.061863519251f,
    0.083964832127f,
};

}  // namespace

void AddJpegQuantMatrices(const jxl::ImageF& qf, bool xyb, float dc_quant,
                          float global_scale,
                          std::vector<jxl::jpeg::JPEGQuantTable>* quant_tables,
                          float* qm) {
  const float* const base_quant_matrix =
      xyb ? kBaseQuantMatrixXYB : kBaseQuantMatrixYCbCr;
  // Scale the base quant matrix based on the scaled XYB scales and the quant
  // field.
  float qfmin, qfmax;
  ImageMinMax(qf, &qfmin, &qfmax);
  const float dc_scale = global_scale / dc_quant;
  const float ac_scale = global_scale / qfmax;
  for (size_t c = 0, ix = 0; c < 3; c++) {
    qm[ix] = dc_scale * base_quant_matrix[ix];
    ix++;
    for (size_t j = 1; j < jxl::kDCTBlockSize; j++, ix++) {
      qm[ix] = ac_scale * base_quant_matrix[ix];
    }
  }

  // Save the quant matrix into the jpeg data and invert it.
  quant_tables->resize(3);
  for (size_t c = 0; c < 3; c++) {
    jxl::jpeg::JPEGQuantTable& quant = (*quant_tables)[c];
    quant.is_last = (c == 2);
    quant.index = c + 1;
    for (size_t j = 0; j < jxl::kDCTBlockSize; j++) {
      int qval = std::round(qm[c * jxl::kDCTBlockSize + j] * 255 * 8);
      quant.values[j] = std::max(1, std::min(qval, 255));
      qm[c * jxl::kDCTBlockSize + j] = 1.0f / quant.values[j];
    }
  }
}

}  // namespace jpegli
