/*
 *  $Id: trig.c,v 1.3 2003/12/31 20:18:51 mike Exp $
 *
 *  trig
 *  trig.c - traceroute with geo-targeting via libipgeo
 *
 *  Copyright (c) 1998 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <libnet.h>
#include <pcap.h>
#include "./libipgeo.h"

int
do_lookup(u_int32_t ipn, ipgeo_t *ipg);

u_int8_t do_cc, do_country, do_city, do_region, do_isp, do_lat, do_long;

int main(int argc, char **argv)
{
    pcap_t *p = NULL;
    libnet_t *l = NULL;
    ipgeo_t *ipg = NULL;
    time_t start;
    u_char *packet;
    int c, ttl, done;
    char *device = NULL;
    extern char *optarg;
    extern int optind;
    struct pcap_pkthdr ph;
    libnet_ptag_t icmp, ip;
    u_int32_t dst_ip;
    struct libnet_icmpv4_hdr *icmp_h;
    struct libnet_ipv4_hdr *ip_h, *oip_h;
    char errbuf[LIBNET_ERRBUF_SIZE];

    printf("Trig 1.0 [geo-targeting traceroute scanner]\n");
    do_cc = do_country = do_city = do_region = do_isp = do_lat = do_long = 0;
    while ((c = getopt(argc, argv, "i:CcyrsLl")) != EOF)
    {
        switch (c)
        {
            case 'i':
                device = optarg;
                break;
            case 'C':
                do_cc = 1;
                break;
            case 'c':
                do_country = 1;
                break;
            case 'y':
                do_city = 1;
                break;
            case 'L':
                do_lat = 1;
                break;
            case 'l':
                do_long = 1;
                break;
            case 'r':
                do_region = 1;
                break;
            case 's':
                do_isp = 1;
                break;
        }
    }

    c = argc - optind;
    if (c != 2)
    {
        fprintf(stderr, "usage:\t%s\t\t [-i interface][-CcyrsLl] host file\n",
                argv[0]);
        goto done;
    }

    if (do_cc == 0 && do_country == 0 && do_city == 0 && do_region == 0 &&
        do_isp == 0 && do_lat == 0 && do_long == 0)
    {
        printf("No IP geo-targeting?\n");
    }

    l = libnet_init(LIBNET_RAW4, device,  errbuf);
    if (l == NULL)
    {
        fprintf(stderr, "libnet: %s\n", errbuf);
        return (EXIT_FAILURE);
    }

    p = pcap_open_live(device, 60, 0, 500, errbuf);
    if (p == NULL)
    {
        fprintf(stderr, "pcap: %s\n", errbuf);
        return (EXIT_FAILURE);
    }
 
   ipg = ipgeo_init(argv[optind + 1], 0, errbuf);
   if (ipg == NULL)
    {
        fprintf(stderr, "ipgeo: %s\n", errbuf);
        return (EXIT_FAILURE);
    }

    dst_ip = libnet_name2addr4(l, argv[optind], LIBNET_RESOLVE);
    if (dst_ip == 0)
    {
        fprintf(stderr, "libnet: %s\n", libnet_geterror(l));
        goto done;
    }

    for (done = icmp = ip = 0, ttl = 1; ttl < 31 && !done; ttl++)
    {

        icmp = libnet_build_icmpv4_echo(
            ICMP_ECHO,                              /* type */
            0,                                      /* code */
            0,                                      /* checksum */
            242,                                    /* id */
            ttl,                                    /* sequence */
            NULL,                                   /* payload */
            0,                                      /* payload size */
            l,                                      /* libnet context */
            icmp);                                  /* libnet id */
        if (icmp == -1)
        {
            fprintf(stderr, "libnet: %s\n", libnet_geterror(l));
            return (EXIT_FAILURE);
        }
        ip = libnet_build_ipv4(
            LIBNET_IPV4_H + LIBNET_ICMPV4_ECHO_H,   /* length */
            0,                                      /* TOS */
            242,                                    /* IP ID */
            0,                                      /* IP Frag */
            ttl,                                    /* TTL */
            IPPROTO_ICMP,                           /* protocol */
            0,                                      /* checksum */
            libnet_get_ipaddr4(l),                  /* src ip */
            dst_ip,                                 /* dst ip */
            NULL,                                   /* payload */
            0,                                      /* payload size */
            l,                                      /* libnet context */
            ip);                                    /* libnet id */
        if (ip == -1)
        {
            fprintf(stderr, "libnet: %s\n", libnet_geterror(l));
            return (EXIT_FAILURE);
        }
        c = libnet_write(l);
        if (c == -1)
        {
            fprintf(stderr, "libnet: %s\n", libnet_geterror(l));
            return (EXIT_FAILURE);
        }

        fprintf(stderr, "%02d: ", ttl);
    
        /* read loop */
        for (start = time(NULL); (time(NULL) - start) < 2; )
        {
            packet = (u_char *)pcap_next(p, &ph);
            if (packet == NULL)
            {
                continue;
            }
            /* assume ethernet here for simplicity */
            ip_h = (struct libnet_ipv4_hdr *)(packet + 14);
            if (ip_h->ip_p == IPPROTO_ICMP)
            {
                icmp_h = (struct libnet_icmpv4_hdr *)(packet + 34);
                /* expired in transit */
                if (icmp_h->icmp_type == ICMP_TIMXCEED &&
                    icmp_h->icmp_code == ICMP_TIMXCEED_INTRANS)
                {
                    oip_h = (struct libnet_ipv4_hdr *)(packet + 42);
                    if (oip_h->ip_id == htons(242))
                    {
                        fprintf(stderr, "%s ",
                            libnet_addr2name4(ip_h->ip_src.s_addr, 0));
                        if (do_lookup(ip_h->ip_src.s_addr, ipg) == -1)
                        {
                            fprintf(stderr, "ipgeo: %s\n", ipgeo_geterror(ipg));
                        }
                        break;
                    }
                }
                /* terminal response */
                if (icmp_h->icmp_type == ICMP_ECHOREPLY)
                {
                    if (icmp_h->icmp_id == 242 && icmp_h->icmp_seq == ttl)
                    {
                        fprintf(stderr, "%s ",
                            libnet_addr2name4(ip_h->ip_src.s_addr, 0));
                        if (do_lookup(ip_h->ip_src.s_addr, ipg) == -1)
                        {
                            fprintf(stderr, "ipgeo: %s\n", ipgeo_geterror(ipg));
                        }
                        done = 1;
                        break;
                    }
                }
            }
        }
    }
done:
    if (l)
    {
        libnet_destroy(l);
    }
    if (p)
    {
        pcap_close(p);
    }
    if (ipg)
    {
        ipgeo_destroy(ipg);
    }
    return (EXIT_SUCCESS);
}

int
do_lookup(u_int32_t ipn, ipgeo_t *ipg)
{
    if (ipgeo_lookup(ipn, 0, ipg) == -1)
    {
        return (-1);
    }
    if (do_cc)
    {
        fprintf(stderr, "%s ", ipgeo_get_cc(ipg));
    }
    if (do_country)
    {
        fprintf(stderr, "%s ", ipgeo_get_country(ipg));
    }
    if (do_city)
    {
        fprintf(stderr, "%s ", ipgeo_get_city(ipg));
    }
    if (do_region)
    {
        fprintf(stderr, "%s ", ipgeo_get_region(ipg));
    }
    if (do_isp)
    {
        fprintf(stderr, "%s ", ipgeo_get_isp(ipg));
    }
    if (do_lat)
    {
        fprintf(stderr, "%.4f ", ipgeo_get_lat(ipg));
    }
    if (do_long)
    {
        fprintf(stderr, "%.4f ", ipgeo_get_long(ipg));
    }
    fprintf(stderr, "\n");
    return (1);
}

/* EOF */
