typedef struct object_data {
    unsigned char *bytes;  /* (compressed) bytes stored in object store */
    size_t        blen;    /* length of data in object store            */
    char          *id;     /* object id (hex chars)                     */
    git_oid_t     id_type; /* type of object id (sha1 or sha256)        */
    char          *type;   /* object type                               */
    char          *dir;    /* object store (fan-out) directory name     */
    char          *file;   /* object store filename                     */
    unsigned char *data;   /* (uncompressed) object data                */
    size_t        dlen;    /* length of (uncompressed) object data      */
} object_data;

/*
 * one == 8b137891791fe96927ad78e64b0aad7bded08bdc (sha1)
 * 4c0d52d180c61d01ce1a91dec5ee58f0cbe65fd59433aea803ab927965493fd7 (sha256)
 */
static unsigned char one_bytes[] = {
    0x31, 0x78, 0x9c, 0xe3, 0x02, 0x00, 0x00, 0x0b,
    0x00, 0x0b,
};

static unsigned char one_data[] = {
    0x0a,
};

static object_data one = {
    one_bytes,
    sizeof(one_bytes),
    "8b137891791fe96927ad78e64b0aad7bded08bdc",
    GIT_OID_SHA1,
    "blob",
    "test-objects/8b",
    "test-objects/8b/137891791fe96927ad78e64b0aad7bded08bdc",
    one_data,
    sizeof(one_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data one_sha256 = {
    one_bytes,
    sizeof(one_bytes),
    "4c0d52d180c61d01ce1a91dec5ee58f0cbe65fd59433aea803ab927965493fd7",
    GIT_OID_SHA256,
    "blob",
    "test-objects/4c",
    "test-objects/4c/0d52d180c61d01ce1a91dec5ee58f0cbe65fd59433aea803ab927965493fd7",
    one_data,
    sizeof(one_data),
};
#endif


/*
 * commit == 3d7f8a6af076c8c3f20071a8935cdbe8228594d1 (sha1)
 * a2a430fb63b294f868af4ef6ccc9c3e8256e370859ce578a23837ac85337f562 (sha256)
 */
static unsigned char commit_bytes[] = {
    0x78, 0x01, 0x85, 0x50, 0xc1, 0x6a, 0xc3, 0x30,
    0x0c, 0xdd, 0xd9, 0x5f, 0xa1, 0xfb, 0x96, 0x12,
    0xbb, 0x29, 0x71, 0x46, 0x19, 0x2b, 0x3d, 0x97,
    0x1d, 0xd6, 0x7d, 0x80, 0x1d, 0xcb, 0x89, 0x21,
    0xb6, 0x82, 0xed, 0x40, 0xf3, 0xf7, 0xf3, 0x48,
    0x29, 0x3b, 0x6d, 0xd2, 0xe5, 0xbd, 0x27, 0xbd,
    0x27, 0x50, 0x4f, 0xde, 0xbb, 0x0c, 0xfb, 0x43,
    0xf3, 0x94, 0x23, 0x22, 0x18, 0x6b, 0x85, 0x51,
    0x5d, 0xad, 0xc5, 0xa1, 0x41, 0xae, 0x51, 0x4b,
    0xd9, 0x19, 0x6e, 0x4b, 0x0b, 0x29, 0x35, 0x72,
    0x59, 0xef, 0x5b, 0x29, 0x8c, 0x65, 0x6a, 0xc9,
    0x23, 0x45, 0x38, 0xc1, 0x17, 0x5c, 0x7f, 0xc0,
    0x71, 0x13, 0xde, 0xf1, 0xa6, 0xfc, 0x3c, 0xe1,
    0xae, 0x27, 0xff, 0x06, 0x5c, 0x88, 0x56, 0xf2,
    0x46, 0x74, 0x2d, 0x3c, 0xd7, 0xa5, 0x58, 0x51,
    0xcb, 0xb9, 0x8c, 0x11, 0xce, 0xf0, 0x01, 0x97,
    0x0d, 0x1e, 0x1f, 0xea, 0x3f, 0x6e, 0x76, 0x02,
    0x0a, 0x58, 0x4d, 0x2e, 0x20, 0x6c, 0x1e, 0x48,
    0x8b, 0xf7, 0x2a, 0xae, 0x8c, 0x5d, 0x47, 0x04,
    0x4d, 0x66, 0x05, 0xb2, 0x90, 0x0b, 0xbe, 0xcf,
    0x3d, 0xa6, 0xa4, 0x06, 0x7c, 0x29, 0x3c, 0x64,
    0xe5, 0x82, 0x0b, 0x03, 0xd8, 0x25, 0x96, 0x8d,
    0x08, 0x78, 0x9b, 0x27, 0x15, 0x54, 0x76, 0x14,
    0xd8, 0xdd, 0x35, 0x2f, 0x71, 0xa6, 0x84, 0x8f,
    0x90, 0x51, 0x85, 0x01, 0x13, 0xb8, 0x90, 0x23,
    0x99, 0xa5, 0x47, 0x03, 0x7a, 0xfd, 0x15, 0xbf,
    0x63, 0xec, 0xd3, 0x0d, 0x01, 0x4d, 0x45, 0xb6,
    0xd2, 0xeb, 0xeb, 0xdf, 0xef, 0x60, 0xdf, 0xef,
    0x1f, 0x78, 0x35,
};

#ifdef GIT_EXPERIMENTAL_SHA256
static unsigned char commit_bytes_sha256[] = {
    0x78, 0x01, 0x85, 0x90, 0xc1, 0x4e, 0xc3, 0x30,
    0x0c, 0x86, 0x39, 0xe7, 0x29, 0x7c, 0x87, 0x4e,
    0x5d, 0x93, 0xa6, 0x2d, 0x9a, 0x10, 0x13, 0x67,
    0xc4, 0x81, 0xf1, 0x00, 0x4e, 0xe3, 0xb4, 0x91,
    0x9a, 0xa4, 0x4a, 0x53, 0x69, 0x7d, 0x7b, 0x82,
    0x3a, 0x4d, 0x9c, 0xc0, 0xa7, 0xcf, 0xbf, 0xfd,
    0xff, 0xb2, 0xdc, 0x07, 0xe7, 0x6c, 0x02, 0xde,
    0xb4, 0x0f, 0x29, 0x12, 0x01, 0x17, 0x28, 0xda,
    0x5a, 0xa8, 0x5a, 0x54, 0xd2, 0x74, 0x95, 0x90,
    0xa5, 0x12, 0x48, 0xbc, 0x26, 0xa9, 0x9b, 0xae,
    0x11, 0x52, 0x91, 0x94, 0x3d, 0x6f, 0x95, 0x31,
    0x5a, 0x92, 0xe1, 0xaa, 0x17, 0xa6, 0xac, 0x39,
    0xe9, 0xa6, 0x45, 0x2e, 0x15, 0x0a, 0x86, 0x6b,
    0x1a, 0x43, 0x84, 0x33, 0x7c, 0xc1, 0xe5, 0x07,
    0x4e, 0xbb, 0xf0, 0x4a, 0x57, 0x74, 0xf3, 0x44,
    0x87, 0x3e, 0xb8, 0x17, 0x38, 0x56, 0x55, 0xd3,
    0x1e, 0x45, 0xd5, 0x35, 0xf0, 0x58, 0xe6, 0x62,
    0x59, 0xcd, 0x67, 0x24, 0x8a, 0xf0, 0x06, 0x1f,
    0xf0, 0xbe, 0xe3, 0xe9, 0xae, 0xfe, 0xe3, 0x66,
    0x67, 0x08, 0x9e, 0x8a, 0xc9, 0x7a, 0x82, 0xdd,
    0x03, 0xcb, 0xea, 0x1c, 0xc6, 0x8d, 0xb1, 0xcb,
    0x48, 0xa0, 0x82, 0xde, 0x20, 0x18, 0x48, 0x99,
    0x6f, 0x73, 0x47, 0xcb, 0x82, 0x03, 0x3d, 0xe5,
    0xde, 0x27, 0xb4, 0xde, 0xfa, 0x01, 0xcc, 0x1a,
    0xf3, 0x46, 0x04, 0xba, 0xce, 0x13, 0x7a, 0x4c,
    0x36, 0x78, 0x76, 0x73, 0xcd, 0x6b, 0x9c, 0xc3,
    0x42, 0xf7, 0x90, 0x11, 0xfd, 0x40, 0x0b, 0x58,
    0x9f, 0x62, 0xd0, 0x6b, 0x4f, 0x1a, 0xd4, 0xf6,
    0x2b, 0xfe, 0xc0, 0xd8, 0xa7, 0x1d, 0x3c, 0xe9,
    0x22, 0x98, 0x42, 0x6d, 0xcf, 0x7f, 0xbf, 0x83,
    0x7d, 0x03, 0x6d, 0x1e, 0x7e, 0xa9
};
#endif

static unsigned char commit_data[] = {
    0x74, 0x72, 0x65, 0x65, 0x20, 0x64, 0x66, 0x66,
    0x32, 0x64, 0x61, 0x39, 0x30, 0x62, 0x32, 0x35,
    0x34, 0x65, 0x31, 0x62, 0x65, 0x62, 0x38, 0x38,
    0x39, 0x64, 0x31, 0x66, 0x31, 0x66, 0x31, 0x32,
    0x38, 0x38, 0x62, 0x65, 0x31, 0x38, 0x30, 0x33,
    0x37, 0x38, 0x32, 0x64, 0x66, 0x0a, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x20, 0x41, 0x20, 0x55,
    0x20, 0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61,
    0x75, 0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78,
    0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f,
    0x6d, 0x3e, 0x20, 0x31, 0x32, 0x32, 0x37, 0x38,
    0x31, 0x34, 0x32, 0x39, 0x37, 0x20, 0x2b, 0x30,
    0x30, 0x30, 0x30, 0x0a, 0x63, 0x6f, 0x6d, 0x6d,
    0x69, 0x74, 0x74, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x41, 0x20, 0x6f, 0x6e, 0x65,
    0x2d, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x73, 0x75, 0x6d,
    0x6d, 0x61, 0x72, 0x79, 0x0a, 0x0a, 0x54, 0x68,
    0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x6f,
    0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x6d, 0x65, 0x73,
    0x73, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f,
    0x6e, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
    0x20, 0x66, 0x75, 0x72, 0x74, 0x68, 0x65, 0x72,
    0x20, 0x65, 0x78, 0x70, 0x6c, 0x61, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70,
    0x6f, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x63, 0x68, 0x61, 0x6e, 0x67,
    0x65, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x72, 0x6f,
    0x64, 0x75, 0x63, 0x65, 0x64, 0x20, 0x62, 0x79,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x2e, 0x0a, 0x0a, 0x53, 0x69,
    0x67, 0x6e, 0x65, 0x64, 0x2d, 0x6f, 0x66, 0x2d,
    0x62, 0x79, 0x3a, 0x20, 0x41, 0x20, 0x55, 0x20,
    0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78, 0x61,
    0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d,
    0x3e, 0x0a,
};

#ifdef GIT_EXPERIMENTAL_SHA256
static unsigned char commit_data_sha256[] = {
    0x74, 0x72, 0x65, 0x65, 0x20, 0x33, 0x34, 0x61,
    0x34, 0x38, 0x35, 0x34, 0x62, 0x35, 0x34, 0x32,
    0x36, 0x66, 0x39, 0x32, 0x34, 0x36, 0x30, 0x62,
    0x34, 0x61, 0x65, 0x33, 0x35, 0x65, 0x36, 0x64,
    0x37, 0x39, 0x37, 0x34, 0x36, 0x62, 0x65, 0x36,
    0x36, 0x63, 0x33, 0x38, 0x62, 0x66, 0x66, 0x64,
    0x36, 0x65, 0x66, 0x33, 0x62, 0x63, 0x34, 0x66,
    0x30, 0x35, 0x33, 0x65, 0x64, 0x37, 0x38, 0x61,
    0x33, 0x36, 0x62, 0x61, 0x34, 0x0a, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x20, 0x41, 0x20, 0x55,
    0x20, 0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61,
    0x75, 0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78,
    0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f,
    0x6d, 0x3e, 0x20, 0x31, 0x32, 0x32, 0x37, 0x38,
    0x31, 0x34, 0x32, 0x39, 0x37, 0x20, 0x2b, 0x30,
    0x30, 0x30, 0x30, 0x0a, 0x63, 0x6f, 0x6d, 0x6d,
    0x69, 0x74, 0x74, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x41, 0x20, 0x6f, 0x6e, 0x65,
    0x2d, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x73, 0x75, 0x6d,
    0x6d, 0x61, 0x72, 0x79, 0x0a, 0x0a, 0x54, 0x68,
    0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x6f,
    0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x6d, 0x65, 0x73,
    0x73, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f,
    0x6e, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
    0x20, 0x66, 0x75, 0x72, 0x74, 0x68, 0x65, 0x72,
    0x20, 0x65, 0x78, 0x70, 0x6c, 0x61, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70,
    0x6f, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x63, 0x68, 0x61, 0x6e, 0x67,
    0x65, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x72, 0x6f,
    0x64, 0x75, 0x63, 0x65, 0x64, 0x20, 0x62, 0x79,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x2e, 0x0a, 0x0a, 0x53, 0x69,
    0x67, 0x6e, 0x65, 0x64, 0x2d, 0x6f, 0x66, 0x2d,
    0x62, 0x79, 0x3a, 0x20, 0x41, 0x20, 0x55, 0x20,
    0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78, 0x61,
    0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d,
    0x3e, 0x0a
};
#endif

static object_data commit = {
    commit_bytes,
    sizeof(commit_bytes),
    "3d7f8a6af076c8c3f20071a8935cdbe8228594d1",
    GIT_OID_SHA1,
    "commit",
    "test-objects/3d",
    "test-objects/3d/7f8a6af076c8c3f20071a8935cdbe8228594d1",
    commit_data,
    sizeof(commit_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data commit_sha256 = {
    commit_bytes_sha256,
    sizeof(commit_bytes_sha256),
    "a2a430fb63b294f868af4ef6ccc9c3e8256e370859ce578a23837ac85337f562",
    GIT_OID_SHA256,
    "commit",
    "test-objects/a2",
    "test-objects/a2/a430fb63b294f868af4ef6ccc9c3e8256e370859ce578a23837ac85337f562",
    commit_data_sha256,
    sizeof(commit_data_sha256),
};
#endif

/*
 * tree == dff2da90b254e1beb889d1f1f1288be1803782df (sha1)
 * 34a4854b5426f92460b4ae35e6d79746be66c38bffd6ef3bc4f053ed78a36ba4 (sha256)
 */
static unsigned char tree_bytes[] = {
    0x78, 0x01, 0x2b, 0x29, 0x4a, 0x4d, 0x55, 0x30,
    0x34, 0x32, 0x63, 0x30, 0x34, 0x30, 0x30, 0x33,
    0x31, 0x51, 0xc8, 0xcf, 0x4b, 0x65, 0xe8, 0x16,
    0xae, 0x98, 0x58, 0x29, 0xff, 0x32, 0x53, 0x7d,
    0x6d, 0xc5, 0x33, 0x6f, 0xae, 0xb5, 0xd5, 0xf7,
    0x2e, 0x74, 0xdf, 0x81, 0x4a, 0x17, 0xe7, 0xe7,
    0xa6, 0x32, 0xfc, 0x6d, 0x31, 0xd8, 0xd3, 0xe6,
    0xf3, 0xe7, 0xea, 0x47, 0xbe, 0xd0, 0x09, 0x3f,
    0x96, 0xb8, 0x3f, 0x90, 0x9e, 0xa2, 0xfd, 0x0f,
    0x2a, 0x5f, 0x52, 0x9e, 0xcf, 0x50, 0x31, 0x43,
    0x52, 0x29, 0xd1, 0x5a, 0xeb, 0x77, 0x82, 0x2a,
    0x8b, 0xfe, 0xb7, 0xbd, 0xed, 0x5d, 0x07, 0x67,
    0xfa, 0xb5, 0x42, 0xa5, 0xab, 0x52, 0x8b, 0xf2,
    0x19, 0x9e, 0xcd, 0x7d, 0x34, 0x7b, 0xd3, 0xc5,
    0x6b, 0xce, 0xde, 0xdd, 0x9a, 0xeb, 0xca, 0xa3,
    0x6e, 0x1c, 0x7a, 0xd2, 0x13, 0x3c, 0x11, 0x00,
    0xe2, 0xaa, 0x38, 0x57,
};

static unsigned char tree_data[] = {
    0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x6f,
    0x6e, 0x65, 0x00, 0x8b, 0x13, 0x78, 0x91, 0x79,
    0x1f, 0xe9, 0x69, 0x27, 0xad, 0x78, 0xe6, 0x4b,
    0x0a, 0xad, 0x7b, 0xde, 0xd0, 0x8b, 0xdc, 0x31,
    0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x73, 0x6f,
    0x6d, 0x65, 0x00, 0xfd, 0x84, 0x30, 0xbc, 0x86,
    0x4c, 0xfc, 0xd5, 0xf1, 0x0e, 0x55, 0x90, 0xf8,
    0xa4, 0x47, 0xe0, 0x1b, 0x94, 0x2b, 0xfe, 0x31,
    0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x74, 0x77,
    0x6f, 0x00, 0x78, 0x98, 0x19, 0x22, 0x61, 0x3b,
    0x2a, 0xfb, 0x60, 0x25, 0x04, 0x2f, 0xf6, 0xbd,
    0x87, 0x8a, 0xc1, 0x99, 0x4e, 0x85, 0x31, 0x30,
    0x30, 0x36, 0x34, 0x34, 0x20, 0x7a, 0x65, 0x72,
    0x6f, 0x00, 0xe6, 0x9d, 0xe2, 0x9b, 0xb2, 0xd1,
    0xd6, 0x43, 0x4b, 0x8b, 0x29, 0xae, 0x77, 0x5a,
    0xd8, 0xc2, 0xe4, 0x8c, 0x53, 0x91,
};

#ifdef GIT_EXPERIMENTAL_SHA256
static unsigned char tree_bytes_sha256[] = {
    0x78, 0x01, 0x2b, 0x29, 0x4a, 0x4d, 0x55, 0x30,
    0x32, 0x32, 0x66, 0x30, 0x34, 0x30, 0x30, 0x33,
    0x31, 0x51, 0xc8, 0x48, 0xcd, 0xc9, 0xc9, 0xd7,
    0x2b, 0xa9, 0x28, 0x61, 0x28, 0x65, 0x3b, 0x7d,
    0xde, 0x27, 0x5c, 0xfb, 0xe5, 0x83, 0x2c, 0xf9,
    0xb7, 0xa6, 0x6b, 0xa2, 0x65, 0x7f, 0x6c, 0x5d,
    0xee, 0xab, 0x76, 0xa0, 0x9e, 0x49, 0xcd, 0xe3,
    0xe9, 0xcd, 0xa8, 0xf9, 0xf9, 0x5a, 0x50, 0x0d,
    0xf9, 0x79, 0xa9, 0x0c, 0x3e, 0xbc, 0x41, 0x17,
    0x1b, 0x8e, 0xc9, 0x32, 0x9e, 0x93, 0x9a, 0x78,
    0xef, 0xe8, 0xbb, 0x88, 0x0f, 0xa7, 0x9f, 0xc5,
    0x5f, 0x9d, 0x62, 0xbc, 0x6e, 0x05, 0xf3, 0xea,
    0x49, 0x95, 0xa9, 0x9e, 0xf6, 0xd7, 0xa1, 0x4a,
    0x8b, 0xf3, 0x73, 0x53, 0x19, 0x38, 0x6c, 0xb4,
    0xbb, 0x5d, 0xc2, 0x1c, 0x2e, 0x16, 0x3e, 0x5f,
    0x95, 0x56, 0xcd, 0x6d, 0xc4, 0x50, 0xc0, 0xf6,
    0xbd, 0xad, 0x50, 0xc0, 0xe8, 0xf5, 0x0e, 0x4d,
    0xc3, 0x33, 0xcb, 0xe6, 0x1c, 0x8c, 0x86, 0xaa,
    0x2d, 0x29, 0xcf, 0x67, 0xf8, 0x91, 0x14, 0xe7,
    0xfc, 0xf3, 0x81, 0xbf, 0x8a, 0xa6, 0x7c, 0xf9,
    0xd9, 0x7d, 0x3e, 0x85, 0x9b, 0x0f, 0x2d, 0xde,
    0xc0, 0x60, 0x9f, 0xe0, 0x38, 0xdb, 0xee, 0x42,
    0x16, 0x6b, 0x6f, 0x59, 0x4e, 0x37, 0x54, 0x69,
    0x55, 0x6a, 0x51, 0x3e, 0x83, 0xcb, 0xbc, 0xd9,
    0x95, 0x21, 0x0a, 0x67, 0xc5, 0xfe, 0x25, 0xac,
    0x0d, 0x9a, 0x71, 0x3e, 0x83, 0x5f, 0x74, 0xf9,
    0x59, 0xad, 0x93, 0x5b, 0xbc, 0x6e, 0x7d, 0x7f,
    0x6b, 0x77, 0x87, 0x97, 0xe3, 0x6e, 0x05, 0x00,
    0xba, 0xd1, 0x5f, 0x75
};

static unsigned char tree_data_sha256[] = {
    0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x68,
    0x65, 0x6c, 0x6c, 0x6f, 0x2e, 0x74, 0x78, 0x74,
    0x00, 0x75, 0x06, 0xcb, 0xcf, 0x4c, 0x57, 0x2b,
    0xe9, 0xe0, 0x6a, 0x1f, 0xed, 0x35, 0xac, 0x5b,
    0x1d, 0xf8, 0xb5, 0xa7, 0x4d, 0x26, 0xc0, 0x7f,
    0x02, 0x26, 0x48, 0xe5, 0xd9, 0x5a, 0x9f, 0x6f,
    0x2a, 0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20,
    0x6f, 0x6e, 0x65, 0x00, 0x4c, 0x0d, 0x52, 0xd1,
    0x80, 0xc6, 0x1d, 0x01, 0xce, 0x1a, 0x91, 0xde,
    0xc5, 0xee, 0x58, 0xf0, 0xcb, 0xe6, 0x5f, 0xd5,
    0x94, 0x33, 0xae, 0xa8, 0x03, 0xab, 0x92, 0x79,
    0x65, 0x49, 0x3f, 0xd7, 0x31, 0x30, 0x30, 0x36,
    0x34, 0x34, 0x20, 0x73, 0x6f, 0x6d, 0x65, 0x00,
    0x08, 0x3c, 0x2b, 0x8b, 0x44, 0x56, 0x40, 0xd1,
    0x71, 0xe7, 0xaa, 0x66, 0x7b, 0x0b, 0x32, 0x00,
    0x70, 0x06, 0xf7, 0x86, 0x71, 0x10, 0x32, 0xeb,
    0xb8, 0x29, 0x31, 0xcc, 0xa6, 0x9c, 0xc1, 0x5b,
    0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x74,
    0x77, 0x6f, 0x00, 0xf8, 0x62, 0x5e, 0x43, 0xf9,
    0xe0, 0x4f, 0x24, 0x29, 0x1f, 0x77, 0xcd, 0xbe,
    0x4c, 0x71, 0xb3, 0xc2, 0xa3, 0xb0, 0x00, 0x3f,
    0x60, 0x41, 0x9b, 0x3e, 0xd0, 0x6a, 0x05, 0x8d,
    0x76, 0x6c, 0x8b, 0x31, 0x30, 0x30, 0x36, 0x34,
    0x34, 0x20, 0x7a, 0x65, 0x72, 0x6f, 0x00, 0x44,
    0x9e, 0x9b, 0x79, 0x54, 0x20, 0xcd, 0x16, 0xfe,
    0x60, 0xad, 0x52, 0x98, 0xcf, 0x68, 0x0f, 0x15,
    0xa7, 0xcd, 0x2a, 0xc9, 0xb4, 0x4a, 0xda, 0xf7,
    0xed, 0x3e, 0xdc, 0x0d, 0x08, 0xdd, 0x78
};
#endif

static object_data tree = {
    tree_bytes,
    sizeof(tree_bytes),
    "dff2da90b254e1beb889d1f1f1288be1803782df",
    GIT_OID_SHA1,
    "tree",
    "test-objects/df",
    "test-objects/df/f2da90b254e1beb889d1f1f1288be1803782df",
    tree_data,
    sizeof(tree_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data tree_sha256 = {
    tree_bytes_sha256,
    sizeof(tree_bytes_sha256),
    "34a4854b5426f92460b4ae35e6d79746be66c38bffd6ef3bc4f053ed78a36ba4",
    GIT_OID_SHA256,
    "tree",
    "test-objects/34",
    "test-objects/34/a4854b5426f92460b4ae35e6d79746be66c38bffd6ef3bc4f053ed78a36ba4",
    tree_data_sha256,
    sizeof(tree_data_sha256),
};
#endif

/*
 * tag == 09d373e1dfdc16b129ceec6dd649739911541e05 (sha1)
 * f535d7595d5d0e5e530b5deb34542c96491fea300a1318036b605306548cb225 (sha256)
 */
static unsigned char tag_bytes[] = {
    0x78, 0x01, 0x35, 0x4e, 0xcb, 0x0a, 0xc2, 0x40,
    0x10, 0xf3, 0xbc, 0x5f, 0x31, 0x77, 0xa1, 0xec,
    0xa3, 0xed, 0x6e, 0x41, 0x44, 0xf0, 0x2c, 0x5e,
    0xfc, 0x81, 0xe9, 0x76, 0xb6, 0xad, 0xb4, 0xb4,
    0x6c, 0x07, 0xd1, 0xbf, 0x77, 0x44, 0x0d, 0x39,
    0x84, 0x10, 0x92, 0x30, 0xf6, 0x60, 0xbc, 0xdb,
    0x2d, 0xed, 0x9d, 0x22, 0x83, 0xeb, 0x7c, 0x0a,
    0x58, 0x63, 0xd2, 0xbe, 0x8e, 0x21, 0xba, 0x64,
    0xb5, 0xf6, 0x06, 0x43, 0xe3, 0xaa, 0xd8, 0xb5,
    0x14, 0xac, 0x0d, 0x55, 0x53, 0x76, 0x46, 0xf1,
    0x6b, 0x25, 0x88, 0xcb, 0x3c, 0x8f, 0xac, 0x58,
    0x3a, 0x1e, 0xba, 0xd0, 0x85, 0xd8, 0xd8, 0xf7,
    0x94, 0xe1, 0x0c, 0x57, 0xb8, 0x8c, 0xcc, 0x22,
    0x0f, 0xdf, 0x90, 0xc8, 0x13, 0x3d, 0x71, 0x5e,
    0x27, 0x2a, 0xc4, 0x39, 0x82, 0xb1, 0xd6, 0x07,
    0x53, 0xda, 0xc6, 0xc3, 0x5e, 0x0b, 0x94, 0xba,
    0x0d, 0xe3, 0x06, 0x42, 0x1e, 0x08, 0x3e, 0x95,
    0xbf, 0x4b, 0x69, 0xc9, 0x90, 0x69, 0x22, 0xdc,
    0xe8, 0xbf, 0xf2, 0x06, 0x42, 0x9a, 0x36, 0xb1,
};

static unsigned char tag_data[] = {
    0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x33,
    0x64, 0x37, 0x66, 0x38, 0x61, 0x36, 0x61, 0x66,
    0x30, 0x37, 0x36, 0x63, 0x38, 0x63, 0x33, 0x66,
    0x32, 0x30, 0x30, 0x37, 0x31, 0x61, 0x38, 0x39,
    0x33, 0x35, 0x63, 0x64, 0x62, 0x65, 0x38, 0x32,
    0x32, 0x38, 0x35, 0x39, 0x34, 0x64, 0x31, 0x0a,
    0x74, 0x79, 0x70, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x0a, 0x74, 0x61, 0x67, 0x20,
    0x76, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x0a, 0x74,
    0x61, 0x67, 0x67, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x54, 0x68, 0x69, 0x73, 0x20,
    0x69, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74,
    0x61, 0x67, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63,
    0x74, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x72, 0x65,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x76, 0x30,
    0x2e, 0x30, 0x2e, 0x31, 0x0a,
};

#ifdef GIT_EXPERIMENTAL_SHA256
static unsigned char tag_bytes_sha256[] = {
    0x78, 0x01, 0x55, 0x8f, 0xd1, 0x4e, 0x84, 0x30,
    0x10, 0x45, 0x7d, 0xee, 0x57, 0xcc, 0xbb, 0x2e,
    0x81, 0x16, 0x68, 0x31, 0x1b, 0xa3, 0xf1, 0xd9,
    0xf8, 0xe0, 0xfa, 0x01, 0x43, 0x99, 0x42, 0x0d,
    0xb4, 0xa4, 0x14, 0xb3, 0xfc, 0xbd, 0xc5, 0xdd,
    0x4d, 0xb4, 0x69, 0xd2, 0x9b, 0xc9, 0xdc, 0x7b,
    0x6e, 0x23, 0xf6, 0x20, 0xa4, 0xba, 0xf3, 0xed,
    0x17, 0xe9, 0x08, 0xc8, 0xb1, 0x14, 0xb9, 0x69,
    0x6b, 0xd1, 0xf2, 0xa6, 0x34, 0xaa, 0x56, 0x68,
    0x4a, 0x32, 0xb5, 0xd6, 0xba, 0xd1, 0x82, 0x14,
    0xaf, 0x6a, 0x12, 0x32, 0x57, 0x55, 0xa3, 0xa9,
    0x92, 0x0a, 0xb9, 0x50, 0x42, 0xa2, 0x56, 0x95,
    0x10, 0xd2, 0x54, 0x35, 0x67, 0x71, 0x9b, 0x09,
    0xb4, 0x9f, 0x26, 0x1b, 0x59, 0x4c, 0xd9, 0xdf,
    0x79, 0x96, 0x67, 0xc5, 0x2e, 0x7b, 0x0a, 0xf0,
    0x0a, 0xef, 0xf0, 0x66, 0x63, 0x4c, 0xf2, 0x78,
    0x59, 0x4a, 0xf2, 0x99, 0xce, 0x38, 0xcd, 0x23,
    0x65, 0x69, 0xf2, 0x04, 0x05, 0xe7, 0x52, 0x15,
    0x25, 0x6f, 0x24, 0xdc, 0xe7, 0xe9, 0x30, 0x76,
    0x1a, 0xec, 0x02, 0xe9, 0xc6, 0x81, 0x60, 0x8f,
    0xbc, 0x56, 0x35, 0x3e, 0x40, 0xa0, 0x91, 0x70,
    0xa1, 0x1b, 0xe5, 0x0a, 0x86, 0x65, 0x9d, 0x26,
    0x0c, 0xdb, 0x6e, 0x25, 0x68, 0x7d, 0xb7, 0x81,
    0x37, 0xbf, 0xf6, 0x0b, 0x13, 0x26, 0x5a, 0x16,
    0xec, 0xe9, 0x21, 0xed, 0xbb, 0x88, 0xd6, 0x59,
    0xd7, 0x83, 0x59, 0x43, 0x02, 0x04, 0xa0, 0xf3,
    0x3c, 0xa2, 0xc3, 0x68, 0xbd, 0x63, 0x57, 0xd7,
    0xbc, 0x86, 0xd9, 0x27, 0xca, 0x2d, 0x64, 0x40,
    0xd7, 0x53, 0xaa, 0xe4, 0x62, 0xf0, 0xdd, 0xaa,
    0xa9, 0x83, 0x76, 0xfb, 0x13, 0x9f, 0x31, 0xf6,
    0x61, 0x7b, 0x47, 0xdd, 0xc1, 0x9b, 0x43, 0xbb,
    0x3d, 0xc2, 0x0b, 0x7c, 0xc2, 0x69, 0x48, 0x75,
    0x8f, 0xb8, 0xc6, 0xf4, 0xfe, 0xfb, 0x30, 0xfb,
    0x01, 0xc9, 0x32, 0x7d, 0xbb
};

static unsigned char tag_data_sha256[] = {
    0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x61,
    0x32, 0x61, 0x34, 0x33, 0x30, 0x66, 0x62, 0x36,
    0x33, 0x62, 0x32, 0x39, 0x34, 0x66, 0x38, 0x36,
    0x38, 0x61, 0x66, 0x34, 0x65, 0x66, 0x36, 0x63,
    0x63, 0x63, 0x39, 0x63, 0x33, 0x65, 0x38, 0x32,
    0x35, 0x36, 0x65, 0x33, 0x37, 0x30, 0x38, 0x35,
    0x39, 0x63, 0x65, 0x35, 0x37, 0x38, 0x61, 0x32,
    0x33, 0x38, 0x33, 0x37, 0x61, 0x63, 0x38, 0x35,
    0x33, 0x33, 0x37, 0x66, 0x35, 0x36, 0x32, 0x0a,
    0x74, 0x79, 0x70, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x0a, 0x74, 0x61, 0x67, 0x20,
    0x76, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x0a, 0x74,
    0x61, 0x67, 0x67, 0x65, 0x72, 0x20, 0x43, 0x20,
    0x4f, 0x20, 0x4d, 0x69, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x3c, 0x63, 0x6f, 0x6d, 0x6d, 0x69, 0x74,
    0x74, 0x65, 0x72, 0x40, 0x65, 0x78, 0x61, 0x6d,
    0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x3e,
    0x20, 0x31, 0x32, 0x32, 0x37, 0x38, 0x31, 0x34,
    0x32, 0x39, 0x37, 0x20, 0x2b, 0x30, 0x30, 0x30,
    0x30, 0x0a, 0x0a, 0x54, 0x68, 0x69, 0x73, 0x20,
    0x69, 0x73, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74,
    0x61, 0x67, 0x20, 0x6f, 0x62, 0x6a, 0x65, 0x63,
    0x74, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x72, 0x65,
    0x6c, 0x65, 0x61, 0x73, 0x65, 0x20, 0x76, 0x30,
    0x2e, 0x30, 0x2e, 0x31, 0x0a, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x73, 0x75, 0x6d,
    0x6d, 0x61, 0x72, 0x79, 0x0a, 0x0a, 0x54, 0x68,
    0x65, 0x20, 0x62, 0x6f, 0x64, 0x79, 0x20, 0x6f,
    0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x6d, 0x69, 0x74, 0x20, 0x6d, 0x65, 0x73,
    0x73, 0x61, 0x67, 0x65, 0x2c, 0x20, 0x63, 0x6f,
    0x6e, 0x74, 0x61, 0x69, 0x6e, 0x69, 0x6e, 0x67,
    0x20, 0x66, 0x75, 0x72, 0x74, 0x68, 0x65, 0x72,
    0x20, 0x65, 0x78, 0x70, 0x6c, 0x61, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x70, 0x75, 0x72, 0x70,
    0x6f, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x63, 0x68, 0x61, 0x6e, 0x67,
    0x65, 0x73, 0x20, 0x69, 0x6e, 0x74, 0x72, 0x6f,
    0x64, 0x75, 0x63, 0x65, 0x64, 0x20, 0x62, 0x79,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f, 0x6d,
    0x6d, 0x69, 0x74, 0x2e, 0x0a, 0x0a, 0x53, 0x69,
    0x67, 0x6e, 0x65, 0x64, 0x2d, 0x6f, 0x66, 0x2d,
    0x62, 0x79, 0x3a, 0x20, 0x41, 0x20, 0x55, 0x20,
    0x54, 0x68, 0x6f, 0x72, 0x20, 0x3c, 0x61, 0x75,
    0x74, 0x68, 0x6f, 0x72, 0x40, 0x65, 0x78, 0x61,
    0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d,
    0x3e, 0x0a
};
#endif

static object_data tag = {
    tag_bytes,
    sizeof(tag_bytes),
    "09d373e1dfdc16b129ceec6dd649739911541e05",
    GIT_OID_SHA1,
    "tag",
    "test-objects/09",
    "test-objects/09/d373e1dfdc16b129ceec6dd649739911541e05",
    tag_data,
    sizeof(tag_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data tag_sha256 = {
    tag_bytes_sha256,
    sizeof(tag_bytes_sha256),
    "f535d7595d5d0e5e530b5deb34542c96491fea300a1318036b605306548cb225",
    GIT_OID_SHA256,
    "tag",
    "test-objects/f5",
    "test-objects/f5/35d7595d5d0e5e530b5deb34542c96491fea300a1318036b605306548cb225",
    tag_data_sha256,
    sizeof(tag_data_sha256),
};
#endif

/*
 * zero == e69de29bb2d1d6434b8b29ae775ad8c2e48c5391 (sha1)
 * 473a0f4c3be8a93681a267e3b1e9a7dcda1185436fe141f7749120a303721813 (sha256)
 */
static unsigned char zero_bytes[] = {
    0x78, 0x01, 0x4b, 0xca, 0xc9, 0x4f, 0x52, 0x30,
    0x60, 0x00, 0x00, 0x09, 0xb0, 0x01, 0xf0,
};

static unsigned char zero_data[] = {
    0x00  /* dummy data */
};

static object_data zero = {
    zero_bytes,
    sizeof(zero_bytes),
    "e69de29bb2d1d6434b8b29ae775ad8c2e48c5391",
    GIT_OID_SHA1,
    "blob",
    "test-objects/e6",
    "test-objects/e6/9de29bb2d1d6434b8b29ae775ad8c2e48c5391",
    zero_data,
    0,
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data zero_sha256 = {
    zero_bytes,
    sizeof(zero_bytes),
    "473a0f4c3be8a93681a267e3b1e9a7dcda1185436fe141f7749120a303721813",
    GIT_OID_SHA256,
    "blob",
    "test-objects/47",
    "test-objects/47/3a0f4c3be8a93681a267e3b1e9a7dcda1185436fe141f7749120a303721813",
    zero_data,
    0,
};
#endif

/*
 * two == 78981922613b2afb6025042ff6bd878ac1994e85 (sha1)
 * f8625e43f9e04f24291f77cdbe4c71b3c2a3b0003f60419b3ed06a058d766c8b (sha256)
 */
static unsigned char two_bytes[] = {
    0x78, 0x01, 0x4b, 0xca, 0xc9, 0x4f, 0x52, 0x30,
    0x62, 0x48, 0xe4, 0x02, 0x00, 0x0e, 0x64, 0x02,
    0x5d,
};

static unsigned char two_data[] = {
    0x61, 0x0a,
};

static object_data two = {
    two_bytes,
    sizeof(two_bytes),
    "78981922613b2afb6025042ff6bd878ac1994e85",
    GIT_OID_SHA1,
    "blob",
    "test-objects/78",
    "test-objects/78/981922613b2afb6025042ff6bd878ac1994e85",
    two_data,
    sizeof(two_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data two_sha256 = {
    two_bytes,
    sizeof(two_bytes),
    "f8625e43f9e04f24291f77cdbe4c71b3c2a3b0003f60419b3ed06a058d766c8b",
    GIT_OID_SHA256,
    "blob",
    "test-objects/f8",
    "test-objects/f8/625e43f9e04f24291f77cdbe4c71b3c2a3b0003f60419b3ed06a058d766c8b",
    two_data,
    sizeof(two_data),
};
#endif

/*
 * some == fd8430bc864cfcd5f10e5590f8a447e01b942bfe (sha1)
 * 083c2b8b445640d171e7aa667b0b32007006f786711032ebb82931cca69cc15b (sha256)
 */
static unsigned char some_bytes[] = {
    0x78, 0x01, 0x7d, 0x54, 0xc1, 0x4e, 0xe3, 0x30,
    0x10, 0xdd, 0x33, 0x5f, 0x31, 0xc7, 0x5d, 0x94,
    0xa5, 0x84, 0xd5, 0x22, 0xad, 0x7a, 0x0a, 0x15,
    0x85, 0x48, 0xd0, 0x56, 0x49, 0x2a, 0xd4, 0xa3,
    0x13, 0x4f, 0x88, 0x85, 0x63, 0x47, 0xb6, 0x43,
    0xc9, 0xdf, 0xef, 0x8c, 0x69, 0x17, 0x56, 0x0b,
    0x7b, 0xaa, 0x62, 0x7b, 0xde, 0xbc, 0xf7, 0xe6,
    0x4d, 0x6b, 0x6d, 0x6b, 0x48, 0xd3, 0xcb, 0x5f,
    0x5f, 0x66, 0xa7, 0x27, 0x70, 0x0a, 0x55, 0xa7,
    0x3c, 0xb4, 0x4a, 0x23, 0xf0, 0xaf, 0x43, 0x04,
    0x6f, 0xdb, 0xb0, 0x17, 0x0e, 0xe7, 0x30, 0xd9,
    0x11, 0x1a, 0x61, 0xc0, 0xa1, 0x54, 0x3e, 0x38,
    0x55, 0x8f, 0x81, 0x9e, 0x05, 0x10, 0x46, 0xce,
    0xac, 0x83, 0xde, 0x4a, 0xd5, 0x4e, 0x0c, 0x42,
    0x67, 0xa3, 0x91, 0xe8, 0x20, 0x74, 0x08, 0x01,
    0x5d, 0xef, 0xc1, 0xb6, 0xf1, 0xe3, 0x66, 0xb5,
    0x85, 0x1b, 0x34, 0xe8, 0x84, 0x86, 0xcd, 0x58,
    0x6b, 0xd5, 0xc0, 0x9d, 0x6a, 0xd0, 0x78, 0x4c,
    0xe0, 0x19, 0x9d, 0x57, 0xd6, 0xc0, 0x45, 0xc2,
    0x18, 0xc2, 0xc3, 0xc0, 0x0f, 0x7c, 0x87, 0x12,
    0xea, 0x29, 0x56, 0x2f, 0x99, 0x4f, 0x79, 0xe0,
    0x03, 0x4b, 0x4b, 0x4d, 0x44, 0xa0, 0x92, 0x33,
    0x2a, 0xe0, 0x9a, 0xdc, 0x80, 0x90, 0x52, 0xf1,
    0x11, 0x04, 0x1b, 0x4b, 0x06, 0xea, 0xae, 0x3c,
    0xe3, 0x7a, 0x50, 0x74, 0x4a, 0x84, 0xfe, 0xc3,
    0x81, 0x41, 0xf8, 0x89, 0x18, 0x43, 0x67, 0x9d,
    0x87, 0x47, 0xf5, 0x8c, 0x51, 0xf6, 0x68, 0xb4,
    0xea, 0x55, 0x20, 0x2a, 0x6f, 0x80, 0xdc, 0x42,
    0x2b, 0xf3, 0x14, 0x2b, 0x1a, 0xdb, 0x0f, 0xe4,
    0x9a, 0x64, 0x84, 0xa3, 0x90, 0xa8, 0xf9, 0x8f,
    0x9d, 0x86, 0x9e, 0xd3, 0xab, 0x5a, 0x99, 0xc8,
    0xd9, 0xc3, 0x5e, 0x85, 0x0e, 0x2c, 0xb5, 0x73,
    0x30, 0x38, 0xfb, 0xe8, 0x44, 0xef, 0x5f, 0x95,
    0x1b, 0xc9, 0xd0, 0xef, 0x3c, 0x26, 0x32, 0x1e,
    0xff, 0x2d, 0xb6, 0x23, 0x7b, 0x3f, 0xd1, 0x3c,
    0x78, 0x1a, 0x0d, 0xcb, 0xe6, 0xf6, 0xd4, 0x44,
    0x99, 0x47, 0x1a, 0x9e, 0xed, 0x23, 0xb5, 0x91,
    0x6a, 0xdf, 0x53, 0x39, 0x03, 0xf8, 0x5a, 0xb1,
    0x0f, 0x1f, 0xce, 0x81, 0x11, 0xde, 0x01, 0x7a,
    0x90, 0x16, 0xc4, 0x30, 0xe8, 0x89, 0xed, 0x7b,
    0x65, 0x4b, 0xd7, 0x03, 0x36, 0xc1, 0xcf, 0xa1,
    0xa5, 0xb1, 0xe3, 0x8b, 0xe8, 0x07, 0x4d, 0xf3,
    0x23, 0x25, 0x13, 0x35, 0x27, 0xf5, 0x8c, 0x11,
    0xd3, 0xa0, 0x9a, 0xa8, 0xf5, 0x38, 0x7d, 0xce,
    0x55, 0xc2, 0x71, 0x79, 0x13, 0xc7, 0xa3, 0xda,
    0x77, 0x68, 0xc0, 0xd8, 0x10, 0xdd, 0x24, 0x8b,
    0x15, 0x59, 0xc5, 0x10, 0xe2, 0x20, 0x99, 0x8e,
    0xf0, 0x05, 0x9b, 0x31, 0x88, 0x5a, 0xe3, 0xd9,
    0x37, 0xba, 0xe2, 0xdb, 0xbf, 0x92, 0xfa, 0x66,
    0x16, 0x97, 0x47, 0xd9, 0x9d, 0x1d, 0x28, 0x7c,
    0x9d, 0x08, 0x1c, 0xc7, 0xbd, 0xd2, 0x1a, 0x6a,
    0x04, 0xf2, 0xa2, 0x1d, 0x75, 0x02, 0x14, 0x5d,
    0xc6, 0x78, 0xc8, 0xab, 0xdb, 0xf5, 0xb6, 0x82,
    0x6c, 0xb5, 0x83, 0x87, 0xac, 0x28, 0xb2, 0x55,
    0xb5, 0x9b, 0xc7, 0xc1, 0xb0, 0xb7, 0xf8, 0x4c,
    0xbc, 0x38, 0x0e, 0x8a, 0x04, 0x2a, 0x62, 0x41,
    0x6b, 0xe0, 0x84, 0x09, 0x13, 0xe9, 0xe1, 0xea,
    0xfb, 0xeb, 0x62, 0x71, 0x4b, 0x25, 0xd9, 0x55,
    0x7e, 0x97, 0x57, 0x3b, 0x20, 0x33, 0x96, 0x79,
    0xb5, 0xba, 0x2e, 0x4b, 0x58, 0xae, 0x0b, 0xc8,
    0x60, 0x93, 0x15, 0x55, 0xbe, 0xd8, 0xde, 0x65,
    0x05, 0x6c, 0xb6, 0xc5, 0x66, 0x5d, 0x5e, 0x93,
    0xf7, 0x25, 0x65, 0x98, 0x41, 0x29, 0x86, 0x0c,
    0xf2, 0xf1, 0x14, 0xa2, 0xb3, 0xbd, 0x75, 0x08,
    0x12, 0x83, 0x50, 0xda, 0x1f, 0x23, 0xbe, 0xa3,
    0x1d, 0xf4, 0x9d, 0x1d, 0xb5, 0x84, 0x4e, 0x50,
    0x38, 0x1d, 0x36, 0x48, 0x21, 0x95, 0xd1, 0xac,
    0x81, 0x99, 0x1d, 0xc1, 0x3f, 0x41, 0xe6, 0x9e,
    0x42, 0x5b, 0x0a, 0x48, 0xcc, 0x5f, 0xe0, 0x7d,
    0x3f, 0xc4, 0x6f, 0x0e, 0xfe, 0xc0, 0x2d, 0xfe,
    0x01, 0x2c, 0xd6, 0x9b, 0x5d, 0xbe, 0xba, 0x21,
    0xca, 0x79, 0xcb, 0xe3, 0x49, 0x60, 0xef, 0x68,
    0x05, 0x28, 0x9b, 0x8c, 0xc1, 0x12, 0x3e, 0xdb,
    0xc7, 0x04, 0x7e, 0xa6, 0x74, 0x29, 0xcc, 0x13,
    0xed, 0x07, 0x94, 0x81, 0xd6, 0x96, 0xaa, 0x97,
    0xaa, 0xa5, 0xc0, 0x2f, 0xb5, 0xb5, 0x2e, 0xe6,
    0xfc, 0xca, 0xfa, 0x60, 0x4d, 0x02, 0xf7, 0x19,
    0x9c, 0x5f, 0xa4, 0xe9, 0xf9, 0xf7, 0xf4, 0xc7,
    0x79, 0x9a, 0xc0, 0xb6, 0xcc, 0x58, 0xec, 0xec,
    0xe4, 0x37, 0x22, 0xfa, 0x8b, 0x53,
};

static unsigned char some_data[] = {
    0x2f, 0x2a, 0x0a, 0x20, 0x2a, 0x20, 0x54, 0x68,
    0x69, 0x73, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20,
    0x69, 0x73, 0x20, 0x66, 0x72, 0x65, 0x65, 0x20,
    0x73, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x3b, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x63, 0x61,
    0x6e, 0x20, 0x72, 0x65, 0x64, 0x69, 0x73, 0x74,
    0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x20, 0x69,
    0x74, 0x20, 0x61, 0x6e, 0x64, 0x2f, 0x6f, 0x72,
    0x20, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x0a,
    0x20, 0x2a, 0x20, 0x69, 0x74, 0x20, 0x75, 0x6e,
    0x64, 0x65, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x74, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x4e, 0x55,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x2c,
    0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e,
    0x20, 0x32, 0x2c, 0x0a, 0x20, 0x2a, 0x20, 0x61,
    0x73, 0x20, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x73,
    0x68, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x46, 0x72, 0x65, 0x65, 0x20,
    0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x20, 0x46, 0x6f, 0x75, 0x6e, 0x64, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x2e, 0x0a, 0x20, 0x2a, 0x0a,
    0x20, 0x2a, 0x20, 0x49, 0x6e, 0x20, 0x61, 0x64,
    0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74,
    0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x65,
    0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e,
    0x73, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65,
    0x20, 0x47, 0x4e, 0x55, 0x20, 0x47, 0x65, 0x6e,
    0x65, 0x72, 0x61, 0x6c, 0x20, 0x50, 0x75, 0x62,
    0x6c, 0x69, 0x63, 0x20, 0x4c, 0x69, 0x63, 0x65,
    0x6e, 0x73, 0x65, 0x2c, 0x0a, 0x20, 0x2a, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x61, 0x75, 0x74, 0x68,
    0x6f, 0x72, 0x73, 0x20, 0x67, 0x69, 0x76, 0x65,
    0x20, 0x79, 0x6f, 0x75, 0x20, 0x75, 0x6e, 0x6c,
    0x69, 0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x70,
    0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
    0x6e, 0x20, 0x74, 0x6f, 0x20, 0x6c, 0x69, 0x6e,
    0x6b, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6f,
    0x6d, 0x70, 0x69, 0x6c, 0x65, 0x64, 0x0a, 0x20,
    0x2a, 0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
    0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x69,
    0x73, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x69,
    0x6e, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6d, 0x62,
    0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73,
    0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6f, 0x74,
    0x68, 0x65, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x67,
    0x72, 0x61, 0x6d, 0x73, 0x2c, 0x0a, 0x20, 0x2a,
    0x20, 0x61, 0x6e, 0x64, 0x20, 0x74, 0x6f, 0x20,
    0x64, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75,
    0x74, 0x65, 0x20, 0x74, 0x68, 0x6f, 0x73, 0x65,
    0x20, 0x63, 0x6f, 0x6d, 0x62, 0x69, 0x6e, 0x61,
    0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x77, 0x69,
    0x74, 0x68, 0x6f, 0x75, 0x74, 0x20, 0x61, 0x6e,
    0x79, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72, 0x69,
    0x63, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x20, 0x2a,
    0x20, 0x63, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x20,
    0x66, 0x72, 0x6f, 0x6d, 0x20, 0x74, 0x68, 0x65,
    0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x66, 0x20,
    0x74, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69, 0x6c,
    0x65, 0x2e, 0x20, 0x20, 0x28, 0x54, 0x68, 0x65,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a,
    0x20, 0x2a, 0x20, 0x72, 0x65, 0x73, 0x74, 0x72,
    0x69, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20,
    0x64, 0x6f, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x79,
    0x20, 0x69, 0x6e, 0x20, 0x6f, 0x74, 0x68, 0x65,
    0x72, 0x20, 0x72, 0x65, 0x73, 0x70, 0x65, 0x63,
    0x74, 0x73, 0x3b, 0x20, 0x66, 0x6f, 0x72, 0x20,
    0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2c,
    0x20, 0x74, 0x68, 0x65, 0x79, 0x20, 0x63, 0x6f,
    0x76, 0x65, 0x72, 0x0a, 0x20, 0x2a, 0x20, 0x6d,
    0x6f, 0x64, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x2c,
    0x20, 0x61, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x73,
    0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x69, 0x6f,
    0x6e, 0x20, 0x77, 0x68, 0x65, 0x6e, 0x20, 0x6e,
    0x6f, 0x74, 0x20, 0x6c, 0x69, 0x6e, 0x6b, 0x65,
    0x64, 0x20, 0x69, 0x6e, 0x74, 0x6f, 0x0a, 0x20,
    0x2a, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x6d, 0x62,
    0x69, 0x6e, 0x65, 0x64, 0x20, 0x65, 0x78, 0x65,
    0x63, 0x75, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x2e,
    0x29, 0x0a, 0x20, 0x2a, 0x0a, 0x20, 0x2a, 0x20,
    0x54, 0x68, 0x69, 0x73, 0x20, 0x66, 0x69, 0x6c,
    0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73,
    0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x64,
    0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x68, 0x6f, 0x70, 0x65, 0x20, 0x74, 0x68, 0x61,
    0x74, 0x20, 0x69, 0x74, 0x20, 0x77, 0x69, 0x6c,
    0x6c, 0x20, 0x62, 0x65, 0x20, 0x75, 0x73, 0x65,
    0x66, 0x75, 0x6c, 0x2c, 0x20, 0x62, 0x75, 0x74,
    0x0a, 0x20, 0x2a, 0x20, 0x57, 0x49, 0x54, 0x48,
    0x4f, 0x55, 0x54, 0x20, 0x41, 0x4e, 0x59, 0x20,
    0x57, 0x41, 0x52, 0x52, 0x41, 0x4e, 0x54, 0x59,
    0x3b, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f, 0x75,
    0x74, 0x20, 0x65, 0x76, 0x65, 0x6e, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x69,
    0x65, 0x64, 0x20, 0x77, 0x61, 0x72, 0x72, 0x61,
    0x6e, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x0a, 0x20,
    0x2a, 0x20, 0x4d, 0x45, 0x52, 0x43, 0x48, 0x41,
    0x4e, 0x54, 0x41, 0x42, 0x49, 0x4c, 0x49, 0x54,
    0x59, 0x20, 0x6f, 0x72, 0x20, 0x46, 0x49, 0x54,
    0x4e, 0x45, 0x53, 0x53, 0x20, 0x46, 0x4f, 0x52,
    0x20, 0x41, 0x20, 0x50, 0x41, 0x52, 0x54, 0x49,
    0x43, 0x55, 0x4c, 0x41, 0x52, 0x20, 0x50, 0x55,
    0x52, 0x50, 0x4f, 0x53, 0x45, 0x2e, 0x20, 0x20,
    0x53, 0x65, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20,
    0x47, 0x4e, 0x55, 0x0a, 0x20, 0x2a, 0x20, 0x47,
    0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x50,
    0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4c, 0x69,
    0x63, 0x65, 0x6e, 0x73, 0x65, 0x20, 0x66, 0x6f,
    0x72, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x64,
    0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x2e, 0x0a,
    0x20, 0x2a, 0x0a, 0x20, 0x2a, 0x20, 0x59, 0x6f,
    0x75, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64,
    0x20, 0x68, 0x61, 0x76, 0x65, 0x20, 0x72, 0x65,
    0x63, 0x65, 0x69, 0x76, 0x65, 0x64, 0x20, 0x61,
    0x20, 0x63, 0x6f, 0x70, 0x79, 0x20, 0x6f, 0x66,
    0x20, 0x74, 0x68, 0x65, 0x20, 0x47, 0x4e, 0x55,
    0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c,
    0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
    0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0x0a,
    0x20, 0x2a, 0x20, 0x61, 0x6c, 0x6f, 0x6e, 0x67,
    0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x74, 0x68,
    0x69, 0x73, 0x20, 0x70, 0x72, 0x6f, 0x67, 0x72,
    0x61, 0x6d, 0x3b, 0x20, 0x73, 0x65, 0x65, 0x20,
    0x74, 0x68, 0x65, 0x20, 0x66, 0x69, 0x6c, 0x65,
    0x20, 0x43, 0x4f, 0x50, 0x59, 0x49, 0x4e, 0x47,
    0x2e, 0x20, 0x20, 0x49, 0x66, 0x20, 0x6e, 0x6f,
    0x74, 0x2c, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65,
    0x20, 0x74, 0x6f, 0x0a, 0x20, 0x2a, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x46, 0x72, 0x65, 0x65, 0x20,
    0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
    0x20, 0x46, 0x6f, 0x75, 0x6e, 0x64, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x2c, 0x20, 0x35, 0x31, 0x20,
    0x46, 0x72, 0x61, 0x6e, 0x6b, 0x6c, 0x69, 0x6e,
    0x20, 0x53, 0x74, 0x72, 0x65, 0x65, 0x74, 0x2c,
    0x20, 0x46, 0x69, 0x66, 0x74, 0x68, 0x20, 0x46,
    0x6c, 0x6f, 0x6f, 0x72, 0x2c, 0x0a, 0x20, 0x2a,
    0x20, 0x42, 0x6f, 0x73, 0x74, 0x6f, 0x6e, 0x2c,
    0x20, 0x4d, 0x41, 0x20, 0x30, 0x32, 0x31, 0x31,
    0x30, 0x2d, 0x31, 0x33, 0x30, 0x31, 0x2c, 0x20,
    0x55, 0x53, 0x41, 0x2e, 0x0a, 0x20, 0x2a, 0x2f,
    0x0a,
};

static object_data some = {
    some_bytes,
    sizeof(some_bytes),
    "fd8430bc864cfcd5f10e5590f8a447e01b942bfe",
    GIT_OID_SHA1,
    "blob",
    "test-objects/fd",
    "test-objects/fd/8430bc864cfcd5f10e5590f8a447e01b942bfe",
    some_data,
    sizeof(some_data),
};

#ifdef GIT_EXPERIMENTAL_SHA256
static object_data some_sha256 = {
    some_bytes,
    sizeof(some_bytes),
    "083c2b8b445640d171e7aa667b0b32007006f786711032ebb82931cca69cc15b",
    GIT_OID_SHA256,
    "blob",
    "test-objects/08",
    "test-objects/08/3c2b8b445640d171e7aa667b0b32007006f786711032ebb82931cca69cc15b",
    some_data,
    sizeof(some_data),
};
#endif
