/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */

/*!
 * @header DSNetwork
 * @discussion DSNetwork.h aims to provide a simple, consistent, abstracted Network library which can operate equally on *nix and Win32.  It includes the Network class (which is probably what you're looking for) and a whole pile of declarations to provide cross-platform consistency.
 */

#ifndef DS_NETWORK_H
#define DS_NETWORK_H

#include <DSCore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#ifdef _WIN32_
#define WIN32_LEAN_AND_MEAN
#include <winsock.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif


#ifndef _WIN32_
/*
Windows defines a few types, which UNIX has equivalent (different) types
for.  These set up aliases on unix to the Windows version.  This should
make code easier to read.
*/

/*!
 * @typedef HOSTENT
 * @discussion Create an alias to struct hostent for platform abstraction.
 */
typedef struct hostent HOSTENT;

/*!
 * @typedef SOCKADDR
 * @discussion Create an alias to struct sockaddr for platform abstraction.
 */
typedef struct sockaddr SOCKADDR;

/*!
 * @typedef SOCKADDR_IN
 * @discussion Create an alias to struct sockaddr_in for platform abstraction.
 */
typedef struct sockaddr_in SOCKADDR_IN;

/*!
 * @typedef SOCKET
 * @discussion Create an alias to int for platform abstraction.
 */
typedef int SOCKET;

/*!
 * @constant INVALID_SOCKET
 * @discussion Create an INVALID_SOCKET for platform abstraction.
 */
const SOCKET INVALID_SOCKET = -1;


/*!
 * @defined closesocket
 * @discussion Create an alias to close for platform abstraction.
 */
#define closesocket(x) close(x)
#endif /* !_WIN32_ */

#ifdef _WIN32_
/*!
 * @defined sleep
 * @discussion Create an alias to Sleep for platform abstraction.
 */
#define sleep(x) Sleep(x*1000)
#endif /* _WIN32_ */

/*!
 * @enum BOOLX
 * @discussion In networking, it is possible for operations not to either succeed or fail, but for them to be not ready to complete just yet.  This is a tri-state type to convey success, failure, and pending.
 * @constant NO The operation has failed.
 * @constant YES The operation has succeeded.
 * @constant NOTYET The operation can't succeed yet, may never succeed, may fail, but hasn't failed yet.
 */
typedef enum
{ NO, YES, NOTYET }
BOOLX;

/*!
 * @class DSNetwork
 * @discussion This class should provide an abstraction for common network functions, and take care of some lower-level issues, so that higher components can build on a more solid networking foundation.
 */
class EXPSYM DSNetwork {
	private:
		SOCKET sock;
	public:
		/*!
		 * @function acceptIncoming
		 * @discussion Accept an incoming connection onto a fresh network connection.
		 * @result A pointer to the newly created Network connection.
		 */
		DSNetwork * acceptIncoming();
		DEPRECATED DSNetwork * Accept();

		/*!
		 * @function connectToHost
		 * @discussion Connect to the specified hostname.  It can process hostnames in the form hostname:port.  Note that this function uses gethostbyname to resolve hostnames, and is therefore not threadsafe.  You should protect accesses to this function via mutexes, and ensure that gethostbyname is either not used elsewhere or is protected with the same locks.
		 * @param host The name of the host to connect to, can include :port syntax.
		 * @param nPort The port to connect to, if no port is specified with the hostname.
		 * @result TRUE if the connection succeeds, FALSE if it does not.
		 */
		BOOL connectToHost(char * host, unsigned short nPort);
		DEPRECATED BOOL ConnectToHost(char * host, unsigned short nPort);

		/*!
		 * @function getLineFromSocket
		 * @discussion Retrieves an incoming line from the network, if it's there.
		 * @param buffer Specifies the buffer where the line will be copied to.
		 * @param size Specifies the maximum size of the buffer.
		 * @result Returns YES if the operation succeeded, NO if an error occurred, and NOTYET if a complete line isn't available for reading...yet.
		 */
		BOOLX getLineFromSocket(char * buffer, int size);
		DEPRECATED BOOLX GetLineFromSocket(char * buffer, int size);
		/*!
		 * @function getLineFromSocketWait
		 * @discussion Retrieves an incoming line from the network, and wait till it arrives.
		 * @param buffer Specifies the buffer where the line will be copied to.
		 * @param size Specifies the maximum size of the buffer.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL getLineFromSocketWait(char * buffer, int size);
		DEPRECATED BOOL GetLineFromSocketWait(char * buffer, int size);

		/*!
		 * @function getChar
		 * @discussion Retrieves a single incoming character.
		 * @param ch Specifies where to place the character.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL getChar(char * ch);
		DEPRECATED BOOL GetChar(char * ch);

		/*!
		 * @function getChar
		 * @discussion Retrieves a single incoming character.
		 * @param ch Specifies where to place the character.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL getChar(unsigned char * ch);
		DEPRECATED BOOL GetChar(unsigned char * ch);

		/*!
		 * @function getSlab
		 * @discussion Retrieves an incoming pile of data from the network, and wait till it arrives.
		 * @param buffer Specifies the buffer where the line will be copied to.
		 * @param nSize Specifies the size to be read.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL getSlab(char * buffer,int nSize);
		DEPRECATED BOOL GetSlab(char * buffer,int nSize);

		/*!
		 * @function getSocket
		 * @discussion Retrieves the socket used for low-level operations.  Use this with great care.  Specifically, only use this only if you need it.
		 * @result The socket in use.
		 */
		SOCKET getSocket();
		DEPRECATED SOCKET GetSocket();

		/*!
		 * @function linger
		 * @discussion Switches the network into 'Linger' mode where all remaining data is sent out before a connection is really closed.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL linger();
		DEPRECATED BOOL Linger();

		/*!
		 * @function sendData
		 * @discussion Sends a pile of data into the Network.
		 * @param szData The data to send to the Network.
		 * @param nSize The size of the data being sent to the Network; if ommitted, the data will be assumed to be a NULL-terminated string and length determined accordingly.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL sendData(const char * szData, int nSize=-1);
		DEPRECATED BOOL Send(const char * szData, int nSize=-1);
		DEPRECATED BOOL Send(char * szData, int nSize=-1);

		/*!
		 * @function startServer
		 * @discussion Starts the server listening for incoming connections to 1452.
		 * @param nPort The port that the server should listen on.
		 * @param nIP The IP address to listen on.  If ommitted, will listen on all IP addresses.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL startServer(unsigned short nPort, unsigned long nIP=0);
		DEPRECATED BOOL StartServer(unsigned short nPort, unsigned long nIP=0);

		/*!
		 * @function terminate
		 * @discussion Closes this Network connection.
		 * @result Returns TRUE if the operation succeeded, and FALSE if an error occurred.
		 */
		BOOL terminate();
		DEPRECATED BOOL Terminate();

		/*!
		 * @function waitForData
		 * @discussion Waits for data to arrive on the connection, either indefinitely or for a period of time, returning TRUE if there is data available to be read.
		 * @param sec The number of seconds to wait for.  If this value is negative, wait indefinitely.
		 * @param usec The number of microseconds to wait for.  This value is ignored if sec is negative.
		 * @result Returns TRUE if there is data available to be read, FALSE if there is not.
		 */
		BOOL waitForData(int sec, int usec);

		/*!
		 * @function waitForDataEx
		 * @discussion Waits for data to arrive on the connection, either indefinitely or for a period of time, returning YES if there is data available to be read, NOTYET if there is no data currently, and NO if an error has occurred on the connection.
		 * @param sec The number of seconds to wait for.  If this value is negative, wait indefinitely.
		 * @param usec The number of microseconds to wait for.  This value is ignored if sec is negative.
		 * @result Returns YES if there is data available to be read, NOTYET if there is no data currently, and NO if an error has occurred on the connection.
		 */
		BOOLX waitForDataEx(int sec, int usec);
};

#endif

