/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#ifndef DS_HASHTABLE_H
#define DS_HASHTABLE_H

#include <DSCore.h>
#include <DSList.h>
/*!
	@header DSHashTable
	@discussion HashTable implements an efficient in-memory HashTable structure that can be used on both string and integer keys, with pointer and integer values.  You should use this HashTable class for data that needs time-critical access.
*/

/*!
	@class DSHashTable
	@abstract Single-threaded generic HashTable implementation.
	@discussion HashTable implements an efficient in-memory HashTable structure that can be used on both string and integer keys, with pointer and integer values.  You should use this HashTable class for data that needs time-critical access.
*/
class EXPSYM DSHashTable
{
	private:
		unsigned int buckets;
		DSList** bucket;
		unsigned int primeA;
		unsigned int primeB;

	public:
		/*!
			@function DSHashTable
			@discussion Create a HashTable.
			@param newBuckets Specify the number of buckets to be used in the HashTable.  The more buckets, the more memory is required; but the faster data access will be on large datasets.  You may wish to increase the default for areas involving large quantities of data.
			@param newPrimeA Specify a large prime number to be used in hashing.  You only need to do this if it's important to you to have a different hashing algorithm between two HashTables.
			@param newPrimeB Specify a large prime number to be used in hashing.  You only need to do this if it's important to you to have a different hashing algorithm between two HashTables.
		*/
		DSHashTable(unsigned int newBuckets=100,unsigned int newPrimeA=89258459,unsigned int newPrimeB=252584539);

		/*!
			@function ~DSHashTable
			@discussion Destroy a HashTable.
		*/
		~DSHashTable();

		/*!
			@function close
			@discussion Purges all data from the HashTable.  Leaves the HashTable intact for subsequent use if desired.
		*/
		void         close();

		DEPRECATED void         delValueInt(unsigned int Key);
		DEPRECATED void         delValueString(const char* Key);

		/*!
			@function delItem
			@discussion Removes one or more entries from the HashTable that have the specified numeric key.
			@param Key A numeric key corresponding to the item(s) to be removed from the HashTable.
		*/
		void         delItem(unsigned int Key);

		/*!
			@function delItem
			@discussion Removes one or more entries from the HashTable that have the specified string key.
			@param Key A string key corresponding to the item(s) to be removed from the HashTable.
		*/
		void         delItem(const char* Key);

		/*!
			@function forEach
			@discussion Iterates through the HashTable calling a function for each entry found.  This can be useful for doing bulk operations on data in a HashTable; but if you find yourself calling this frequently, consider using a different data structure.
			@param callback A function to be called for each entry found in the HashTable.  The function will be given a pointer to the key, a pointer to the value, and extra, discussed below.
			@param extra A pointer to any additional data to be passed to the function for every item encountered.  Can be NULL.
		*/
		void         forEach(void (*callback)(char *, char *, void *), void * extra);
		/*!
			@function getBuckets
			@discussion Returns the number of buckets in the HashTable.
			@result The number of buckets in this HashTable.
		*/
		unsigned int getBuckets();

		/*!
			@function getList
			@discussion Returns a list of data corresponding to the data in the entire HashTable, in no particular order.
			@result A list of items in the HashTable.  This list is dynamically created, remember to delete it when you're finished.)
		*/
		DSList*        getList();

		/*!
			@function getList
			@discussion Returns a list of data at a particular bucket.  Please avoid using this function, it exposes the internal workings of the HashTable.  If you need bulk access to data, consider using forEach.
			@param bucket Specifies the bucket number to be retrieved.
			@result A list of items at this bucket (or NULL, if there are no entries in this bucket.)
		*/
		DSList*        getList(unsigned int bucket);
		DEPRECATED DSListElement* getElementInt(unsigned int Key);
		DEPRECATED DSListElement* getElementString(const char* Key);

		/*!
			@function getElement
			@discussion Returns an element matching a specified Key, if such an element exists.  Note that this returns a pointer to the raw ListElement object, and this can be used to modify details relating to this particular item.  For this reason, use with caution.
			@param Key A numeric key matching the element requested.
			@result The ListElement requested, or NULL, if the element could not be found.
		*/
		DSListElement* getElement(unsigned int Key);

		/*!
			@function getElement
			@discussion Returns an element matching a specified Key, if such an element exists.  Note that this returns a pointer to the raw ListElement object, and this can be used to modify details relating to this particular item.  For this reason, use with caution.
			@param Key A string key matching the element requested.
			@result The ListElement requested, or NULL, if the element could not be found.
		*/
		DSListElement* getElement(const char* Key);

		DEPRECATED void*        getValueString(const char* Key);
		DEPRECATED void*        getValueInt(unsigned int Key);

		/*!
			@function getPtrValue
			@discussion Returns the data matching a specified Key, if such an element exists.  Note that this returns a pointer to the data, and this can be used to modify details relating to this particular item.  For this reason, use with caution.
			@param Key A string key matching the element requested.
			@result A pointer to the element's data, or NULL, if the element could not be found.
		*/
		void*        getPtrValue(const char* Key);

		/*!
			@function getPtrValue
			@discussion Returns the data matching a specified Key, if such an element exists.  Note that this returns a pointer to the data, and this can be used to modify details relating to this particular item.  For this reason, use with caution.
			@param Key A numeric key matching the element requested.
			@result A pointer to the element's data, or NULL, if the element could not be found.
		*/
		void*        getPtrValue(unsigned int Key);

		DEPRECATED unsigned int getIntString(const char* Key);
		DEPRECATED unsigned int getIntInt(unsigned int Key);

		/*!
			@function getNumericValue
			@discussion Returns the data matching a specified Key, if such an element exists.
			@param Key A string key matching the element requested.
			@result The element's data, or NULL, if the element could not be found.
		*/
		unsigned int getNumericValue(const char* Key);

		/*!
			@function getNumericValue
			@discussion Returns the data matching a specified Key, if such an element exists.
			@param Key A numeric key matching the element requested.
			@result The element's data, or NULL, if the element could not be found.
		*/
		unsigned int getNumericValue(unsigned int Key);

		/*!
			@function hash
			@discussion Hashes the key to generate the bucket that the specified Key would be located in, if it were to be entered.
			@param Key A string key.
			@result The bucket which the key corresponds to.
		*/
		unsigned int hash(unsigned char * Key);

		/*!
			@function hash
			@discussion Hashes the key to generate the bucket that the specified Key would be located in, if it were to be entered.
			@param Key A numeric key.
			@result The bucket which the key corresponds to.
		*/
		unsigned int hash(unsigned int value);

		/*!
			@function init
			@discussion Create a HashTable.  This function is only provided to allow initiation of a HashTable that has been created by malloc.  It should not be used in any other way.  The use of malloc is strongly discouraged and new should be used instead.  This function may be removed at some point in the future.
			@param newBuckets Specify the number of buckets to be used in the HashTable.  The more buckets, the more memory is required; but the faster data access will be on large datasets.  You may wish to increase the default for areas involving large quantities of data.
			@param newPrimeA Specify a large prime number to be used in hashing.  You only need to do this if it's important to you to have a different hashing algorithm between two HashTables.
			@param newPrimeB Specify a large prime number to be used in hashing.  You only need to do this if it's important to you to have a different hashing algorithm between two HashTables.
		*/
		void         init(unsigned int newBuckets=100,unsigned int newPrimeA=89258459,unsigned int newPrimeB=252584539);

		/*!
			@function insert
			@discussion Inserts the specified data into the HashTable.
			@param Key A string key.  Note that this pointer will be inserted into the HashTable.  The HashTable will not copy the data associated with it, so unless the data is permanent, you need to allocate your own memory and set the value of cleanup accordingly.
			@param value A pointer value.  Note that this pointer will be inserted into the HashTable.  The HashTable will not copy the data associated with it, so unless the data is permanent, you need to allocate your own memory and set the value of cleanup accordingly.
			@param cleanup A bitwise combination of flags telling the HashTable what memory it should deallocate if the item is removed.  The default is to perform no cleanup, and to leave it to the application.  The permissable flags are documented in the ListElement class.
		*/
		void         insert(char * Key, void * value, int cleanup=0);

		/*!
			@function insert
			@discussion Inserts the specified data into the HashTable.
			@param Key A numeric key.
			@param value A pointer value.  Note that this pointer will be inserted into the HashTable.  The HashTable will not copy the data associated with it, so unless the data is permanent, you need to allocate your own memory and set the value of cleanup accordingly.
			@param cleanup A bitwise combination of flags telling the HashTable what memory it should deallocate if the item is removed.  The default is to perform no cleanup, and to leave it to the application.  The permissable flags are documented in the ListElement class.
		*/
		void         insert(unsigned int Key, void * value, int cleanup=0);

		/*!
			@function insert
			@discussion Inserts the specified data into the HashTable.
			@param Key A string key.  Note that this pointer will be inserted into the HashTable.  The HashTable will not copy the data associated with it, so unless the data is permanent, you need to allocate your own memory and set the value of cleanup accordingly.
			@param value A numeric value.
			@param cleanup A bitwise combination of flags telling the HashTable what memory it should deallocate if the item is removed.  The default is to perform no cleanup, and to leave it to the application.  The permissable flags are documented in the ListElement class.
		*/
		void         insert(char * Key, unsigned int value, int cleanup=0);

		/*!
			@function insert
			@discussion Inserts the specified data into the HashTable.
			@param Key A numeric key.
			@param value A numeric value.
			@param cleanup A bitwise combination of flags telling the HashTable what memory it should deallocate if the item is removed.  This value is not relevant for this form of insert, and should be set to 0.
		*/
		void         insert(unsigned int Key, unsigned int value, int cleanup=0);

		/*!
			@function occurenceCount
			@discussion Counts the number of times the specified Key occurs in the HashTable.
			@param Key A string Key.
			@result The number of times the specified Key occurs in the HashTable.
		*/
		unsigned int occurenceCount(const char * key);

		/*!
			@function print
			@discussion Dumps the contents of the HashTable to stdout for debugging.  Should be removed soon.
		*/
		void         print();

		/*!
		 * @function setValue
		 * @discussion Attempts to modify the data of an item matching the specified key, if one exists.  This will not modify the cleanup flags of the element.
		 * @param Key A string key matching the element requested.
		 * @param data A pointer to the data to be used.  Note that this pointer is used directly and will be cleaned up in accordance with the cleanup value of the element.
		 * @result TRUE if the value was successfully updated; FALSE if the operation could not be completed.
		*/
		BOOL         setValue(const char* Key, void * data);

		/*!
		 * @function setValue
		 * @discussion Attempts to modify the data of an item matching the specified key, if one exists.  This will not modify the cleanup flags of the element.
		 * @param Key A string key matching the element requested.
		 * @param data A piece of numeric data to be used.
		 * @result TRUE if the value was successfully updated; FALSE if the operation could not be completed.
		*/
		BOOL         setValue(const char* Key, unsigned int data);

		/*!
		 * @function setValue
		 * @discussion Attempts to modify the data of an item matching the specified key, if one exists.  This will not modify the cleanup flags of the element.
		 * @param Key A numeric key matching the element requested.
		 * @param data A pointer to the data to be used.  Note that this pointer is used directly and will be cleaned up in accordance with the cleanup value of the element.
		 * @result TRUE if the value was successfully updated; FALSE if the operation could not be completed.
		*/
		BOOL         setValue(unsigned int Key, void * data);

		/*!
		 * @function setValue
		 * @discussion Attempts to modify the data of an item matching the specified key, if one exists.  This will not modify the cleanup flags of the element.
		 * @param Key A numeric key matching the element requested.
		 * @param data A piece of numeric data to be used.
		 * @result TRUE if the value was successfully updated; FALSE if the operation could not be completed.
		*/
		BOOL         setValue(unsigned int Key, unsigned int data);

};

#endif
