/* LIBDS
 * =====
 * This software is Copyright (c) 2002-04 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */

#ifndef WIN32_NO_CONFIG_H
#include "../config.h"
#else
#include "../winconf.h"
#endif
#include "DSConfigFile.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

/*
 * Constructs a new DSConfigFile object.
 */
DSConfigFile::DSConfigFile()
{
	sections = new DSHashTable(100);
}

/*
 * Destroys a DSConfigFile object.
 */
DSConfigFile::~DSConfigFile()
{
	close();
	delete sections;
}

/*
 * Closes a DSConfigFile.  All data contained in the DSConfigFile is dropped.
 */
void DSConfigFile::close()
{
	int buckets, i;
	DSList *bucklist;
	DSListElement *current;
	DSHashTable *tbl;

	buckets = sections->getBuckets();
	for (i = 0; i < buckets; i++) {
		bucklist = sections->getList(i);
		if (bucklist) {
			for (current = bucklist->getHead();
			     current != NULL; current = current->getNext()) {
				tbl = (DSHashTable *) current->getDataPtr();
				tbl->close();
				delete tbl;
			}
		}
	}
	sections->close();
}

/*
 * Callback driver.  Call this method if you want a callback function
 * executed for every entry in the specified section of the DSConfigFile.
 * Takes in the section, callback routine, and custom data passed unchanged
 * into your callback routines.
 */
void DSConfigFile::forEach(char *section,
			   void (*callback) (char *, char *, void *),
			   void *extra)
{
	DSHashTable *tbl;
	if (section) {
		tbl = getSection(section);
	} else {
		tbl = sections;
	}
	if (tbl)
		tbl->forEach(callback, extra);
}

/*
 * Load a file from disk into the DSConfigFile object.
 */
int DSConfigFile::loadFile(const char *filename)
{
	int retval;
	FILE *fp;
	char file_line[2048];
	DSHashTable *currentSection;
	char *a;
	char *key;
	char *value;
	currentSection = NULL;
	fp = fopen(filename, "r");
	if (fp) {
		while (!feof(fp)) {
			retval = fscanf(fp, "%[^\n]%*c", file_line);
			if (retval <= 0)
				fscanf(fp, "%*c");
			else {

				a = strchr(file_line, '\n');
				if (a)
					*a = '\0';
				a = strchr(file_line, '\r');
				if (a)
					*a = '\0';

				a = strchr(file_line, '#');
				if (a)
					*a = '\0';

				if (file_line[0] != '\0') {

					if (file_line[0] == '[') {
						/* A new section has been encountered.  Better
						 * make room and initialise it.*/

						file_line[strlen(file_line) -
							  1] = '\0';
						key = (char *)
							malloc(strlen
							       (file_line));
						strcpy(key, &file_line[1]);
						currentSection =
							new DSHashTable(50);
						sections->insert(key,
								 currentSection,
								 DSListElement::
								 CLEANUP_KEY_FREE);
					} else {
						/* Just an ordinary entry */
						a = strchr(file_line, '=');
						if (currentSection) {
							int flags = 0;
							flags = DSListElement::CLEANUP_KEY_FREE;
							if (a) {
								/* Entry=value */
								*a = '\0';
								a++;
								value = (char
									 *)
									malloc
									(strlen
									 (a)
									 + 1);
								strcpy(value,
								       a);
								flags = flags
									|
									DSListElement::
									CLEANUP_VALUE_FREE;
							} else {
								/* Entry without value */
								value = NULL;
							}
							key = (char *)
								malloc(strlen
								       (file_line)
								       + 1);
							strcpy(key,
							       file_line);
							currentSection->
								insert(key,
								       value,
								       flags);
						}
					}
				}
			}
		}
		fclose(fp);
		return (1);
	}
	return (0);
}

/*
 * Save the contents of the DSConfigFile object to a file.
 */
int DSConfigFile::saveFile(const char *filename)
{
	FILE *fp;
	DSHashTable *currentSection;
	currentSection = NULL;
	int buckets, i;
	DSList *list;
	DSListElement *le;
#ifdef _WIN32_
	fp = fopen(filename, "wb");
#else
	fp = fopen(filename, "w");
#endif
	if (fp) {
		buckets = sections->getBuckets();
		for (i = 0; i < buckets; i++) {
			list = sections->getList(i);
			if (list) {
				for (le = list->getHead(); le != NULL;
				     le = le->getNext()) {
					fprintf(fp, "\n[%s]\n",
						le->getKeyString());
					currentSection =
						(DSHashTable *) le->
						getDataPtr();
					printkeys(fp, currentSection);
				}
			}
		}
		fclose(fp);
		return (1);
	}
	return (0);
}

/*
 * Returns a POINTER to the value from this section/key.
 */
char *DSConfigFile::getValue(const char *section, const char *key,
			     char *defval)
{
	DSHashTable *currentSection;
	char *hashreports;
	currentSection = (DSHashTable *) sections->getPtrValue(section);
	if (!currentSection)
		return defval;
	hashreports = (char *) currentSection->getPtrValue(key);
	if (hashreports)
		return hashreports;
	return defval;
}

/*
 * Returns an unsigned int from this section/key.
 */
unsigned int
	DSConfigFile::getValueInt(const char *section, const char *key,
				  unsigned int defval)
{
	DSHashTable *currentSection;
	char *hashreports;
	currentSection = (DSHashTable *) sections->getPtrValue(section);
	if (!currentSection)
		return defval;
	hashreports = (char *) currentSection->getPtrValue(key);
	if (hashreports) {
		unsigned int tmp;
		char *endptr;
		tmp = strtoul(hashreports, &endptr, 10);
		if ((*hashreports) != '\0' && (*endptr) == '\0')
			return tmp;
	}
	return defval;
}

/*
 * Returns nonzero if the specified section/key exists.
 */
int DSConfigFile::isValue(const char *section, const char *key)
{
	DSHashTable *currentSection;
	currentSection = (DSHashTable *) sections->getPtrValue(section);
	if (!currentSection)
		return 0;
	return (currentSection->occurenceCount(key));

}

/*
 * Returns the entire DSHashTable representing a section.  Avoid using this,
 * please.
 */
DSHashTable *DSConfigFile::getSection(char *section)
{
	return ((DSHashTable *) sections->getPtrValue(section));
}

/*
 * Set a section/key to value.
 */
void DSConfigFile::setValue(const char *section, char *key, char *value)
{
	char *tmp;
	DSHashTable *currentSection;
	int flags;
	delValue(section, key);
	flags = DSListElement::CLEANUP_KEY_FREE;
	currentSection = (DSHashTable *) sections->getPtrValue(section);
	if (!currentSection) {
		tmp = (char *) malloc(strlen(section) + 1);
		strcpy(tmp, section);
		currentSection = new DSHashTable(50);
		sections->insert(tmp, currentSection,
				 DSListElement::CLEANUP_KEY_FREE);
	}
	if (value)
		flags |= DSListElement::CLEANUP_VALUE_FREE;
	currentSection->insert(key, value, flags);
}

/*
 * Delete a section/key.
 */
void DSConfigFile::delValue(const char *section, const char *key)
{
	DSHashTable *currentSection;
	currentSection = (DSHashTable *) sections->getPtrValue(section);
	if (currentSection) {
		currentSection->delItem(key);
	}
}


/*
 * Should print all the data in the specified section to fp.
 */
void DSConfigFile::printkeys(FILE * fp, DSHashTable * currentSection)
{
	int buckets, i;
	DSList *list;
	DSListElement *le;
	buckets = currentSection->getBuckets();
	for (i = 0; i < buckets; i++) {
		list = currentSection->getList(i);
		if (list) {
			for (le = list->getHead(); le != NULL;
			     le = le->getNext()) {
				fprintf(fp, "%s", le->getKeyString());
				if (le->getDataPtr())
					fprintf(fp, "=%s",
						(char *) le->getDataPtr());
				fprintf(fp, "\n");
			}
		}
	}
}
