/*
 * Library to support cross-platform C locale functions
 *
 * Copyright (C) 2010-2015, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#if !defined( _LIBCLOCALE_H )
#define _LIBCLOCALE_H

#include <libclocale/definitions.h>
#include <libclocale/error.h>
#include <libclocale/extern.h>
#include <libclocale/features.h>
#include <libclocale/types.h>

#if defined( __cplusplus )
extern "C" {
#endif

/* -------------------------------------------------------------------------
 * Support functions
 * ------------------------------------------------------------------------- */

/* Returns the library version as a string
 */
LIBCLOCALE_EXTERN \
const char *libclocale_get_version(
             void );

/* TODO: WARNING this function is likely to change */
/* Initializes the locale values
 * Returns 1 if successful or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_initialize(
     const char *domain_name,
     libclocale_error_t **error );

/* -------------------------------------------------------------------------
 * Codepage functions
 * ------------------------------------------------------------------------- */

LIBCLOCALE_EXTERN \
int libclocale_codepage;

/* Retrieves the narrow system string codepage
 * A value of 0 represents no codepage, UTF-8 encoding is used instead
 * Returns 1 if successful or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_codepage_get(
     int *codepage,
     libclocale_error_t **error );

/* Sets the narrow system string codepage
 * A value of 0 represents no codepage, UTF-8 encoding is used instead
 * Returns 1 if successful or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_codepage_set(
     int codepage,
     libclocale_error_t **error );

/* Copies the codepage from a string
 * Returns 1 if successful, 0 if unsupported value or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_codepage_copy_from_string(
     int *codepage,
     const char *string,
     size_t string_length,
     uint32_t feature_flags,
     libclocale_error_t **error );

#if defined( LIBCLOCALE_HAVE_WIDE_CHARACTER_TYPE )

/* Copies the codepage from a string
 * Returns 1 if successful, 0 if unsupported value or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_codepage_copy_from_string_wide(
     int *codepage,
     const wchar_t *string,
     size_t string_length,
     uint32_t feature_flags,
     libclocale_error_t **error );

#endif /* defined( LIBCLOCALE_HAVE_WIDE_CHARACTER_TYPE ) */

/* -------------------------------------------------------------------------
 * Locale functions
 * ------------------------------------------------------------------------- */

/* Retrieves the codepage for the locale character set
 * The codepage is set to 0 if the character set is UTF-8
 * and will default to LIBCLOCALE_CODEPAGE_ASCII the codepage cannot be determined
 * Returns 1 if success, 0 if no codepage could be found or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_locale_get_codepage(
     int *codepage,
     libclocale_error_t **error );

/* Retrieves the decimal point
 * Returns 1 if successful or -1 on error
 */
LIBCLOCALE_EXTERN \
int libclocale_locale_get_decimal_point(
     int *decimal_point,
     libclocale_error_t **error );

#if defined( __cplusplus )
}
#endif

#endif

