/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2006  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include <netdb.h>
#include <sys/socket.h>

#include "../Include/Archive.h"
#include "../Include/BufferWriter.h"
#include "../Include/DataChannel.h"
#include "../Include/DataRepository.h"
#include "../Include/FetchStatus.h"
#include "../Include/URI.h"

#include "Common.h"
#include "LineBuffer.h"

class BlockingHTTPChannel : public Arxx::DataChannel
{
public:
	BlockingHTTPChannel(const Arxx::URI & URI) :
		Arxx::DataChannel(URI)
	{
	}
	
	virtual bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus)
	{
		int iSocket(socket(PF_INET, SOCK_STREAM, 0));
		
		if(iSocket == -1)
		{
			FetchStatus = Arxx::SYSTEMERROR;
			
			return true;
		}
		FetchStatus = Arxx::RESOLVINGHOST;
		
		struct addrinfo Hints;
		
		Hints.ai_flags = 0;
		Hints.ai_family = AF_INET;
		Hints.ai_socktype = SOCK_STREAM;
		Hints.ai_protocol = 0;
		
		struct addrinfo * pAddrInfos;
		
		if(getaddrinfo(URI.sGetAuthority().c_str(), URI.sGetScheme().c_str(), &Hints, &pAddrInfos) != 0)
		{
			FetchStatus = Arxx::HOSTUNREACHABLE;
			
			return true;
		}
		
		struct addrinfo * pAddrInfo(pAddrInfos);
		
		if(pAddrInfo != 0)
		{
			FetchStatus = Arxx::CONNECTING;
			if(connect(iSocket, pAddrInfo->ai_addr, pAddrInfo->ai_addrlen) != 0)
			{
				FetchStatus = Arxx::SERVICEUNAVAILABLE;
				
				return true;
			}
			freeaddrinfo(pAddrInfos);
			
			std::string Query;
			
			Query = "GET " + URI.sGetPath() + " HTTP/1.0";
			std::cout << g_sYellow << "\tSENDING" << g_sWhite << ": \"" << g_sGreen << Query << g_sWhite << "\"\n" << std::endl;
			Query += "\n\n";
			FetchStatus = Arxx::REQUESTING;
			write(iSocket, Query.c_str(), Query.length());
			fsync(iSocket);
			
			std::string ResponseCode;
			char * pBuffer(new char[301]);
			bool Quit(false);
			size_t stRead;
			Arxx::BufferWriter BufferWriter(Buffer);
			bool InHeader(true);
			LineBuffer LineBuffer;
			
			FetchStatus = Arxx::TRANSFERING;
			while((Quit == false) && ((stRead = read(iSocket, pBuffer, 300)) != 0))
			{
				if(stRead > 0)
				{
					pBuffer[stRead] = 0;
					if(InHeader == true)
					{
						LineBuffer.PushString(pBuffer);
						while((LineBuffer.IsEmpty() == false) && (InHeader == true) && (Quit == false))
						{
							std::string HeaderLine(LineBuffer.PopLine());
							
							if(HeaderLine.empty() == true)
							{
								InHeader = false;
								HeaderLine = LineBuffer.PopAll();
								BufferWriter.vWrite(HeaderLine.length(), reinterpret_cast< const unsigned char * >(HeaderLine.c_str()));
							}
							else
							{
								if(HeaderLine.substr(0, 4) == "HTTP")
								{
									ResponseCode = HeaderLine.substr(HeaderLine.find(' ') + 1, 3);
									if(ResponseCode == "404")
									{
										Quit = true;
									}
								}
							}
						}
					}
					else
					{
						BufferWriter.vWrite(stRead, const_cast< const unsigned char * >(reinterpret_cast< unsigned char * >(pBuffer)));
					}
				}
			}
			FetchStatus = Arxx::DISCONNECTING;
			delete[] pBuffer;
			close(iSocket);
			if(ResponseCode == "200")
			{
				FetchStatus = Arxx::FETCHED;
			}
			else if(ResponseCode == "404")
			{
				FetchStatus = Arxx::DATANOTFOUND;
			}
		}
		else
		{
			FetchStatus = Arxx::HOSTUNREACHABLE;
			
			return true;
		}
		
		return true;
	}
};

int main(int argc, char ** argv)
{
	Arxx::URI URI;
	BlockingHTTPChannel BlockingHTTPChannel("http:");
	
	Arxx::Repository.bRegisterDataChannel(&BlockingHTTPChannel);
	
	StartTest("requesting a document on an non-existent host using a blocking http data channel.");
	{
		URI = "http://www.foofoofoofoo.com/";
		std::cout << g_sYellow << "\tQUERYING" << g_sWhite << ": \"" << g_sGreen << URI << g_sWhite << "\"" << std::endl;
		
		Arxx::Item Item;
		
		Item.vSetExternal(URI);
		EqualTest(Item.bFetch(), true);
		EqualTest(Item.bIsFetched(), false);
		EqualTest(Item.stGetLength(), static_cast< Arxx::u4byte >(0));
	}
	EndTest();
	
	StartTest("requesting the document using a blocking http data channel.");
	{
		URI = "http://www.worldforge.org/";
		std::cout << g_sYellow << "\tQUERYING" << g_sWhite << ": \"" << g_sGreen << URI << g_sWhite << "\"" << std::endl;
		
		Arxx::Item Item;
		
		Item.vSetExternal(URI);
		EqualTest(Item.bFetch(), true);
		EqualTest(Item.bIsFetched(), true);
		UnequalTest(Item.stGetLength(), static_cast< Arxx::u4byte >(0));
	}
	EndTest();
	
	Arxx::Repository.bUnregisterDataChannel(&BlockingHTTPChannel);
	
	return iEndTests();
}
