/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "../Include/URI.h"

#include <iostream>


///////////////////////////////////////////////////////////////////////////////////////////////////
//  Arxx::URI                                                                                     //
///////////////////////////////////////////////////////////////////////////////////////////////////

Arxx::URI::URI(void) :
	m_bIsURIValid(false),
	m_bArePartsValid(false),
	m_bIsQueryValid(false),
	m_bAreQueryPartsValid(false)
{
}

Arxx::URI::URI(const std::string & sURI) :
	m_sURI(sURI),
	m_bIsURIValid(true),
	m_bArePartsValid(false),
	m_bIsQueryValid(false),
	m_bAreQueryPartsValid(false)
{
}

Arxx::URI::URI(const char * pcURI) :
	m_sURI(pcURI),
	m_bIsURIValid(true),
	m_bArePartsValid(false),
	m_bIsQueryValid(false),
	m_bAreQueryPartsValid(false)
{
}

bool Arxx::URI::bIsValid(void) const
{
	return ((m_bIsURIValid == true) || ((m_bArePartsValid == true) && ((m_bIsQueryValid == true) || (m_bAreQueryPartsValid == true))));
}

const std::string & Arxx::URI::sGetURI(void) const
{
	if(m_bIsURIValid == false)
	{
		vConcatenate();
	}
	
	return m_sURI;
}

const std::string & Arxx::URI::sGetScheme(void) const
{
	if(m_bArePartsValid == false)
	{
		vPartition();
	}
	
	return m_sScheme;
}

const std::string & Arxx::URI::sGetAuthority(void) const
{
	if(m_bArePartsValid == false)
	{
		vPartition();
	}
	
	return m_sAuthority;
}

const std::string & Arxx::URI::sGetPath(void) const
{
	if(m_bArePartsValid == false)
	{
		vPartition();
	}
	
	return m_sPath;
}

const std::string & Arxx::URI::sGetQuery(void) const
{
	if(m_bArePartsValid == false)
	{
		vPartition();
	}
	else
	{
		// in 'else' block because after vPartition m_bIsQueryValid is true anyway
		if((m_bIsQueryValid == false) && (m_bAreQueryPartsValid == true))
		{
			vConcatenateQuery();
		}
	}
	
	return m_sQuery;
}

const std::string & Arxx::URI::sGetParameter(const std::string & sName) const
{
	static std::string sEmpty("");
	
	if(m_bAreQueryPartsValid == false)
	{
		if(m_bIsQueryValid == false)
		{
			sGetQuery();
		}
		vPartitionQuery();
	}
	
	std::map< std::string, std::string >::const_iterator iParameter(m_Query.find(sName));
	
	if(iParameter == m_Query.end())
	{
		return sEmpty;
	}
	else
	{
		return iParameter->second;
	}
}

const std::string & Arxx::URI::sGetFragment(void) const
{
	if(m_bArePartsValid == false)
	{
		vPartition();
	}
	
	return m_sFragment;
}

void Arxx::URI::vSetURI(const std::string & sURI)
{
	m_sURI = sURI;
	m_bIsURIValid = true;
	m_bArePartsValid = false;
	m_bIsQueryValid = false;
	m_bAreQueryPartsValid = false;
}

void Arxx::URI::vSetScheme(const std::string & sScheme)
{
	if(m_bArePartsValid == true)
	{
		m_sScheme = sScheme;
	}
	else
	{
		if(m_bIsURIValid == true)
		{
			vPartition();
			m_sScheme = sScheme;
		}
		else
		{
			return;
		}
	}
	m_bIsURIValid = false;
}

void Arxx::URI::vSetAuthority(const std::string & sAuthority)
{
	if(m_bArePartsValid == true)
	{
		m_sAuthority = sAuthority;
	}
	else
	{
		if(m_bIsURIValid == true)
		{
			vPartition();
			m_sAuthority = sAuthority;
		}
		else
		{
			return;
		}
	}
	m_bIsURIValid = false;
}

void Arxx::URI::vSetPath(const std::string & sPath)
{
	if(m_bArePartsValid == true)
	{
		m_sPath = sPath;
	}
	else
	{
		if(m_bIsURIValid == true)
		{
			vPartition();
			m_sPath = sPath;
		}
		else
		{
			return;
		}
	}
	m_bIsURIValid = false;
}

void Arxx::URI::vSetQuery(const std::string & sQuery)
{
	if(m_bArePartsValid == true)
	{
		m_sQuery = sQuery;
	}
	else
	{
		if(m_bIsURIValid == true)
		{
			vPartition();
			m_sQuery = sQuery;
		}
		else
		{
			return;
		}
	}
	m_bIsURIValid = false;
	m_bIsQueryValid = true;
	m_bAreQueryPartsValid = false;
}

void Arxx::URI::vSetParameter(const std::string & sName, const std::string & sValue)
{
	if(m_bAreQueryPartsValid == true)
	{
		m_Query[sName] = sValue;
	}
	else
	{
		if(m_bIsQueryValid == true)
		{
			vPartitionQuery();
		}
		else
		{
			if(m_bIsURIValid == true)
			{
				vPartition();
				vPartitionQuery();
			}
			else
			{
				return;
			}
		}
		m_Query[sName] = sValue;
	}
	m_bIsURIValid = m_bIsQueryValid = false;
}

void Arxx::URI::vSetFragment(const std::string & sFragment)
{
	if(m_bArePartsValid == true)
	{
		m_sFragment = sFragment;
	}
	else
	{
		if(m_bIsURIValid == true)
		{
			vPartition();
			m_sFragment = sFragment;
		}
		else
		{
			return;
		}
	}
	m_bIsURIValid = false;
}

void Arxx::URI::vInvalidate(void)
{
	m_bIsURIValid = m_bArePartsValid = m_bIsQueryValid = m_bAreQueryPartsValid = false;
}

bool Arxx::URI::operator<(const Arxx::URI & URI) const
{
	if(sGetScheme() == URI.sGetScheme())
	{
		if(sGetAuthority() == URI.sGetAuthority())
		{
			if(sGetPath() == URI.sGetPath())
			{
				if(sGetQuery() == URI.sGetQuery())
				{
					if(sGetFragment() == URI.sGetFragment())
					{
						return false;
					}
					else
					{
						return sGetFragment() < URI.sGetFragment();
					}
				}
				else
				{
					return sGetQuery() < URI.sGetQuery();
				}
			}
			else
			{
				return sGetPath() < URI.sGetPath();
			}
		}
		else
		{
			return sGetAuthority() < URI.sGetAuthority();
		}
	}
	else
	{
		return sGetScheme() < URI.sGetScheme();
	}
}

void Arxx::URI::vPartition(void) const
{
	if((m_bIsURIValid == false) || (m_bArePartsValid == true))
	{
		return;
	}
	m_sScheme = "";
	m_sAuthority = "";
	m_sPath = "";
	m_sQuery = "";
	m_sFragment = "";
	
	std::string::size_type stSchemeStart(std::string::npos);
	std::string::size_type stSchemeLength(std::string::npos);
	std::string::size_type stAuthorityStart(std::string::npos);
	std::string::size_type stAuthorityLength(std::string::npos);
	std::string::size_type stPathStart(std::string::npos);
	std::string::size_type stPathLength(std::string::npos);
	std::string::size_type stQueryStart(std::string::npos);
	std::string::size_type stQueryLength(std::string::npos);
	std::string::size_type stFragmentStart(std::string::npos);
	std::string::size_type stDelimiter1(m_sURI.find_first_of(":/?#", 0));
	
	if(stDelimiter1 == std::string::npos)
	{
		// we found no delimiter so the path starts at 0 and ends at the end
		stPathStart = 0;
	}
	else
	{
		if(m_sURI[stDelimiter1] == '#')
		{
			// the first delimiter marks the beginning of the fragment
			// => everything BEFORE the first delimiter is path
			if(stDelimiter1 != 0)
			{
				stPathStart = 0;
				stPathLength = stDelimiter1;
			}
			stFragmentStart = stDelimiter1 + 1;
		}
		else if(m_sURI[stDelimiter1] == '?')
		{
			// the first delimiter marks the beginning of the query
			// => everything BEFORE the first delimiter is path
			if(stDelimiter1 != 0)
			{
				stPathStart = 0;
				stPathLength = stDelimiter1;
			}
			// the first delimiter we find marks the query starting at stDelimiter1 + 1
			stQueryStart = stDelimiter1 + 1;
			
			std::string::size_type stDelimiter2(m_sURI.find('#', stDelimiter1 + 1));
			
			// stDelimiter2 marks the start of the fragment or the end
			if(stDelimiter2 != std::string::npos)
			{
				// query length is the difference
				stQueryLength = stDelimiter2 - stQueryStart;
				// fragment starts at stDelimiter2 + 1
				stFragmentStart = stDelimiter2 + 1;
			}
		}
		else if(m_sURI[stDelimiter1] == '/')
		{
			if(m_sURI[stDelimiter1 + 1] == '/')
			{
				// the first delimiter we find marks the authority starting at stDelimiter1 + 2
				stAuthorityStart = stDelimiter1 + 2;
				
				// search for the second delimiter
				std::string::size_type stDelimiter2(m_sURI.find_first_of("/?#", stDelimiter1 + 2));
				
				if(stDelimiter2 != std::string::npos)
				{
					// authority length is the difference
					stAuthorityLength = stDelimiter2 - stAuthorityStart;
					// what is the next part?
					if(m_sURI[stDelimiter2] == '#')
					{
						// the second delimiter we find marks the fragment starting at stDelimiter2 + 1
						stFragmentStart = stDelimiter2 + 1;
					}
					else if(m_sURI[stDelimiter2] == '?')
					{
						// the second delimiter we find marks the query starting at stDelimiter2 + 1
						stQueryStart = stDelimiter2 + 1;
						
						// search for the third delimiter
						std::string::size_type stDelimiter3(m_sURI.find('#', stDelimiter2 + 1));
						
						// the third delimiter marks the start of the fragment or the end
						if(stDelimiter3 != std::string::npos)
						{
							// query length is the difference
							stQueryLength = stDelimiter3 - stQueryStart;
							// fragment starts at stDelimiter3 + 1
							stFragmentStart = stDelimiter3 + 1;
						}
					}
					else if(m_sURI[stDelimiter2] == '/')
					{
						// the second delimiter we find marks the path starting at stDelimiter2
						stPathStart = stDelimiter2;
						
						// search for the third delimiter
						std::string::size_type stDelimiter3(m_sURI.find_first_of("?#", stDelimiter2 + 1));
						
						// the third delimiter marks the start of the fragment or the end
						if(stDelimiter3 != std::string::npos)
						{
							// path length is the difference
							stPathLength = stDelimiter3 - stPathStart;
							// what is the next part?
							if(m_sURI[stDelimiter3] == '#')
							{
								// the third delimiter we find marks the fragment starting at stDelimiter3 + 1
								stFragmentStart = stDelimiter3 + 1;
							}
							else
							{
								// the third delimiter we find marks the query starting at stDelimiter3 + 1
								stQueryStart = stDelimiter3 + 1;
								
								// search for the fourth delimiter
								std::string::size_type stDelimiter4(m_sURI.find('#', stDelimiter3 + 1));
								
								// the fourth delimiter marks the start of the fragment or the end
								if(stDelimiter4 != std::string::npos)
								{
									// query length is the difference
									stQueryLength = stDelimiter4 - stQueryStart;
									// fragment starts at stDelimiter4 + 1
									stFragmentStart = stDelimiter4 + 1;
								}
							}
						}
					}
				}
			}
			else
			{
				// the first delimiter we find is a slash and it is not a double slash "//"
				// => so this slash is part of the path and so is anything BEFORE the first slash!
				stPathStart = 0;
				
				// search for the second delimiter
				std::string::size_type stDelimiter2(m_sURI.find_first_of("?#", stDelimiter1 + 1));
				
				// the second delimiter marks the start of the fragment, the query or the end
				if(stDelimiter2 != std::string::npos)
				{
					// path length is the difference
					stPathLength = stDelimiter2 - stPathStart;
					// what is the next part?
					if(m_sURI[stDelimiter2] == '#')
					{
						// the second delimiter we find marks the fragment starting at stDelimiter2 + 1
						stFragmentStart = stDelimiter2 + 1;
					}
					else
					{
						// the second delimiter we find marks the query starting at stDelimiter2 + 1
						stQueryStart = stDelimiter2 + 1;
						
						// search for the third delimiter
						std::string::size_type stDelimiter3(m_sURI.find('#', stDelimiter2 + 1));
						
						// the third delimiter marks the start of the fragment or the end
						if(stDelimiter3 != std::string::npos)
						{
							// query length is the difference
							stQueryLength = stDelimiter3 - stQueryStart;
							// fragment starts at stDelimiter3 + 1
							stFragmentStart = stDelimiter3 + 1;
						}
					}
				}
			}
		}
		else
		{
			// the first delimiter we find is a ':' BUT this may as well be a part of the path
			if(stDelimiter1 == 0)
			{
				// since ':' is the very first character the scheme is empty and no ':' is required to end it
				//   the ':' is the first character of the path. Authority has to start with "//".
				stPathStart = stDelimiter1;
				
				// search for the second delimiter
				std::string::size_type stDelimiter2(m_sURI.find_first_of("?#", stDelimiter1 + 1));
				
				// the second delimiter marks the start of the fragment, the query or the end
				if(stDelimiter2 != std::string::npos)
				{
					// path length is the difference
					stPathLength = stDelimiter2 - stPathStart;
					// what is the next part?
					if(m_sURI[stDelimiter2] == '#')
					{
						// the second delimiter we find marks the fragment starting at stDelimiter2 + 1
						stFragmentStart = stDelimiter2 + 1;
					}
					else
					{
						// the second delimiter we find marks the query starting at stDelimiter2 + 1
						stQueryStart = stDelimiter2 + 1;
						
						// search for the third delimiter
						std::string::size_type stDelimiter3(m_sURI.find('#', stDelimiter2 + 1));
						
						// the third delimiter marks the start of the fragment or the end
						if(stDelimiter3 != std::string::npos)
						{
							// query length is the difference
							stQueryLength = stDelimiter3 - stQueryStart;
							// fragment starts at stDelimiter3 + 1
							stFragmentStart = stDelimiter3 + 1;
						}
					}
				}
			}
			else
			{
				stSchemeStart = 0;
				// the first delimiter we find marks the end of the scheme
				stSchemeLength = stDelimiter1;
				
				// search for the second delimiter
				std::string::size_type stDelimiter2(m_sURI.find_first_of("/?#", stDelimiter1 + 1));
				
				if(stDelimiter2 == std::string::npos)
				{
					stPathStart = stDelimiter1 + 1;
				}
				else
				{
					// what is the next part?
					if(m_sURI[stDelimiter2] == '#')
					{
						if(stDelimiter2 > stDelimiter1 + 1)
						{
							// what is between the ':' and the '#' is path
							stPathStart = stDelimiter1 + 1;
							stPathLength = stDelimiter2 - stPathStart;
						}
						// the second delimiter we find marks the fragment starting at stDelimiter2 + 1
						stFragmentStart = stDelimiter2 + 1;
					}
					else if(m_sURI[stDelimiter2] == '?')
					{
						if(stDelimiter2 > stDelimiter1 + 1)
						{
							// what is between the ':' and the '#' is path
							stPathStart = stDelimiter1 + 1;
							stPathLength = stDelimiter2 - stPathStart;
						}
						
						// the second delimiter we find marks the query starting at stDelimiter2 + 1
						stQueryStart = stDelimiter2 + 1;
						
						// search for the third delimiter
						std::string::size_type stDelimiter3(m_sURI.find('#', stDelimiter2 + 1));
						
						// the third delimiter marks the start of the fragment or the end
						if(stDelimiter3 != std::string::npos)
						{
							// query length is the difference
							stQueryLength = stDelimiter3 - stQueryStart;
							// fragment starts at stDelimiter3 + 1
							stFragmentStart = stDelimiter3 + 1;
						}
					}
					else if(m_sURI[stDelimiter2] == '/')
					{
						if(m_sURI[stDelimiter2 + 1] == '/')
						{
							// the second delimiter we find marks the authority starting at stDelimiter2 + 2
							stAuthorityStart = stDelimiter2 + 2;
							
							// search for the third delimiter
							std::string::size_type stDelimiter3(m_sURI.find_first_of("/?#", stDelimiter2 + 2));
							
							if(stDelimiter3 != std::string::npos)
							{
								// authority length is the difference
								stAuthorityLength = stDelimiter3 - stAuthorityStart;
								// what is the next part?
								if(m_sURI[stDelimiter3] == '#')
								{
									// the third delimiter we find marks the fragment starting at stDelimiter3 + 1
									stFragmentStart = stDelimiter3 + 1;
								}
								else if(m_sURI[stDelimiter3] == '?')
								{
									// the third delimiter we find marks the query starting at stDelimiter3 + 1
									stQueryStart = stDelimiter3 + 1;
									
									// search for the fourth delimiter
									std::string::size_type stDelimiter4(m_sURI.find('#', stDelimiter3 + 1));
									
									// the fourth delimiter marks the start of the fragment or the end
									if(stDelimiter4 != std::string::npos)
									{
										// query length is the difference
										stQueryLength = stDelimiter4 - stQueryStart;
										// fragment starts at stDelimiter4 + 1
										stFragmentStart = stDelimiter4 + 1;
									}
								}
								else if(m_sURI[stDelimiter3] == '/')
								{
									// the third delimiter we find marks the path starting at stDelimiter3
									stPathStart = stDelimiter3;
									
									// search for the fourth delimiter
									std::string::size_type stDelimiter4(m_sURI.find_first_of("?#", stDelimiter3 + 1));
									
									// the fourth delimiter marks the start of the fragment or the end
									if(stDelimiter4 != std::string::npos)
									{
										// path length is the difference
										stPathLength = stDelimiter4 - stPathStart;
										// what is the next part?
										if(m_sURI[stDelimiter4] == '#')
										{
											// the fourth delimiter we find marks the fragment starting at stDelimiter4 + 1
											stFragmentStart = stDelimiter4 + 1;
										}
										else
										{
											// the fourth delimiter we find marks the query starting at stDelimiter4 + 1
											stQueryStart = stDelimiter4 + 1;
											
											// search for the fifth delimiter
											std::string::size_type stDelimiter5(m_sURI.find('#', stDelimiter4 + 1));
											
											// the fifth delimiter marks the start of the fragment or the end
											if(stDelimiter5 != std::string::npos)
											{
												// query length is the difference
												stQueryLength = stDelimiter5 - stQueryStart;
												// fragment starts at stDelimiter5 + 1
												stFragmentStart = stDelimiter5 + 1;
											}
										}
									}
								}
							}
						}
						else
						{
							// the second delimiter we find marks a part of the path.
							// => but everything between the scheme and this slash is as well
							stPathStart = stDelimiter1 + 1;
							
							// search for the third delimiter
							std::string::size_type stDelimiter3(m_sURI.find_first_of("?#", stDelimiter2 + 1));
							
							// the third delimiter marks the start of the fragment, the query or the end
							if(stDelimiter3 != std::string::npos)
							{
								// path length is the difference
								stPathLength = stDelimiter3 - stPathStart;
								// what is the next part?
								if(m_sURI[stDelimiter3] == '#')
								{
									// the third delimiter we find marks the fragment starting at stDelimiter3 + 1
									stFragmentStart = stDelimiter3 + 1;
								}
								else
								{
									// the third delimiter we find marks the query starting at stDelimiter3 + 1
									stQueryStart = stDelimiter3 + 1;
									
									// search for the fourth delimiter
									std::string::size_type stDelimiter4(m_sURI.find('#', stDelimiter3 + 1));
									
									// the fourth delimiter marks the start of the fragment or the end
									if(stDelimiter4 != std::string::npos)
									{
										// query length is the difference
										stQueryLength = stDelimiter4 - stQueryStart;
										// fragment starts at stDelimiter4 + 1
										stFragmentStart = stDelimiter4 + 1;
									}
								}
							}
						}
					}
				}
			}
		}
	}
	if(stSchemeStart != std::string::npos)
	{
		m_sScheme = m_sURI.substr(stSchemeStart, stSchemeLength);
	}
	if(stAuthorityStart != std::string::npos)
	{
		m_sAuthority = m_sURI.substr(stAuthorityStart, stAuthorityLength);
	}
	if(stPathStart != std::string::npos)
	{
		m_sPath = m_sURI.substr(stPathStart, stPathLength);
	}
	if(stQueryStart != std::string::npos)
	{
		m_sQuery = m_sURI.substr(stQueryStart, stQueryLength);
	}
	if(stFragmentStart != std::string::npos)
	{
		m_sFragment = m_sURI.substr(stFragmentStart);
	}
	m_bArePartsValid = true;
	m_bAreQueryPartsValid = false;
	m_bIsQueryValid = true;
}

void Arxx::URI::vPartitionQuery(void) const
{
	if(m_bAreQueryPartsValid == true)
	{
		return;
	}
	m_Query.clear();
	
	const std::string & sQuery(sGetQuery());
	std::string::size_type stDelimiter;
	std::string::size_type stNameBegin;
	std::string::size_type stNameLength;
	std::string::size_type stValueBegin;
	std::string::size_type stValueLength;
	
	for(std::string::size_type stI = 0; stI < sQuery.length(); ++stI)
	{
		if(sQuery[stI] == '&')
		{
			// empty parameter
			continue;
		}
		stNameBegin = stI;
		stDelimiter = sQuery.find_first_of("&=", stNameBegin);
		if(stDelimiter != std::string::npos)
		{
			stNameLength = stDelimiter - stNameBegin;
			if(sQuery[stDelimiter] == '=')
			{
				stValueBegin = stDelimiter + 1;
				stDelimiter = sQuery.find('&', stValueBegin);
				if(stDelimiter != std::string::npos)
				{
					stValueLength = stDelimiter - stValueBegin;
					stI = stDelimiter;
				}
				else
				{
					stValueLength = std::string::npos;
					stI = sQuery.length();
				}
				m_Query[sQuery.substr(stNameBegin, stNameLength)] = sQuery.substr(stValueBegin, stValueLength);
			}
			else
			{
				stI = stDelimiter;
				m_Query[sQuery.substr(stNameBegin, stNameLength)] = "";
			}
		}
		else
		{
			stI = sQuery.length();
			m_Query[sQuery.substr(stNameBegin)] = "";
		}
	}
	m_bAreQueryPartsValid = true;
}

void Arxx::URI::vConcatenate(void) const
{
	if((m_bArePartsValid == false) || (m_bIsURIValid == true))
	{
		return;
	}
	m_sURI = "";
	if(m_sScheme != "")
	{
		m_sURI += m_sScheme;
		m_sURI += ':';
	}
	if(m_sAuthority != "")
	{
		m_sURI += "//";
		m_sURI += m_sAuthority;
	}
	m_sURI += m_sPath;
	if(m_bIsQueryValid == false)
	{
		vConcatenateQuery();
		if(m_sQuery != "")
		{
			m_sURI += '?';
			m_sURI += m_sQuery;
		}
	}
	if(m_sFragment != "")
	{
		m_sURI += '#';
		m_sURI += m_sFragment;
	}
	m_bIsURIValid = true;
}

void Arxx::URI::vConcatenateQuery(void) const
{
	m_sQuery = "";
	std::map< std::string, std::string >::const_iterator iQueryPart(m_Query.begin());
	
	while(iQueryPart != m_Query.end())
	{
		if(m_sQuery.empty() == false)
		{
			m_sQuery += '&';
		}
		m_sQuery += iQueryPart->first;
		m_sQuery += '=';
		m_sQuery += iQueryPart->second;
		++iQueryPart;
	}
}

std::ostream & Arxx::operator<<(std::ostream & OStream, const Arxx::URI & URI)
{
	return OStream << URI.sGetURI();
}
