/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include <iostream>
#include <sstream>

#include "../config.h"
#include "../Include/BufferWriter.h"
#include "LocalArchiveChannel.h"


///////////////////////////////////////////////////////////////////////////////////////////////////
// Arxx::LocalArchiveChannel                                                                      //
///////////////////////////////////////////////////////////////////////////////////////////////////

Arxx::LocalArchiveChannel::LocalArchiveChannel(const Arxx::URI & URI) :
	Arxx::DataChannel(URI),
	m_IStream(URI.sGetPath().c_str()),
	m_u4DataOffset(0)
{
	m_bWantsAutoDeletion = true;
#ifdef DEBUG
	std::cerr << "+ Arxx::LocalArchiveChannel [\"" << GetURI() << "\"]." << std::endl;
#endif
}

Arxx::LocalArchiveChannel::~LocalArchiveChannel(void)
{
	vOnUnreferenced();
#ifdef DEBUG
	std::cerr << "- Arxx::LocalArchiveChannel [\"" << GetURI() << "\"]." << std::endl;
#endif
}

std::istream & Arxx::LocalArchiveChannel::GetStream(void)
{
	return m_IStream;
}

void Arxx::LocalArchiveChannel::vMarkDataOffset(void)
{
	m_u4DataOffset = m_IStream.tellg();
}

bool Arxx::LocalArchiveChannel::bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus)
{
	if(!m_IStream)
	{
		std::cerr << " *** Arxx::LocalArchiveChannel::bFetchData: The input stream is not open for fetching." << std::endl;
		
		return false;
	}
	
	const std::string & sOffset(URI.sGetParameter("Offset"));
	
	if(sOffset == "")
	{
		std::cerr << " *** Arxx::LocalArchiveChannel::bFetchData: Missing \"Offset\" parameter in URI \"" << URI.sGetURI() << "\"." << std::endl;
		
		return false;
	}
	
	const std::string & sLength(URI.sGetParameter("Length"));
	
	if(sLength == "")
	{
		std::cerr << " *** Arxx::LocalArchiveChannel::bFetchData: Missing \"Length\" parameter in URI \"" << URI.sGetURI() << "\"." << std::endl;
		
		return false;
	}
	
	std::stringstream ssOffset(sOffset);
	std::stringstream ssLength(sLength);
	Arxx::u4byte u4Offset;
	Arxx::u4byte u4Length;
	
	ssOffset >> u4Offset;
	ssLength >> u4Length;
	m_IStream.seekg(m_u4DataOffset + u4Offset);
	Buffer.vSetLength(0);
	
	Arxx::BufferWriter BufferWriter(Buffer);
	
	FetchStatus = Arxx::TRANSFERING;
	BufferWriter << std::make_pair(static_cast< Arxx::Buffer::size_type >(u4Length), reinterpret_cast< std::istream * >(&m_IStream));
	FetchStatus = Arxx::FETCHED;
	
	return true;
}

void Arxx::LocalArchiveChannel::vOnUnreferenced(void)
{
	if(m_IStream)
	{
		m_IStream.close();
	}
}
