/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_ARCHIVEFILE_H
#define ARXX_ARCHIVEFILE_H

#include "../Include/Common.h"

/**
 * @file ArchiveFile.h
 * 
 * A hidden header file with Arxx internal structs and function definitions for reading an ARX archive file.
 **/

namespace Arxx
{
#pragma pack(1)
	/**
	 * @brief The representation of an archive's header in a file.
	 *
	 * This struct is the internal representation of an archive's header as it occures in a general ARX archive file. Reading the first sizeof(ArchiveHeader) bytes from an ARX archive file give you information about the version of the archive, the number of items contained in the archive and the ID of the root item. Thus, these information are considered absolutely neccessary for any loading process and on the other hand easy to get at the saving process.
	 **/
	class ArchiveHeader
	{
	public:
		union
		{
			/**
			 * @brief The version number of the archive.
			 *
			 * The archive's version number in one data item of 4 byte length. Use the u1bytes the get specific version information conveniently.
			 **/
			u4byte m_u4VersionNumber;
			struct
			{
				/**
				 * @brief The major major version number of the archive.
				 * @note Represents the XX.xx.xx.xx.
				 *
				 * This is the leading version number. A one here means it IS of version 1. Changes in this version number should be due to heavy design rewrites, so only if ONE thing changed FUNDAMENTLY. In general a change in this version number should also mean a new Arxx::Library and therefore a new Arxx::Item class since things really changed then.
				 **/
				u1byte m_u1MajorMajorVersion;
				
				/**
				 * @brief The major minor version number of the archive.
				 * @note Represents the xx.XX.xx.xx.
				 *
				 * This is the second leading version number. A change in this version occures when improvements change the way a library is stored. In other words whenever a structural change was necessary due to speed optimization procedures or for accesability benefits. A change here also means a new Arxx::Item class which should be backward compatible.
				 **/
				u1byte m_u1MajorMinorVersion;
				
				/**
				 * @brief The minor major version number of the archive.
				 * @note Represents the xx.xx.XX.xx.
				 *
				 * This is the first minor version number. Changes here have to be backward compatible in memory structure representatives.
				 **/
				u1byte m_u1MinorMajorVersion;
				
				/**
				 * @brief The minor minor version number of the archive.
				 * @note Represents the xx.xx.xx.XX.
				 *
				 * This is the second minor version number. This version number shouldn't be necessary.
				 **/
				u1byte m_u1MinorMinorVersion;
			};
		};
		
		/**
		 * @brief The unique ID of the root item of the archive.
		 *
		 * This member might equal Arxx::g_u4InvalidID when there is no root item specified or this archive.
		 **/
		u4byte m_u4RootItemID;
		
		/**
		 * @brief The number of items stored within the archive.
		 **/
		u4byte m_u4ItemCount;
	};
	
	/**
	 * @brief The memory representation of a single item.
	 * @note This class is located within a "pragma pack(1)" block.
	 *
	 * This class contains all the meta information associated with a specific item in an ARX archive.
	 **/
	class ItemHeader
	{
	public:
		/**
		 * @brief Default constructor creating an empty ItemHeader.
		 **/
		ItemHeader(void) :
			m_u4UniqueID(0),
			m_u4Type(0),
			m_u4SubType(0),
			m_u4VersionNumber(0),
			m_u4DataCompression(0),
			m_u4NameLength(0),
			m_u4DataDecompressedLength(0),
			m_u4DataCompressedLength(0),
			m_u4StructureDataLength(0)
		{
		}
		
		/**
		 * @brief The unique ID of the item.
		 *
		 * The unique ID of an item is unique in respect to the parenting archive.
		 **/
		u4byte m_u4UniqueID;
		
		/**
		 * @brief The type of the item.
		 *
		 * The type of the item.
		 **/
		u4byte m_u4Type;
		
		/**
		 * @brief The subtype of the item.
		 *
		 * The subtype of the item.
		 **/
		u4byte m_u4SubType;
		
		union
		{
			/**
			 * @brief The version number of the item.
			 *
			 * The item's version number in one data item of 4 byte length. Use the u1bytes the get specific version information easy.
			 **/
			u4byte m_u4VersionNumber;
			struct
			{
				/**
				 * @brief The major major version number of the item.
				 * @note Represents the XX.xx.xx.xx.
				 *
				 * This is the leading version number.
				 **/
				u1byte m_u1MajorMajorVersion;
				
				/**
				 * @brief The major minor version number of the item.
				 * @note Represents the xx.XX.xx.xx.
				 *
				 * This is the second leading version number.
				 **/
				u1byte m_u1MajorMinorVersion;
				
				/**
				 * @brief The minor major version number of the item.
				 * @note Represents the xx.xx.XX.xx.
				 *
				 * This is the first minor version number.
				 **/
				u1byte m_u1MinorMajorVersion;
				
				/**
				 * @brief The minor minor version number of the item.
				 * @note Represents the xx.xx.xx.XX.
				 *
				 * This is the second minor version number.
				 **/
				u1byte m_u1MinorMinorVersion;
			};
		};
		
		/**
		 * @brief Information about the compression of the item.
		 *
		 * The state of compression for the item. Defines the level of compression that was used with the zlib functions or 0 to indicate no compression.
		 **/
		u4byte m_u4DataCompression;
		
		/**
		 * @brief Defines whether the data is internal or external.
		 **/
		u1byte m_u1IsDataExternal;
		
		/**
		 * @brief The length of the item's name.
		 *
		 * The length of the item's name. The name is stored right after the ItemHeader and without a final zero character.
		 **/
		u4byte m_u4NameLength;
		
		/**
		 * @brief The offset/position of the item's data.
		 *
		 * The offset specifies the number of bytes to be added to the begin/offset of the data part of the archive.
		 **/
		u4byte m_u4DataOffset;
		
		/**
		 * @brief The length of the item's data when decompressed.
		 *
		 * The length of the data when it is decompressed. This member does not imply that the data actually IS decompressed. It merely gives you a 'would-be' information.
		 **/
		u4byte m_u4DataDecompressedLength;
		
		/**
		 * @brief The length of the item's data when compressed.
		 *
		 * The length of the data when it is compressed. This member does not imply that the data actually IS compressed. It merely gives you a 'would-be' information. If the item's data IS compressed, this member states the length of the compressed data block. On the other hand, when the item's data is decompressed, this member is invalid. (In the current implementation this member actually states the length of the last compression, but it CAN only give you a hint about the size of the data, in case you decompressed it externaly.
		 **/
		u4byte m_u4DataCompressedLength;
		
		/**
		 * @brief This component describes the length of the structure description for the item.
		 *
		 * The length of the structure buffer is needed when it will be read on loading an archive. The structure of the archive is part of its header information and therefore placed in the ItemHeader. The structure data follows immediately after the closing 0 of the item's name.
		 **/
		u4byte m_u4StructureDataLength;
	};
#pragma pack()

	/**
	 * @brief The output operator for a Arxx::LibraryHeader struct.
	 * 
	 * This function takes care of the correct output of a LibraryHeader to any kind of std::ostream.
	 **/
	inline std::ostream & operator<<(std::ostream & OStream, Arxx::ArchiveHeader & ArchiveHeader)
	{
		OStream.write(reinterpret_cast< const char * >(&ArchiveHeader), sizeof(Arxx::ArchiveHeader));
	
		return OStream;
	}
	
	/**
	 * @brief The output operator for a Arxx::ItemHeader struct.
	 * 
	 * This function takes care of the correct output of a ItemHeader to any kind of std::ostream.
	 **/
	inline std::ostream & operator<<(std::ostream & OStream, Arxx::ItemHeader & ItemHeader)
	{
		OStream.write(reinterpret_cast< const char * >(&ItemHeader), sizeof(Arxx::ItemHeader));
	
		return OStream;
	}
	
	/**
	 * @brief The input operator for a Arxx::LibraryHeader struct.
	 * 
	 * This function takes care of the correct reading of a LibraryHeader from any kind of std::istream.
	 **/
	inline std::istream & operator>>(std::istream & IStream, Arxx::ArchiveHeader & ArchiveHeader)
	{
		IStream.read(reinterpret_cast< char * >(&ArchiveHeader), sizeof(Arxx::ArchiveHeader));
		
		return IStream;
	}
	
	/**
	 * @brief The input operator for a Arxx::ItemHeader struct.
	 * 
	 * This function takes care of the correct reading of a ItemHeader from any kind of std::istream.
	 **/
	inline std::istream & operator>>(std::istream & IStream, Arxx::ItemHeader & ItemHeader)
	{
		IStream.read(reinterpret_cast< char * >(&ItemHeader), sizeof(Arxx::ItemHeader));
		
		return IStream;
	}
}

#endif
