/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include <stdlib.h>
#include <sys/wait.h>

#include <iostream>
#include <sstream>

#include "ArchiveFile.h"
#include "DefaultItemFactory.h"
#include "LocalArchiveChannel.h"
#include "ReferenceImplementation.h"
#include "../config.h"
#include "../Include/Archive.h"
#include "../Include/BufferWriter.h"
#include "../Include/DataRepository.h"
#include "../Include/Item.h"
#include "../Include/Merge.h"
#include "../Include/Reference.h"
#include "../Include/URI.h"

Arxx::DefaultItemFactory DefaultFactory;


///////////////////////////////////////////////////////////////////////////////////////////////////
// Arxx::Archive                                                                                  //
///////////////////////////////////////////////////////////////////////////////////////////////////

Arxx::Archive::Archive(void) :
	m_pRootItem(0),
	m_pItemFactory(&DefaultFactory)
{
	srand(time(0));
}

Arxx::Archive::Archive(Arxx::ItemFactory & ItemFactory) :
	m_pRootItem(0),
	m_pItemFactory(&ItemFactory)
{
	srand(time(0));
}

Arxx::Archive::~Archive(void)
{
	vClose();
}

bool Arxx::Archive::bLoad(Arxx::URI URI)
{
	if(URI.sGetScheme() == "")
	{
		URI.vSetScheme("file");
	}
	else if(URI.sGetScheme() != "file")
	{
		throw std::runtime_error("Invalid argument in Arxx::Archive::bLoad: URI = \"" + URI.sGetURI() + "\".");
	}
	
	Arxx::LocalArchiveChannel * pLocalArchiveChannel(new LocalArchiveChannel(URI));
	std::istream & IStream(pLocalArchiveChannel->GetStream());
	
	if(IStream == false)
	{
		return false;
	}
	// this position I chose after only a few looks at the file => prove me wrong ;)
	vClose();
	Arxx::Repository.bRegisterDataChannel(pLocalArchiveChannel);
	
	Arxx::ArchiveHeader ArchiveHeader;
	
	IStream >> ArchiveHeader;
	if((ArchiveHeader.m_u1MajorMajorVersion == 1) && (ArchiveHeader.m_u1MajorMinorVersion == 0) && (ArchiveHeader.m_u1MinorMajorVersion == 0) && (ArchiveHeader.m_u1MinorMinorVersion == 0))
	{
		vRead(pLocalArchiveChannel, ArchiveHeader.m_u4ItemCount);
		pLocalArchiveChannel->vMarkDataOffset();
		m_pRootItem = pGetItem(ArchiveHeader.m_u4RootItemID);
	}
	else
	{
		std::cerr << "Unknown ARX archive format: version = " << static_cast< Arxx::u4byte >(ArchiveHeader.m_u1MajorMajorVersion) << '.' << static_cast< Arxx::u4byte >(ArchiveHeader.m_u1MajorMinorVersion) << '.' << static_cast< Arxx::u4byte >(ArchiveHeader.m_u1MinorMajorVersion) << '.' << static_cast< Arxx::u4byte >(ArchiveHeader.m_u1MinorMinorVersion) << '.' << std::endl;
		throw bad_file_format(URI.sGetURI());
	}
	
	return true;
}

void Arxx::Archive::vClose(void)
{
	while(m_Items.begin() != m_Items.end())
	{
		Arxx::Item * pItem(m_Items.begin()->second);
		Arxx::ItemFactory * pItemFactory(pItem->pGetItemFactory());
		
		vUnregisterItem(*pItem);
		if(pItemFactory != 0)
		{
			pItemFactory->vDeleteItem(pItem);
		}
	}
	m_pRootItem = 0;
	
	std::map< Arxx::u4byte, Arxx::Reference >::iterator iReference(m_References.begin());
	
	while(iReference != m_References.end())
	{
		iReference->second.vUnresolve();
		m_References.erase(iReference);
		iReference = m_References.begin();
	}
}

bool Arxx::Archive::bMerge(Arxx::Archive & Archive, std::vector< Arxx::Rule * > & Rules)
{
	Arxx::Context Context(*this, Archive);
	
	// yep, this will completely empty the source library
	while(Archive.begin() != Archive.end())
	{
		Arxx::Item & Item(*(Archive.begin()));
		bool bActionPerformed = false;
		
		// unregister the item from the source library to be able to change it without constraints
		Archive.vUnregisterItem(Item);
		for(std::vector< Arxx::Rule * >::iterator iRule(Rules.begin()); (iRule != Rules.end()) && (bActionPerformed == false); ++iRule)
		{
			Context.vSetCurrentItem(Item);
			if((*iRule)->bMatch(Context) == true)
			{
				switch((*iRule)->GetAction(Context))
				{
				case Arxx::ADD:
					{
						Context.vAddingItem(Item);
						bActionPerformed = true;
						
						break;
					}
				case Arxx::DROP:
					{
						bActionPerformed = true;
						
						break;
					}
				case Arxx::PASS:
					{
						break;
					}
				case Arxx::REPLACE:
					{
						Arxx::Item * pItem(pGetItem(Item.u4GetUniqueID()));
						
						if(pItem != 0)
						{
							vUnregisterItem(*pItem);
						}
						try
						{
							vRegisterItem(Item);
						}
						catch(Arxx::id_not_unique & Exception)
						{
							return false;
						}
						bActionPerformed = true;
						
						break;
					}
				default:
					{
						throw std::logic_error("Unkown Action on merging.");
					}
				}
			}
		}
	}
	Context.vAddItems();
	
	return true;
}

void Arxx::Archive::vRegisterItem(Arxx::Item & Item)
{
	if(Item.pGetArchive() == 0)
	{
		Item.m_pArchive = this;
	}
	if(Item.pGetArchive() != this)
	{
		throw std::invalid_argument("Item");
	}
	if(Item.u4GetUniqueID() == g_u4InvalidID)
	{
		std::map< Arxx::u4byte, Arxx::Item * >::iterator iIterator;
		Arxx::u4byte u4NewID;
		
		do
		{
			u4NewID = rand();
			iIterator = m_Items.find(u4NewID);
		} while((iIterator != m_Items.end()) || (u4NewID == g_u4InvalidID));
		Item.vSetUniqueID(u4NewID);
	}
	else
	{
		std::map< Arxx::u4byte, Arxx::Item * >::iterator iIterator = m_Items.find(Item.u4GetUniqueID());
		
		if(iIterator != m_Items.end())
		{
			std::stringstream ssID;
			
			ssID << Item.u4GetUniqueID();
			
			throw id_not_unique(ssID.str());
		}
		m_Items[Item.u4GetUniqueID()] = &Item;
		
		std::map< Arxx::u4byte, Arxx::Reference >::iterator iReference(m_References.find(Item.u4GetUniqueID()));
		
		if(iReference == m_References.end())
		{
			m_References.insert(std::make_pair(Item.u4GetUniqueID(), Arxx::Reference(Item)));
		}
		else
		{
			iReference->second.vResolve(Item);
		}
		
		/** @todo the following is the brute force implementation. It can be done much faster with a merge sort **/
		Arxx::Structure::iterator iRelation(Item.GetStructure().begin());
		Arxx::Structure::iterator iEndRelation(Item.GetStructure().end());
		
		while(iRelation != iEndRelation)
		{
			Arxx::Structure::Relation::iterator iReference(iRelation->begin());
			Arxx::Structure::Relation::iterator iEndReference(iRelation->end());
			
			while(iReference != iEndReference)
			{
				// use GetReference() to create or retrieve the Reference in the archive's Reference map and then attach the Item's Reference to it
				Arxx::Reference Reference(GetReference(iReference->u4GetUniqueID()));
				
				Reference.bAttach(*iReference);
				++iReference;
			}
			++iRelation;
		}
	}
}

void Arxx::Archive::vUnregisterItem(Item & Item)
{
	if(Item.pGetArchive() != this)
	{
		throw std::invalid_argument("Item");
	}
	
	std::map< Arxx::u4byte, Arxx::Item * >::iterator iIterator(m_Items.find(Item.u4GetUniqueID()));
	std::map< Arxx::u4byte, Arxx::Reference >::iterator iReference(m_References.find(Item.u4GetUniqueID()));
	
	if((iIterator == m_Items.end()) || (iReference == m_References.end()))
	{
		throw std::invalid_argument("Item");
	}
	
	// before removing the Item from the Archive we correct its references
	Arxx::Structure::iterator iRelation(Item.GetStructure().begin());
	Arxx::Structure::iterator iEndRelation(Item.GetStructure().end());
	
	while(iRelation != iEndRelation)
	{
		Arxx::Structure::Relation::iterator iReference(iRelation->begin());
		Arxx::Structure::Relation::iterator iEndReference(iRelation->end());
		
		while(iReference != iEndReference)
		{
			iReference->bDetach();
			++iReference;
		}
		++iRelation;
	}
	// now we can remove the Item
	m_Items.erase(iIterator);
	Item.m_pArchive = 0;
	iReference->second.vUnresolve();
	iReference->second.vDecoupleFromArchive();
	m_References.erase(iReference);
}

void Arxx::Archive::vSetRootItem(Item * pItem)
{
	if((pItem != 0) && (pItem->pGetArchive() != this))
	{
		throw std::invalid_argument("pItem");
	}
	m_pRootItem = pItem;
}

Arxx::ItemFactory * Arxx::Archive::pGetItemFactory(void)
{
	return m_pItemFactory;
}

const Arxx::Item * Arxx::Archive::pGetItem(Arxx::u4byte u4UniqueID) const
{
	if(u4UniqueID == g_u4InvalidID)
	{
		return 0;
	}
	
	std::map< Arxx::u4byte, Arxx::Item * >::const_iterator iIterator(m_Items.find(u4UniqueID));
	
	if(iIterator == m_Items.end())
	{
		return 0;
	}
	
	return (*iIterator).second;
}

Arxx::Item * Arxx::Archive::pGetItem(Arxx::u4byte u4UniqueID)
{
	if(u4UniqueID == g_u4InvalidID)
	{
		return 0;
	}
	
	std::map< Arxx::u4byte, Arxx::Item * >::iterator iIterator(m_Items.find(u4UniqueID));
	
	if(iIterator == m_Items.end())
	{
		return 0;
	}
	
	return (*iIterator).second;
}

Arxx::Item * Arxx::Archive::GetItem(std::string Path)
{
	if(Path[0] != '/')
	{
		return 0;
	}
	
	std::string ChildRelation;
	Arxx::Item * Item(pGetRootItem());
	
	while((Item != 0) && ((Path != "/") || (Path == "")))
	{
		std::string::size_type PathSeparatorIndex(Path.find('/', 1));
		std::string ChildName(Path.substr(1, PathSeparatorIndex - 1));
		
		if(PathSeparatorIndex == std::string::npos)
		{
			Path = "/";
		}
		else
		{
			Path = Path.substr(PathSeparatorIndex);
		}
		
		std::string::size_type RelationSeparatorIndex(ChildName.find("::"));
		
		if(RelationSeparatorIndex != std::string::npos)
		{
			ChildRelation = ChildName.substr(0, RelationSeparatorIndex);
			ChildName = ChildName.substr(RelationSeparatorIndex + 2);
		}
		else
		{
			ChildRelation = "child";
		}
		if(Item->GetStructure().bHasRelation(ChildRelation) == false)
		{
			return 0;
		}
		else
		{
			Arxx::Structure::Relation & Relation(Item->GetStructure().GetRelation(ChildRelation));
			Arxx::Structure::Relation::iterator ReferenceIterator(Relation.begin());
			
			for(; ReferenceIterator != Relation.end(); ++ReferenceIterator)
			{
				if((ReferenceIterator->pGetItem() != 0) && (ReferenceIterator->pGetItem()->sGetName() == ChildName))
				{
					Item = ReferenceIterator->pGetItem();
					
					break;
				}
			}
			if(ReferenceIterator == Relation.end())
			{
				return 0;
			}
		}
	}
	
	return Item;
}

const Arxx::Item * Arxx::Archive::GetItem(std::string Path) const
{
	if(Path[0] != '/')
	{
		return 0;
	}
	
	std::string ChildRelation;
	const Arxx::Item * Item(pGetRootItem());
	
	while((Item != 0) && ((Path != "/") || (Path == "")))
	{
		std::string::size_type PathSeparatorIndex(Path.find('/', 1));
		std::string ChildName(Path.substr(1, PathSeparatorIndex - 1));
		
		if(PathSeparatorIndex == std::string::npos)
		{
			Path = "/";
		}
		else
		{
			Path = Path.substr(PathSeparatorIndex);
		}
		
		std::string::size_type RelationSeparatorIndex(ChildName.find("::"));
		
		if(RelationSeparatorIndex != std::string::npos)
		{
			ChildRelation = ChildName.substr(0, RelationSeparatorIndex);
			ChildName = ChildName.substr(RelationSeparatorIndex + 2);
		}
		else
		{
			ChildRelation = "child";
		}
		if(Item->GetStructure().bHasRelation(ChildRelation) == false)
		{
			return 0;
		}
		else
		{
			const Arxx::Structure::Relation & Relation(Item->GetStructure().GetRelation(ChildRelation));
			Arxx::Structure::Relation::const_iterator ReferenceIterator(Relation.begin());
			
			for(; ReferenceIterator != Relation.end(); ++ReferenceIterator)
			{
				if((ReferenceIterator->pGetItem() != 0) && (ReferenceIterator->pGetItem()->sGetName() == ChildName))
				{
					Item = ReferenceIterator->pGetItem();
					
					break;
				}
			}
			if(ReferenceIterator == Relation.end())
			{
				return 0;
			}
		}
	}
	
	return Item;
}

Arxx::Item * Arxx::Archive::pGetRootItem(void)
{
	return m_pRootItem;
}

const Arxx::Item * Arxx::Archive::pGetRootItem(void) const
{
	return m_pRootItem;
}

Arxx::Reference Arxx::Archive::GetReference(Arxx::u4byte u4UniqueID)
{
	std::map< Arxx::u4byte, Arxx::Reference >::iterator iReference(m_References.find(u4UniqueID));
	
	if(iReference == m_References.end())
	{
		return m_References.insert(std::make_pair(u4UniqueID, Arxx::Reference(u4UniqueID, this))).first->second;
	}
	else
	{
		return iReference->second;
	}
}

void Arxx::Archive::vReleaseReference(Arxx::ReferenceImplementation * pReference)
{
	std::map< Arxx::u4byte, Arxx::Reference >::iterator iReference(m_References.find(pReference->u4GetUniqueID()));
	
	if(iReference == m_References.end())
	{
		throw std::runtime_error("Arxx::Archive::vReleaseReference: Reference not found in the reference map.");
	}
	m_References.erase(iReference);
}

void Arxx::Archive::vRead(Arxx::LocalArchiveChannel * pLocalArchiveChannel, u4byte u4ItemCount)
{
	std::istream & IStream(pLocalArchiveChannel->GetStream());
	Arxx::ItemHeader ItemHeader;
	Arxx::u4byte u4I(0);
	std::map< Arxx::u4byte, Arxx::Item * > ExternalData;
	
	while(++u4I <= u4ItemCount)
	{
		IStream >> ItemHeader;
		
		Item * pItem = pGetItemFactory()->pNewItem(*this, ItemHeader.m_u4UniqueID);
		
		pItem->vSetType(ItemHeader.m_u4Type);
		pItem->vSetSubType(ItemHeader.m_u4SubType);
		pItem->vSetVersion(ItemHeader.m_u4VersionNumber);
		
		char * pcName = new char[ItemHeader.m_u4NameLength + 1];
		
		IStream.read(pcName, ItemHeader.m_u4NameLength + 1);
		pcName[ItemHeader.m_u4NameLength] = '\0';
		pItem->vSetName(pcName);
		delete[] pcName;
		
		if(ItemHeader.m_u1IsDataExternal == 1)
		{
			ExternalData[ItemHeader.m_u4DataOffset] = pItem;
		}
		else
		{
			std::stringstream ssURI;
			
			ssURI << pLocalArchiveChannel->GetURI() << "?Offset=" << ItemHeader.m_u4DataOffset << "&Length=" << ((ItemHeader.m_u4DataCompression == 0) ? (ItemHeader.m_u4DataDecompressedLength) : (ItemHeader.m_u4DataCompressedLength));
			pItem->vSetExternal(ssURI.str(), static_cast< Arxx::Data::Compression >(ItemHeader.m_u4DataCompression), ItemHeader.m_u4DataDecompressedLength, ItemHeader.m_u4DataCompressedLength);
			pItem->vInternalize();
		}
		pItem->GetStructure().vReadFromStream(ItemHeader.m_u4StructureDataLength, IStream);
	}
	// reading the number of external references
	IStream.read(reinterpret_cast< char * >(&u4I), 4);
#ifdef DEBUG
	std::cerr << "  Reading " << u4I << " external data URIs from the file." << std::endl;
#endif
	
	Arxx::u4byte u4ExternalDataID(0);
	Arxx::u4byte u4ParentExternalDataID(0);
	std::string sURI;
	char cChar;
	
	while(u4I > 0)
	{
		// reading the external data ID
		IStream.read(reinterpret_cast< char * >(&u4ExternalDataID), 4);
		sURI = "";
		while((IStream.get(cChar)) && (cChar != '\0'))
		{
			sURI += cChar;
		}
#ifdef DEBUG
		std::cerr << "  ID = " << u4ExternalDataID << " ; URI = \"" << sURI << "\"" << std::endl;
#endif
		/// @todo Since we don't save parent external data IDs yet this is safe, but we have to handle it.
		IStream.read(reinterpret_cast< char * >(&u4ParentExternalDataID), 4);
		/// @todo Here the handling of the data sizes and compression state from the ItemHeader is missing.
		ExternalData[u4ExternalDataID]->vSetExternal(sURI);
		u4I--;
	}
}

void Arxx::Archive::vSave(Arxx::URI URI, bool bAutoCompress)
{
	if(URI.sGetScheme() == "")
	{
		URI.vSetScheme("file");
	}
	else if(URI.sGetScheme() != "file")
	{
		throw std::runtime_error("Invalid argument in Arxx::Archive::bLoad: URI = \"" + URI.sGetURI() + "\".");
	}
	
	std::stringstream ssTemporaryFilePath;
	
	ssTemporaryFilePath << "/tmp/ARX" << time(0);
	
	std::ofstream OStream(ssTemporaryFilePath.str().c_str());
	
	if(!OStream)
	{
		throw file_error(ssTemporaryFilePath.str());
	}
	
	Arxx::u4byte u4DataOffset(0);
	Arxx::ArchiveHeader ArchiveHeader;
	
	ArchiveHeader.m_u1MajorMajorVersion = 1;
	ArchiveHeader.m_u1MajorMinorVersion = 0;
	ArchiveHeader.m_u1MinorMajorVersion = 0;
	ArchiveHeader.m_u1MinorMinorVersion = 0;
	ArchiveHeader.m_u4RootItemID = ((m_pRootItem != 0) ? (m_pRootItem->u4GetUniqueID()) : (g_u4InvalidID));
	ArchiveHeader.m_u4ItemCount = m_Items.size();
	OStream << ArchiveHeader;
	
	std::map< Arxx::u4byte, Arxx::Item * > ExternalData;
	std::map< Arxx::u4byte, Arxx::Item * >::iterator iIterator(m_Items.begin());
	
	while(iIterator != m_Items.end())
	{
		Item & Item(*((*iIterator).second));
		
		if(Item.bIsInternal() == true)
		{
			if(Item.bIsFetched() == false)
			{
				Item.bFetch();
			}
			if((Item.bIsFetched() == true) && (Item.bIsCompressed() == false) && (Item.u4GetDecompressedLength() > 32) && (bAutoCompress == true))
			{
				Item.vCompress();
			}
		}
		
		Arxx::ItemHeader ItemHeader;
		
		ItemHeader.m_u4UniqueID = Item.u4GetUniqueID();
		ItemHeader.m_u4Type = Item.u4GetType();
		ItemHeader.m_u4SubType = Item.u4GetSubType();
		ItemHeader.m_u4VersionNumber = Item.u4GetVersion();
		ItemHeader.m_u4DataCompression = Item.GetCompression();
		ItemHeader.m_u4NameLength = Item.sGetName().length();
		ItemHeader.m_u4DataDecompressedLength = Item.u4GetDecompressedLength();
		ItemHeader.m_u4DataCompressedLength = Item.u4GetCompressedLength();
		if(Item.bIsExternal() == true)
		{
			ItemHeader.m_u1IsDataExternal = 1;
			ItemHeader.m_u4DataOffset = ExternalData.size();
			ExternalData[ExternalData.size()] = &Item;
		}
		else
		{
			ItemHeader.m_u1IsDataExternal = 0;
			ItemHeader.m_u4DataOffset = u4DataOffset;
			u4DataOffset += ((Item.bIsCompressed() == true) ? (Item.u4GetCompressedLength()) : (Item.u4GetDecompressedLength()));
		}
		
		Arxx::Data StructureBuffer;
		
		StructureBuffer << Item.GetStructure();
		ItemHeader.m_u4StructureDataLength = StructureBuffer.stGetLength();
		OStream << ItemHeader << Item.sGetName() << '\0' << StructureBuffer;
		++iIterator;
	}
	
	Arxx::u4byte u4ExternalDataLength(ExternalData.size());
	Arxx::Buffer ExternalDataBuffer;
	Arxx::BufferWriter ExternalDataBufferWriter(ExternalDataBuffer);
	
#ifdef DEBUG
	std::cerr << "  Writing " << u4ExternalDataLength << " external data URIs to the file." << std::endl;
#endif
	iIterator = ExternalData.begin();
	while(iIterator != ExternalData.end())
	{
#ifdef DEBUG
		std::cerr << "    ID = " << iIterator->first << " ; URI = \"" << iIterator->second->GetURI() << "\"." << std::endl;
#endif
		ExternalDataBufferWriter << iIterator->first << iIterator->second->GetURI().sGetURI() << static_cast< Arxx::u4byte >(0);
		++iIterator;
	}
	OStream.write(reinterpret_cast< char * >(&u4ExternalDataLength), 4);
	OStream << ExternalDataBuffer;
	iIterator = m_Items.begin();
	while(iIterator != m_Items.end())
	{
		if(iIterator->second->bIsInternal() == true)
		{
			Arxx::Buffer & Buffer(*(iIterator->second));
			
			OStream << Buffer;
		}
		++iIterator;
	}
	OStream.close();
	
	int iChildPID = 0;
	int iReturn = 0;
	
	if((iChildPID = fork()) == 0)
	{
		execlp("mv", "mv", "--force", ssTemporaryFilePath.str().c_str(), URI.sGetPath().c_str(), reinterpret_cast< char * >(0));
	}
	waitpid(iChildPID, &iReturn, 0);
}

Arxx::Archive::iterator Arxx::Archive::begin(void)
{
	// will be fed into an implicite constructor for Arxx::Archive::iterator
	return m_Items.begin();
}

Arxx::Archive::iterator Arxx::Archive::end(void)
{
	// will be fed into an implicite constructor for Arxx::Archive::iterator
	return m_Items.end();
}

Arxx::Archive::const_iterator Arxx::Archive::begin(void) const
{
	// will be fed into an implicite constructor for Arxx::Archive::const_iterator
	return m_Items.begin();
}

Arxx::Archive::const_iterator Arxx::Archive::end(void) const
{
	// will be fed into an implicite constructor for Arxx::Archive::const_iterator
	return m_Items.end();
}

Arxx::Archive::size_type Arxx::Archive::size(void) const
{
	return m_Items.size();
}

Arxx::u4byte Arxx::Archive::u4GetNumberOfReferences(void) const
{
	return m_References.size();
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Arxx::Archive::iterator                                                                        //
///////////////////////////////////////////////////////////////////////////////////////////////////

Arxx::Archive::iterator::iterator(std::map< Arxx::u4byte, Arxx::Item * >::iterator iItem) :
	m_iItem(iItem)
{
}

Arxx::Archive::iterator::~iterator(void)
{
}

Arxx::Archive::iterator & Arxx::Archive::iterator::operator++(void)
{
	++m_iItem;

	return *this;
}

Arxx::Item & Arxx::Archive::iterator::operator*(void)
{
	return *(m_iItem->second);
}

const Arxx::Item & Arxx::Archive::iterator::operator*(void) const
{
	return *(m_iItem->second);
}

Arxx::Item * Arxx::Archive::iterator::operator->(void)
{
	return m_iItem->second;
}

bool Arxx::Archive::iterator::operator==(const Arxx::Archive::iterator & iItem) const
{
	return m_iItem == iItem.m_iItem;
}

bool Arxx::Archive::iterator::operator!=(const Arxx::Archive::iterator & iItem) const
{
	return m_iItem != iItem.m_iItem;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Arxx::Archive::const_iterator                                                                  //
///////////////////////////////////////////////////////////////////////////////////////////////////

Arxx::Archive::const_iterator::const_iterator(std::map< Arxx::u4byte, Arxx::Item * >::const_iterator iItem) :
	m_iItem(iItem)
{
}

Arxx::Archive::const_iterator::~const_iterator(void)
{
}

Arxx::Archive::const_iterator & Arxx::Archive::const_iterator::operator++(void)
{
	++m_iItem;

	return *this;
}

const Arxx::Item & Arxx::Archive::const_iterator::operator*(void) const
{
	return *(m_iItem->second);
}

const Arxx::Item * Arxx::Archive::const_iterator::operator->(void) const
{
	return m_iItem->second;
}

bool Arxx::Archive::const_iterator::operator==(const Arxx::Archive::const_iterator & iItem) const
{
	return m_iItem == iItem.m_iItem;
}

bool Arxx::Archive::const_iterator::operator!=(const Arxx::Archive::const_iterator & iItem) const
{
	return m_iItem != iItem.m_iItem;
}
