/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_URI_H
#define ARXX_URI_H

#include <map>
#include <string>

/**
 * @file URI.h
 * 
 * This file declares the Arxx::URI class a convenience wrapper with parsing capabilities around a string to handle URIs.
 **/

namespace Arxx
{
	/**
	 * @brief A URI class.
	 **/
	class URI
	{
	public:
		/**
		 * @brief The default constructor.
		 * 
		 * This default constructor implements an "initialized but invalid" semantic.
		 **/
		URI(void);
		
		/**
		 * @brief The constructor for URIs.
		 * @param sURI The URI std::string that this object should represent.
		 **/
		URI(const std::string & sURI);
		
		/**
		 * @brief The constructor for URIs.
		 * @param pcURI The URI C string that this object should represent.
		 **/
		URI(const char * pcURI);
		
		/**
		 * @brief Returns whether concatenated URI or the parts are valid.
		 **/
		bool bIsValid(void) const;
		
		/**
		 * @brief Returns the URI in string representation.
		 **/
		const std::string & sGetURI(void) const;
		
		/**
		 * @brief Returns the scheme part of the URI.
		 **/
		const std::string & sGetScheme(void) const;
		
		/**
		 * @brief Returns the authority part of the URI.
		 **/
		const std::string & sGetAuthority(void) const;
		
		/**
		 * @brief Returns the path part of the URI.
		 **/
		const std::string & sGetPath(void) const;
		
		/**
		 * @brief Returns the query part of the URI.
		 **/
		const std::string & sGetQuery(void) const;
		
		/**
		 * @brief Returns a specific value from the query string.
		 * @param sName The name of the value to be returned.
		 * @return The value of the query parameter identified by @a sName or "" if the parameter could not be found.
		 * 
		 * @note Parameters have to be separated with '&' and `name => value` assignments are done with '='.
		 **/
		const std::string & sGetParameter(const std::string & sName) const;
		
		/**
		 * @brief Returns the fragment part of the URI.
		 **/
		const std::string & sGetFragment(void) const;
		
		/**
		 * @brief Resets the URI.
		 * @param sURI The string representation of the URI.
		 **/
		void vSetURI(const std::string & sURI);
		
		/**
		 * @brief Sets the scheme of the URI.
		 * @param sScheme The string representation of the scheme of the URI.
		 **/
		void vSetScheme(const std::string & sScheme);
		
		/**
		 * @brief Sets the authority of the URI.
		 * @param sAuthority The string representation of the authority of the URI.
		 **/
		void vSetAuthority(const std::string & sAuthority);
		
		/**
		 * @brief Sets the path of the URI.
		 * @param sPath The string representation of the path of the URI.
		 **/
		void vSetPath(const std::string & sPath);
		
		/**
		 * @brief Sets the query of the URI.
		 * @param sQuery The string representation of the query of the URI.
		 **/
		void vSetQuery(const std::string & sQuery);
		
		/**
		 * @brief Sets a part of the query of the URI.
		 * @param sName The name of the query parameter of the URI.
		 * @param sValue The value of the query parameter of the URI.
		 **/
		void vSetParameter(const std::string & sName, const std::string & sValue);
		
		/**
		 * @brief Sets the fragment of the URI.
		 * @param sFragment The string representation of the fragment of the URI.
		 **/
		void vSetFragment(const std::string & sFragment);
		
		/**
		 * @brief Operator returning whether one URI is less than another.
		 **/
		bool operator<(const Arxx::URI & URI) const;
		
		/**
		 * @brief Invalidates the URI object.
		 **/
		void vInvalidate(void);
	private:
		/**
		 * @brief This function partitions the URI and fills the individual parts of the URI instance.
		 **/
		void vPartition(void) const;
		
		/**
		 * @brief This function partitions the query part of the URI and fills the query map.
		 **/
		void vPartitionQuery(void) const;
	
		/**
		 * @brief This function concatenates the URI from the individual parts of the URI instance.
		 **/
		void vConcatenate(void) const;
	
		/**
		 * @brief This function concatenates the query of the URI from the individual query parts in the query map.
		 **/
		void vConcatenateQuery(void) const;
		
		/**
		 * @brief The string representation of this URI.
		 **/
		mutable std::string m_sURI;
		
		/**
		 * @brief The scheme part of the URI.
		 **/
		mutable std::string m_sScheme;
		
		/**
		 * @brief The authority part of the URI.
		 **/
		mutable std::string m_sAuthority;
		
		/**
		 * @brief The path part of the URI.
		 **/
		mutable std::string m_sPath;
		
		/**
		 * @brief The query part of the URI.
		 **/
		mutable std::string m_sQuery;
		
		/**
		 * @brief The fragment part of the URI.
		 **/
		mutable std::string m_sFragment;
		
		/**
		 * @brief Indicator whether the concatenated URI is valid.
		 **/
		mutable bool m_bIsURIValid;
		
		/**
		 * @brief Indicator whether the URI parts are valid.
		 **/
		mutable bool m_bArePartsValid;
		
		/**
		 * @brief Indicator whether the query part are valid.
		 **/
		mutable bool m_bIsQueryValid;
		
		/**
		 * @brief Indicator whether the query parts are valid.
		 * 
		 * If m_ArePartsValid is false this is false too.
		 **/
		mutable bool m_bAreQueryPartsValid;
		
		/**
		 * @brief The query parts of the URI.
		 * 
		 * The parts of a query are separated by '&' and `name => value` assignment is indicated by '='.
		 **/
		mutable std::map< std::string, std::string > m_Query;
	};
	
	std::ostream & operator<<(std::ostream & OStream, const Arxx::URI & URI);
}

#endif
