/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_MERGE_H
#define ARXX_MERGE_H

#include <assert.h>

#include <vector>

#include "Common.h"

/**
 * @defgroup merging Merging
 * 
 * Merging is a very difficult process but almost certainly the only thing that differentiates it from almost every other library tool that I know. Merging is what unionfs is about.
 * 
 * Merging, in general, is the name of the process when two libraries are taken to be the data base of one new library. However in order to simplify code and get closer to intuition, merging in libarxx means to have a source archive which will be merged into a target archive. So if you have an ARX archive loaded into memory - consider this to be the "old" / "base" / "target" archive - and then merge another ARX archive into it - this is the "merge" / "source" archive - you will in fact change the content of the target archive. In this manner the three-place relation has become a two-place relation.
 * 
 * Especially note that merging changes both the source and the target Arxx::Archive objects. As stated above it it clear that the target Archive has to be changed. In addition to this the source Archive is changed as well! Arxx::Item objects that are added to the target Archive are excluded from the source Archive and re-included into the target Archive. If you want to keep the unchanged source Archive you should take a copy of it before starting the merge operation.
 * 
 * Merging will take place whenever the user calls Arxx::Library::vMerge().
 **/

namespace Arxx
{
	class Archive;
	
	class Item;
	
	/**
	 * @brief Gathers information about a merging process.
	 * 
	 * The context is created to assist the merging process. It is handed to the rules so they can compute their matching according to external conditions provided by the context.
	 **/
	class Context
	{
	public:
		Context(Arxx::Archive & TargetArchive, Arxx::Archive & SourceArchive);
		Arxx::Item & GetCurrentItem(void) const;
		const Arxx::Archive & GetTargetArchive(void) const;
		const Arxx::Archive & GetSourceArchive(void) const;
		void vSetCurrentItem(Arxx::Item & Item);
		void vAddingItem(Arxx::Item & Item);
		void vAddItems(void);
	private:
		/**
		 * @brief The target Archive of the merging process.
		 * 
		 * The target Archive is the Archive that contains the old items. It serves as the base data set and Item objects from the source Archive will replace the Item objects of the target Archive or will be added to it.
		 **/
		Arxx::Archive & m_TargetArchive;
		Arxx::Archive & m_SourceArchive;
		Arxx::Item * m_pCurrentItem;
		std::vector< Arxx::Item * > m_AddingItems;
		std::vector< Arxx::Item * > m_ReplaceItems;
	};
	
	/**
	 * @brief Enumeration of the possible actions a rule can induce.
	 * 
	 * Whenever a Arxx::Rule is matched during a merging process the rule's Arxx::Rule::bMatch() function will be called to determine the action theat is to be taken. The result of that function should be one of the values:
	 * - ADD
	 * - REPLACE
	 * - DROP
	 * - PASS
	 * or else the merging processor will throw an exception.
	 **/
	enum Action
	{
		/**
		 * @brief Signifies that the current item is to be added to the target Archive.
		 * 
		 * The item will be unregistered from the source library and appended to the Arxx::Context::m_AddingItems vector so it can be registered at the target library after all matchings have been performed.
		 **/
		ADD,
		
		/**
		 * @brief Replace the item with the same ID in the target Archive.
		 * 
		 * The item that is to be replaced in the target librarywill be unregistered via Arxx::Archive::vUnregisterItem and the current item will be registered in the taget library.
		 **/
		REPLACE,
		DROP,
		PASS
	};
	
	/**
	 * @brief A class representing a matching rule in the merging process.
	 * 
	 * During the process of merging every item of the source library will be subjected to match against every rule in a rule vector sequentially.
	 **/
	class Rule
	{
	public:
		Rule(Arxx::Action Action) :
			m_Action(Action)
		{
		}
		
		Rule(const Arxx::Rule & Rule) :
			m_Action(Rule.m_Action)
		{
		}
		
		virtual ~Rule(void)
		{
		}
		
		/**
		 * @brief This function returns whether the current item matches in the context.
		 * @param Context The current settings in the merging process.
		 * 
		 * This function is pure virtual and you will need to overwrite it in order to make your rules work. Depending on the output of this function the merger will act:
		 * 
		 * @return true - The merger will call GetAction().
		 * @return false - The merger will simply skip this rule and proceed to the next.
		 **/
		virtual bool bMatch(Arxx::Context & Context) const = 0;
		
		virtual void vAct(Arxx::Context & Context) const
		{
		}
		
		/**
		 * @brief This function is called by the merger whenever a rule is matched.
		 * @param Context The context in which the merger asks for the action.
		 * 
		 * This function serves a two-part objective. It has to call vAct in order to allow the application programmer to specify additional actions after the matching is done and it will return the m_Action member variable.
		 **/
		Arxx::Action GetAction(Arxx::Context & Context) const
		{
			vAct(Context);
			
			return m_Action;
		}
	private:
		Arxx::Action m_Action;
	};
}

#endif
