/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_DATAREPOSITORY_H
#define ARXX_DATAREPOSITORY_H

#include <map>

#include "FetchStatus.h"

/**
 * @file DataRepository.h
 * 
 * This file declares the Arxx::DataRepository class.
 **/

namespace Arxx
{
	class Buffer;
	
	class DataChannel;
	
	class URI;
	
	/**
	 * @defgroup unifieddatasources Unified Data Sources
	 * 
	 * Unified Data Sources define the semantics for data retrieval from any source be it on the local or a remote computer, be it inside an ARX archive or outside. ARX supports the notion that an item's data may come from various sources. This creates a very high degree of flexibility as this notion isn't destroyed or even touched when @link merging Merging @endlink takes place.
	 * 
	 * In libarxx a data source is accessed by means of a data channel. A data channel is an object that is a descendant from Arxx::DataChannel and implements its abstract interface including a function Arxx::DataChannel::vFetchData(Arxx::Item &) which is responsible for retrieving the data from the data source. The retrival logic is completely implemented in a Arxx::DataChannel descendant. This way it is possible to have many various data sources referenced in an ARX archive.
	 * 
	 * When you set an item's data source, you specify compression type, compressed and decompressed data size and the a URI that locates the data source. This URI is translated to a specific data channel by the Arxx::DataRepository, an object that employs URI matching algorithms to identify the most appropriate Arxx::DataChannel that it has registered.
	 **/
	
	/**
	 * @brief This class is a global player helping Arxx::Item and Arxx::Archive objectst to fetch their data.
	 **/
	class DataRepository
	{
	public:
		/**
		 * @brief DataRepository default constructor.
		 **/
		DataRepository(void);
		
		/**
		 * @brief Register a data channel to be used by the fetching algorithm.
		 **/
		bool bRegisterDataChannel(Arxx::DataChannel * pDataChannel);
		
		/**
		 * @brief Unregister a data channel from the repository.
		 **/
		bool bUnregisterDataChannel(Arxx::DataChannel * pDataChannel);
		
		/**
		 * @brief Fetches data from a URI using the most specific DataChannel object.
		 * @param URI The URI used to identify the DataChannel for fetching.
		 * @param Buffer The Buffer to which the data is written upon fetching.
		 * @param FetchStatus This variable will be updated with the status of the fetching process.
		 * @return A boolean value indicating whether the fetch was successful.
		 **/
		bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus);
		
		/**
		 * @brief References the data channel matching the URI.
		 **/
		void vReferenceDataChannel(const Arxx::URI & URI);
		
		/**
		 * @brief Dereferences the data channel matching the URI.
		 **/
		void vDereferenceDataChannel(const Arxx::URI & URI);
		
		/**
		 * @brief Returns whether a channel for this URI is registered in the repository.
		 * @param URI THe URI for which the data channel is searched.
		 * 
		 * This function uses the same matching algorithm as vReferenceDataChannel(), vDereferenceDataChannel() and vFetchData(). It does not check for an exact match.
		 **/
		bool bHasDataChannel(const Arxx::URI & URI);
		
		/**
		 * @brief Returns the DataChannel object matching the given URI.
		 * 
		 * Since the DataChannel object is const this function is only for testing puposes.
		 **/
		const Arxx::DataChannel * pGetDataChannel(const Arxx::URI & URI);
	private:
		/**
		 * @brief Helper function to retrieve the channel object nearest an applicable to the URI.
		 * @param URI The URI for which the lookup and matching is performed.
		 **/
		std::map< Arxx::URI, Arxx::DataChannel * >::iterator iGetDataChannel(const Arxx::URI & URI);
		
		/**
		 * @brief The registered DataChannel objects, mapped by URI.
		 **/
		std::map< Arxx::URI, Arxx::DataChannel * > m_DataChannels;
	};
	
	extern DataRepository Repository;
}

#endif
