/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_DATACHANNEL_H
#define ARXX_DATACHANNEL_H

#include "Buffer.h"
#include "Common.h"
#include "FetchStatus.h"
#include "URI.h"

/**
 * @file DataChannel.h
 * 
 * Defines the interface class DataChannel.
 **/

namespace Arxx
{
	/**
	 * @brief An abstract class defining the interface for data sources.
	 *
	 * Whenever Arxx::Item::vFetchData() is called it will indirectly lead to a call of Arxx::DataChannel::vFetchData() given an appropriate Arxx::DataChannel object is registered at the global Arxx::DataRepository. Appropriate here means that the Arxx::DataChannel's URI is an ancestor of the Arxx::Item's URI. The most appropriate URI is the one that gives the longest match and that one is chosen by the Arxx::DataRepository.
	 *
	 * By derieving from this class it is possible to specify any kind of external data source.
	 *
	 * @todo In order to really support different data sources it is crucial to define some way of asynchron data retrieval.
	 **/
	class DataChannel
	{
	public:
		/**
		 * @brief The default constructor.
		 * @param URI The URI that is handled by this data channel.
		 *
		 * Initializes the use counter @a m_u4UseCounter with 0.
		 * 
		 * Initializes @a m_bWantsAutoDeletion with false.
		 **/
		DataChannel(const Arxx::URI & URI);
		
		/**
		 * @brief A virtual destructor for there is a virtual function.
		 * 
		 * Will write an error to std::cerr if the reference counter @em m_u4ReferenceCounter is not 0.
		 **/
		virtual ~DataChannel(void);
		
		/**
		 * @brief Adds a reference.
		 * 
		 * Is called by the Arxx::DataRepository whenever some component references a URI that matches closest with @a m_URI.
		 **/
		void vReference(void);
		
		/**
		 * @brief Removes a reference.
		 * 
		 * Is called by the Arxx::DataRepository whenever some component dereferences a URI that matches closest with @a m_URI.
		 **/
		void vDereference(void);
		
		/**
		 * @brief Returns whether the DataChannel is referenced at all.
		 * 
		 * If there is at least one reference remaining this function returns true.
		 **/
		bool bIsReferenced(void) const;
		
		/**
		 * @brief Returns @a m_URI.
		 **/
		const Arxx::URI & GetURI(void) const;
		
		/**
		 * @brief This abstract function is responsible for retrieving the data from the data source and writing it into a Buffers::Buffer.
		 * @param URI The URI that identifies the data location to be retrieved.
		 * @param Buffer The Buffer that the retrieved data should be written into.
		 * @param FetchStatus This variable takes the status of the fetching process.
		 * @return A boolean value indicating whether the fetch was successful.
		 * 
		 * When this function is called the derieved class is supposed to get the data from the external source and write it into the @a Buffer.
		 **/
		virtual bool bFetchData(const Arxx::URI & URI, Arxx::Buffer & Buffer, Arxx::FetchStatus & FetchStatus) = 0;
		
		/**
		 * @brief Returns the @a m_bWantsAutoDeletion member.
		 * @sa m_bWantsAutoDeletion.
		 **/
		bool bWantsAutoDeletion(void) const;
	protected:
		/**
		 * @brief Function that is called every time the reference counter goes to zero.
		 * 
		 * The default implementation of this function is intentionally empty.
		 **/
		virtual void vOnUnreferenced(void);
		
		/**
		 * @brief Derieved classes can set this flag true to indicate auto deletion to the data repository.
		 * 
		 * Upon unreferencing a data channel the data repository will call bWantsAutoDeletion() which returns this value.
		 * 
		 * Either set it in the class constructor if you want class wide auto deletion or add an initializer to the constructor to allow different values per instance.
		 **/
		bool m_bWantsAutoDeletion;
	private:
		/**
		 * @brief Hidden copy constructor.
		 **/
		DataChannel(const Arxx::DataChannel & DataChannel);
		
		/**
		 * @brief Hidden assignment operator.
		 **/
		Arxx::DataChannel & operator=(const Arxx::DataChannel & DataChannel);
		
		/**
		 * @brief The channel's reference counter.
		 * 
		 * Every component having an interest in data that is to be fetched using this data channel will increment this reference counter by calling Arxx::DataChannel::vReference(). The referencing compoenents are responsible to dereference the data channel once they have no use of it anymore.
		 **/
		Arxx::u4byte m_u4ReferenceCounter;
		
		/**
		 * @brief The URI that this channel is applied to.
		 **/
		const Arxx::URI m_URI;
	};
}

#endif
