/**
 * libarxx - Advanced Resource files in C++
 * Copyright (C) 2005  Hagen Möbius
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#ifndef ARXX_DATA_H
#define ARXX_DATA_H

#include "Buffer.h"
#include "Common.h"
#include "FetchStatus.h"
#include "URI.h"

/**
 * @file Data.h
 * 
 * Declares the interface of Arxx::Data.
 **/

namespace Arxx
{
	/**
	 * @brief A buffer with compression, decompression and external data referencing features.
	 *
	 * This is the component underlying the Arxx::Item class. It delivers the ability to compress and decompress the data. Because of the public inhertitance from Arxx::Buffer it is still possible to call Buffer::stGetLength() but be aware that it will give you the state of the buffer which might not be what you want for two reasons:
	 * - the buffer content might be compressed or decompressed
	 * - the data might not be fetched yet, which makes Arxx::Buffer::stGetLength() return `0` since there is nothing in the buffer
	 * - the data might not be fetched yet, but you have written something in the buffer. Arxx::Buffer::stGetLength() will of course return the length of that data instead of the length of the Item's data, be it compressed or decompressed.
	 **/
	class Data : public Arxx::Buffer
	{
	public:
		typedef enum
		{
			NONE = 0,
			ZLIB_0 = 1,
			ZLIB_1 = 2,
			ZLIB_2 = 3,
			ZLIB_3 = 4,
			ZLIB_4 = 5,
			ZLIB_5 = 6,
			ZLIB_6 = 7,
			ZLIB_7 = 8,
			ZLIB_8 = 9,
			ZLIB_9 = 10,
			BZLIB = 11
		} Compression;
		
		static Arxx::Data::Compression m_DefaultCompression;
		
		/**
		 * @brief The default constructor for a Arxx::Data.
		 **/
		Data(void);
		
		/**
		 * @brief The destructor of the Arxx::Data.
		 **/
		~Data(void);
		
		/**
		 * @brief Decompresses the data of the buffer.
		 **/
		void vDecompress(void);
		
		/**
		 * @brief Compresses the data of the buffer.
		 **/
		void vCompress(const Arxx::Data::Compression & Compression = Arxx::Data::m_DefaultCompression);
		
		/**
		 * @brief Returns true if the data inside the buffer is compressed.
		 **/
		bool bIsCompressed(void) const;
		
		/**
		 * @brief Returns true if the data inside the buffer is decompressed.
		 **/
		bool bIsDecompressed(void) const;
		
		/**
		 * @brief Returns the compression information, which will equal the compression rate of the zlib library.
		 **/
		Arxx::Data::Compression GetCompression(void) const;
		
		/**
		 * @brief Gets the length of the decompressed data.
		 **/
		Arxx::u4byte u4GetDecompressedLength(void) const;
		
		/**
		 * @brief Gets the length of the compressed data.
		 **/
		Arxx::u4byte u4GetCompressedLength(void) const;
		
		/**
		 * @brief Returns the internal locator used by the external data fetcher.
		 * 
		 * Whether this URI is of importance or not is for the fetcher object to decide.
		 **/
		const Arxx::URI & GetURI(void) const;
		
		/**
		 * @brief This function requests the item's data at the appropriate data channel.
		 * @return Wether the request to fetch the data could be handed to a data channel.
		 * 
		 * Note that fetching the data does not automatically internalize it.
		 * 
		 * In order to only try to fetch the data m_URI must be valid and bIsFetched() and bIsFetching() must be false. Otherwise the function will return false.
		 * 
		 * For asynchronous fetches the return value may be true because an Arxx::DataChannel could be found but the actuall fetching may fail. This information may be retrieved via Arxx::Data::GetFetchStatus().
		 **/
		bool bFetch(void);
		
		/**
		 * @brief Unfetching the data generally means deleting the data thus releasing the memory in the buffer.
		 * 
		 * This function will empty the data buffer and release the memory associated with it. So why an extra function if this could be done with Buffers::Buffer::vDelete()?
		 * 
		 * Because if you vDelete the content of the buffer you change the data. Hence you don't change the flag whether the data is fetched and will not be able the refetch the data using vFetch() again.
		 * 
		 * This function requires the bIsFetched() to be true.
		 * 
		 * This function sets @a m_FetchStatus to Arxx::UNFETCHED.
		 **/
		void vUnfetch(void);
		
		/**
		 * @brief Use this function to release the data source.
		 * 
		 * Using this function you can dereference the associated data fetcher thus making it impossible to fetch or refetch the item's data.
		 *
		 * It will also set the m_URI to be invalid.
		 * 
		 * This function does NOT internalize the data, though. It's just meant to release the data channel to allow the releasing of resources associated with the data channel as the name quite plaily suggests.
		 **/
		void vReleaseDataChannel(void);
		
		/**
		 * @brief Sets fake data to the buffer.
		 * @param Compression The state of the data. If this value if greater than 0 the fake data is considered compressed.
		 * @param u4DecompressedLength The length of the fake data if in decompressed state. Elsewise the value which Arxx::Buffer::u4GetDecompressedLength() will return and this data will serve no internal functionality, so you may use it as you like.
		 * @param u4CompressedLength The length of the fake data if in compressed state. Elsewise the value which Arxx::Buffer::u4GetCompressedDataLength() will return and this data will serve no internal functionality, so you may use it as you like.
		 * @param URI The URI of the external data.
		 * 
		 * This function will dereference any old @a m_URI and reference the new @a URI at the Arxx::DataRepository.
		 * 
		 * This function will set m_bIsExternal to true.
		 * 
		 * This function will set @a m_FetchStatus to Arxx::UNFETCHED.
		 **/
		virtual void vSetExternal(const Arxx::URI & URI = "", const Arxx::Data::Compression & Compression = Arxx::Data::NONE, Arxx::u4byte u4DecompressedLength = 0, Arxx::u4byte u4CompressedLength = 0);
		
		/**
		 * @brief Returns whether the data is meant to be stored internally.
		 * 
		 * Equals !bIsExternal().
		 **/
		bool bIsInternal(void) const;
		
		/**
		 * @brief Returns whether the data is external and meant to be referenced by URI.
		 * 
		 * Equals !bIsInternal().
		 **/
		bool bIsExternal(void) const;
		
		/**
		 * @brief Use this function to change the data from external to internal.
		 * 
		 * This function will set the m_bIsExternal flag to false.
		 **/
		void vInternalize(void);
		
		/**
		 * @brief Returns the fetch status of the data.
		 * 
		 * For a normal data buffer this will be Arxx::FETCHED but generally every status in Arxx::FetchStatus is used.
		 **/
		Arxx::FetchStatus GetFetchStatus(void) const;
		
		/**
		 * @brief Returns wether a fetching process is in progress.
		 **/
		bool bIsFetching(void) const;
		
		/**
		 * @brief Returns wether the data is fetched.
		 **/
		bool bIsFetched(void) const;
	private:
		/**
		 * @brief A locator used to further describe the location to the fetcher.
		 * 
		 * After creating an Item this value is intentionally invalid.
		 **/
		Arxx::URI m_URI;
		
		/**
		 * @brief The compressed state the data is currently in.
		 * 
		 * Is set to either 0 for to indicate no compression or some value greater than 0 to indicate the compression level of the data as in the zlib library.
		 **/
		Arxx::Data::Compression m_Compression;
		
		/**
		 * @brief The length of the data if decompressed.
		 **/
		mutable Arxx::u4byte m_u4DecompressedLength;
		
		/**
		 * @brief The length of the data if compressed or 0.
		 * 
		 * Since the compressed size is not known as long as the data is decompressed this value is 0 whenever the data is decompressed.
		 **/
		mutable Arxx::u4byte m_u4CompressedLength;
		
		/**
		 * @brief A flag indicating whether the data is meant to be internal or external.
		 * 
		 * This flag is changed by vSetExternalData(), vInternalize(). It can be queried using bIsExternal() or bIsInternal().
		 * 
		 * When saving an Arxx::Archive this flag decides whether the data or the URI reference is stored in the archive.
		 * 
		 * After creating an Item this value is intentionally false.
		 **/
		bool m_bIsExternal;
		
		/**
		 * @brief A status indicator showing the fetch status of the data.
		 * 
		 * It can be queried using GetFetchStatus().
		 * 
		 * After creating an Item this value is intentionally Arxx::FETCHED.
		 **/
		Arxx::FetchStatus m_FetchStatus;
	};
}

#endif
