/* -*- mode: c++; c-basic-offset: 2; -*- */

#include <ldas_tools_config.h>

#include <signal.h>
#include <pthread.h>
#include <unistd.h>

#include <cassert>

#include <iostream>
#include <sstream>
#include <stdexcept>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/IOLock.hh"
#include "ldastoolsal/unordered_map.hh"
#include "ldastoolsal/ldasexception.hh"
#include "ldastoolsal/Thread.hh"

#include "Thread.icc"

#if ALT_PTHREAD_STACK_MIN
#undef PTHREAD_STACK_MIN
#define PTHREAD_STACK_MIN ALT_PTHREAD_STACK_MIN
#endif /* ALT_PTHREAD_STACK_MIN */

using LDASTools::AL::Thread;
using LDASTools::AL::AtExit;
using LDASTools::AL::MutexLock;
using LDASTools::AL::unordered_map;

namespace {
  static size_t				stack_size = PTHREAD_STACK_MIN;

  static MutexLock::baton_type
  stack_size_baton( )
  {
    static MutexLock::baton_type	baton;

    return baton;
  }
}

static void
wakeup( int SignalNumber )
{
  ::signal( SignalNumber, wakeup );
}

class thread_registry
{
public:
  typedef Thread::thread_type::handle_type	key_type;
  typedef Thread* value_type;

  static void Add( key_type, value_type );

  static value_type Get( key_type );

  static inline key_type Self( )
  {
    return pthread_self( );
  }

  static void Remove( key_type );

private:
  friend class Thread::Self;

  typedef unordered_map< key_type, value_type
#if NEED_PTHREAD_T_HASH_FUNC
			 , LDASTools::AL::hash< const void* >
#endif /* NEEDED_PTHREAD__T_HASH_FUNC */
			 > thread_registry_type;

  inline static thread_registry_type
  storage( )
  {
    static thread_registry_type    m_storage;

    return m_storage;
  }

  inline static MutexLock::baton_type
  baton( )
  {
    static MutexLock::baton_type baton;

    return baton;
  }

  static value_type get( key_type );
};

void thread_registry::
Add( thread_registry::key_type Key, thread_registry::value_type Value )
{
  MutexLock	l( baton( ),
		   __FILE__, __LINE__ );

  storage( )[ Key ] = Value;
}

thread_registry::value_type thread_registry::
Get( thread_registry::key_type Key )
{
  MutexLock	l( baton( ),
		   __FILE__, __LINE__ );

  return get( Key );
}

void thread_registry::
Remove( thread_registry::key_type Key )
{
  MutexLock	l( baton( ),
		   __FILE__, __LINE__ );

  storage( ).erase( Key );
}

thread_registry::value_type thread_registry::
get( thread_registry::key_type Key )
{
  thread_registry_type::const_iterator pos( storage( ).find( Key ) );
  if ( pos == storage( ).end( ) )
  {
    std::ostringstream	msg;
    msg << "There is no managed thread with key: " << Key;

    throw std::range_error( msg.str( ) );
  }
  return pos->second;
}

namespace LDASTools
{
  namespace AL
  {
    //-------------------------------------------------------------------
    // Thread::Self class
    //-------------------------------------------------------------------

    void Thread::Self::
    Cancel( )
    {
      thread_registry::value_type v = NULL;
      try
      {
	v = thread_registry::Get( thread_registry::Self( ) );

      }
      catch( ... )
      {
      }
      if ( v )
      {
	v->Cancel( );
      }
    }

    void Thread::Self::
    CancellationCheck( const std::string& Header,
		       const char* File, const int Line )
    {
      thread_registry::value_type v = NULL;
      try
      {
	v = thread_registry::Get( thread_registry::Self( ) );

      }
      catch( ... )
      {
      }
      if ( v )
      {
	v->CancellationCheck( Header, File, Line );
      }
    }

    Thread::cancel_type Thread::Self::
    CancellationType( int& AuxInfo )
    {
      thread_registry::value_type v = NULL;

      try
      {
	v = thread_registry::Get( thread_registry::Self( ) );

      }
      catch( ... )
      {
      }
      if ( v )
      {
	return v->CancellationType( AuxInfo );
      }
      return CANCEL_EXCEPTION;
    }

    void Thread::Self::
    CancellationType( cancel_type Type, int AuxInfo )
    {
      thread_registry::value_type v = NULL;

      try
      {
	v = thread_registry::Get( thread_registry::Self( ) );

      }
      catch( ... )
      {
      }
      if ( v )
      {
	v->CancellationType( Type, AuxInfo );
	v->CancellationEnable( ( Type != CANCEL_IGNORE ) );
      }
    }

    Thread::thread_type Thread::Self::
    Id( )
    {
      return thread_registry::Self( );
    }
    //---------------------------------------------------------------------
    // Thread class
    //---------------------------------------------------------------------
    Thread::
    Thread( )
      : p( new impl ),
	m_cancel_via_signal( 0 ),
	m_cancel_state( false ),
	m_cancel_thread( false ),
	m_cancellation_type( CANCEL_IGNORE )
    {
    }

    Thread::
    ~Thread( )
    {
      Join( );
    }

#if WORKING
    bool Thread::
    operator==( const Thread& Source ) const
    {
      return( Source.m_tid= m_tid );
    }
#endif /* WORKING */

    void Thread::
    Cancel( )
    {
      if ( p->tid )
      {
	MutexLock	l( p->key, __FILE__, __LINE__ );
    
	m_cancel_thread = true;
	switch( m_cancellation_type )
	{
	case CANCEL_EXCEPTION:
	case CANCEL_IGNORE:
	  break;
	case CANCEL_BY_SIGNAL:
	  {
	    int sig( m_cancel_via_signal );

	    if ( sig > 0 )
	    {
	      l.Release( __FILE__, __LINE__ );
	      Kill( sig );
	    }
	  }
	  break;
	case CANCEL_ASYNCHRONOUS:
	case CANCEL_DEFERRED:
	  {
	    thread_type	t( p->tid );

	    l.Release( __FILE__, __LINE__ );
	    pthread_cancel( t );
	  }
	  break;
	}
      }
    }

    //---------------------------------------------------------------------
    /// Deliver any pending cancellation requests to the calling thread.
    /// If cancellation is to be done via exception,
    /// then a cancellation exception is throw.
    /// If not, then a call to the appropriate thread library's
    /// cancellation routine is made.
    ///
    /// \note
    ///     This call must only be made by a child thread.
    //---------------------------------------------------------------------
    void Thread::
    CancellationCheck( const std::string& Header,
		       const char* File, const int Line ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );
    
      if ( ! pthread_equal( p->tid, pthread_self( ) ) )
      {
	return;
      }
      if ( m_cancel_thread )
      {
	switch( m_cancellation_type )
	{
	case CANCEL_IGNORE:
	  break;
	case CANCEL_BY_SIGNAL:
	case CANCEL_EXCEPTION:
	  throw cancellation( Header, File, Line );
	  break;
	case CANCEL_ASYNCHRONOUS:
	case CANCEL_DEFERRED:
	  pthread_testcancel( );
	  break;
	}
      }
    }

    void Thread::
    CancellationEnable( bool Value )
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );
    
      cancellation_enable( Value );
    }

    Thread::cancel_type Thread::
    CancellationType( int& AuxInfo ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );

      switch( m_cancellation_type )
      {
      case CANCEL_BY_SIGNAL:
	AuxInfo = m_cancel_via_signal;
	break;
      default:
	AuxInfo = 0;
	break;
      }
      return m_cancellation_type;
    }

    bool Thread::
    IsCancelled( ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );

      return m_cancel_thread;
    }

    inline size_t Thread::
    StackSizeDefault( )
    {
      MutexLock	l( stack_size_baton( ), __FILE__, __LINE__ );

      return stack_size;
    }

    //---------------------------------------------------------------------
    /// Establish the cancellability of a thread.
    /// If a Value of true is passed, then the thread will allow
    /// cancellation as determined by the call to CancellationType.
    ///
    /// \note
    ///     This call must only be made by a child thread.
    ///
    /// \see
    ///     CancellationType
    //---------------------------------------------------------------------
    void Thread::
    cancellation_enable( bool Value )
    {
      if ( ! pthread_equal( p->tid, pthread_self( ) ) )
      {
	return;
      }
      int oldstate;

      pthread_setcancelstate( PTHREAD_CANCEL_DISABLE, &oldstate );
      m_cancel_state = Value;
      if ( m_cancel_state )
      {
	switch( m_cancellation_type )
	{
	case CANCEL_IGNORE:
	case CANCEL_EXCEPTION:
	  // Nothing to do to setup for this one
	  break;
	case CANCEL_BY_SIGNAL:
	  if ( m_cancel_via_signal > 0 )
	  {
	    //---------------------------------------------------------
	    // Make sure the signal handler is registered
	    //---------------------------------------------------------
	    wakeup( m_cancel_via_signal );
	    //---------------------------------------------------------
	    // Setup to cancel via signal.
	    //---------------------------------------------------------
	    sigset_t	signal_set;
	    sigset_t	old_signal_set;

	    sigemptyset( &signal_set );
	    sigaddset( &signal_set, m_cancel_via_signal );

	    //---------------------------------------------------------
	    // Preserve the old signal mask
	    //---------------------------------------------------------
	    pthread_sigmask( SIG_SETMASK, NULL, &old_signal_set );
	    //---------------------------------------------------------
	    // Set the new signal mask
	    //---------------------------------------------------------
	    pthread_sigmask( SIG_UNBLOCK, &signal_set, NULL );
	  }
	  break;
	case CANCEL_ASYNCHRONOUS:
	case CANCEL_DEFERRED:
	  pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, &oldstate );
	  break;
	}
      }
      else
      {
	switch( m_cancellation_type )
	{
	case CANCEL_IGNORE:
	case CANCEL_EXCEPTION:
	  // Nothing to do to setup for this one
	  break;
	case CANCEL_BY_SIGNAL:
	  if ( m_cancel_via_signal > 0 )
	  {
	    //-------------------------------------------------------------
	    // Make sure the signal handler is registered
	    //-------------------------------------------------------------
	    wakeup( m_cancel_via_signal );
	    //-------------------------------------------------------------
	    // Setup to cancel via signal.
	    //-------------------------------------------------------------
	    sigset_t	signal_set;
	    sigset_t	old_signal_set;

	    sigemptyset( &signal_set );
	    sigaddset( &signal_set, m_cancel_via_signal );

	    //---------------------------------------------------------
	    // Preserve the old signal mask
	    //---------------------------------------------------------
	    pthread_sigmask( SIG_SETMASK, NULL, &old_signal_set );
	    //---------------------------------------------------------
	    // Set the new signal mask
	    //---------------------------------------------------------
	    pthread_sigmask( SIG_BLOCK, &signal_set, NULL );
	  }
	  break;
	case CANCEL_ASYNCHRONOUS:
	case CANCEL_DEFERRED:
	  pthread_setcancelstate( PTHREAD_CANCEL_DISABLE, &oldstate );
	  break;
	}
      }
    
    }

    void Thread::
    set_cancelled_state( )
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );

      m_cancel_thread = true;
    }

    inline Thread::thread_type Thread::
    threadId( ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );

      return p->tid;
    }

    //---------------------------------------------------------------------
    /// Sets a how a thread can be cancelled.
    /// <ul>
    ///   <li><b>CANCEL_ASYNCHRONOUS</b>
    ///       A cancellation request is immediated delivered to the
    ///       thread.
    ///   <li><b>CANCEL_DEFERRED</b>
    ///       A cancellation request is marked pending for the thread
    ///       and the thread is cancelled when it gets to a system
    ///       cancellation point or when CancellationCheck is called.
    ///   <li><b>CANCEL_EXCEPTION</b>
    ///       A cancellation request is marked pending for the thread
    ///       and the thread throws a cancellation exception upon
    ///       calling CancellationCheck.
    /// </ul>
    ///
    /// \note
    ///     This call must only be made by a chivld thread.
    ///
    /// \see
    ///     CancellationEnable, CancellationCheck
    //---------------------------------------------------------------------
    void Thread::
    CancellationType( cancel_type Type, int AuxInfo )
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );
    
      if ( ( ! pthread_equal( p->tid, pthread_self( ) ) )
	   || ( ( Type != CANCEL_BY_SIGNAL )
		&& ( m_cancellation_type == Type ) )
	   || ( ( Type == CANCEL_BY_SIGNAL )
		&& ( m_cancellation_type == Type )
		&& ( AuxInfo == m_cancel_via_signal ) ) )
      {
	return;
      }
      int oldtype;
      int oldstate;

      pthread_setcancelstate( PTHREAD_CANCEL_DISABLE, &oldstate );
      m_cancel_via_signal = 0;
      m_cancellation_type = Type;
      switch( m_cancellation_type )
      {
      case CANCEL_ASYNCHRONOUS:
	pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype );
	break;
      case CANCEL_DEFERRED:
	pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, &oldtype );
	break;
      case CANCEL_BY_SIGNAL:
	m_cancel_via_signal = AuxInfo;
	break;
      case CANCEL_EXCEPTION:
      case CANCEL_IGNORE:
	return;
	break;
      }
      //-------------------------------------------------------------------
      // Restore the state of cancellation
      //-------------------------------------------------------------------
      cancellation_enable( m_cancel_state );
    }

    bool Thread::
    IsAlive( ) const
    {
      //-------------------------------------------------------------------
      // Preliminary checks of the thread.
      //-------------------------------------------------------------------
      if ( IsCancelled( )
	   && IsParentThread( ) )
      {
	try
	{
	  //---------------------------------------------------------------
	  // Thread has been cancelled; make sure the thread has been
	  // reaped.
	  //---------------------------------------------------------------
	  const_cast< Thread* >(this)->Join( );
	}
	catch( ... )
	{
	}
    
      }
      //-------------------------------------------------------------------
      // Check status by trying to send it signal zero.
      // This signal does not disturb the process in any way.
      // Just want to determine if the process is still there.
      //-------------------------------------------------------------------
      return ( Kill( 0 ) == 0 );
    }

    bool Thread::
    IsParentThread( ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );
    
      return ( pthread_equal( pthread_self( ),
			      p->parent_tid ) );
    }

    void Thread::
    Detach( ) const
    {
      if ( p->tid )
      {
#if WORKING
	thread_registry::Remove( p->tid );
#endif /* WORKING */
	// Detach the current thread (makes it non-joinable).
	volatile Thread::thread_type tmp = p->tid;

	{
	  MutexLock	l( p->key, __FILE__, __LINE__ );

#if WORKING
	  p->tid.Reset( );
#endif /* WORKING */
	}
	// thread_registry( ).erase( p->tid );
	pthread_detach( tmp.handle );
      }
    }

    void Thread::
    Join( )
    {
      if ( p->tid )
      {
	bool release( true );

	int status = pthread_join( p->tid, NULL );
	try
	{
	  switch( status )
	  {
	  case ESRCH:
	    throw range_error( "Thread::Join", __FILE__, __LINE__ );
	    break;
	  case EINVAL: 
	    throw invalid_argument( "Thread::Join", __FILE__, __LINE__ );
	    break;
	  case EDEADLK: 
	    release = false;
	    throw deadlock( "Thead::Join", __FILE__, __LINE__ );
	    break;
	  }
	}
	catch( ... )
	{
	  if ( release )
	  {
	    // get_thread_registry( ).erase( p->tid );
	  
	    MutexLock	l( p->key, __FILE__, __LINE__ );
	    p->tid = Thread::thread_type::NULL_THREAD;
	  }
	  throw; // rethrow the exception
	}
	{
	  // get_thread_registry( ).erase( p->tid );

	  MutexLock	l( p->key, __FILE__, __LINE__ );
	  p->tid = Thread::thread_type::NULL_THREAD;
	}
      }
    }

    int Thread::
    Kill( int Signal ) const
    {
      MutexLock	l( p->key, __FILE__, __LINE__ );
    
      if ( p->tid )
      {
	return pthread_kill( p->tid, Signal );
      }
      // Process could not be found since it is null
      return ESRCH;
    }

    inline Thread::thread_type Thread::
    ParentThread( ) const
    {
      return p->parent_tid;
    }

    int Thread::
    Spawn( )
    {
      return spawn( );
    }

    void Thread::
    cancelCleanup( Thread* Source )
    {
      if ( Source )
      {
	Source->set_cancelled_state( );
      }
    }

    //-------------------------------------------------------------
    /// Establish the default value of the stack size used when
    /// creating new threads.
    /// This routine does validate the requested value..
    //-------------------------------------------------------------
    void Thread::
    StackSizeDefault( size_t StackSize )
    {
      int			ret = 0;
      pthread_attr_t	attr;

      pthread_attr_init( &attr );
      if ( StackSize > 0 )
      {
	(void)pthread_attr_setstacksize( &attr, StackSize );
	ret = pthread_attr_getstacksize( &attr, &StackSize );
      }
      else
      {
	StackSize = 0;
      }
      pthread_attr_destroy( &attr );
    
      if ( ret == 0 )
      {
	MutexLock	l( stack_size_baton( ), __FILE__, __LINE__ );

	stack_size = StackSize;
      }
    }

    int Thread::
    spawn( start_function_type StartFunction )
    {
      int retval = 0;
      if ( ! p->tid )
      {

	pthread_attr_t attr;
	pthread_attr_init (&attr);
	if ( StackSizeDefault( ) > 0 )
	{
	  pthread_attr_setstacksize ( &attr, StackSizeDefault( ) );
	}

	retval = pthread_create( p->tid, &attr, StartFunction, this );
	pthread_attr_destroy (&attr);

	if ( retval != 0 )
	{
	  throw std::runtime_error( "Thread::Start: failed to start thread" );
	}
      }
      return retval;
    }

    Thread* Thread::
    start_routine( Thread* ThreadSource )
    {
      if ( AtExit::IsExiting( ) == false )
      {
	try
	{
	  thread_registry::Add( thread_registry::Self( ), ThreadSource );

	  try
	  {
	    ThreadSource->CancellationType( CANCEL_EXCEPTION );
	    ThreadSource->CancellationEnable( true );
	    ThreadSource->action( );
#if 0
	    throw std::runtime_error( "Bad action" );
#endif /* 0 */
	  }
	  catch( const std::exception& exception )
	  {
	    if ( ThreadSource->IsCancelled( ) )
	    {
	      int aux_info;

	      switch( ThreadSource->CancellationType( aux_info ) )
	      {
	      case CANCEL_ASYNCHRONOUS:
	      case CANCEL_DEFERRED:
		thread_registry::Remove( thread_registry::Self( ) );
		throw;	// Rethrow the cancelation exception
		break;
	      default:
		break;
	      }
	    }
	  }

	  thread_registry::Remove( thread_registry::Self( ) );
	  return ThreadSource;
	}
	catch( const std::exception& Exception )
	{
	  std::cerr << "Thread Exception: " << Exception.what( )
		    << std::endl
	    ;
	  assert( 0 );
	}
      }
      return (Thread*)NULL;
    }

    Thread::cancellation::
    cancellation( const std::string& Header,
		  const char* File, const int Line )
      : std::runtime_error( format( Header, File, Line ) )
    {
    }

    std::string Thread::cancellation::
    format( const std::string& Header,
	    const char* File, const int Line )
    {
      std::ostringstream	msg;

      msg << Header << ": thread cancellation."
	  << " (File: " << File << " Line: " << Line << ")"
	;
      return msg.str( );
    }

    Thread::deadlock::
    deadlock( const std::string& Header,
	      const char* File, const int Line )
      : std::logic_error( format( Header, File, Line ) )
    {
    }

    std::string Thread::deadlock::
    format( const std::string& Header,
	    const char* File, const int Line )
    {
      std::ostringstream	msg;

      msg << Header << ": attempt to join with self."
	  << " (File: " << File << " Line: " << Line << ")"
	;
      return msg.str( );
    }

    Thread::invalid_argument::
    invalid_argument( const std::string& Header,
		      const char* File, const int Line )
      : std::invalid_argument( format( Header, File, Line ) )
    {
    }

    std::string Thread::invalid_argument::
    format( const std::string& Header,
	    const char* File, const int Line )
    {
      std::ostringstream	msg;

      msg << Header << ": invalid argument."
	  << " (File: " << File << " Line: " << Line << ")"
	;
      return msg.str( );
    }

    Thread::range_error::
    range_error( const std::string& Header,
		 const char* File, const int Line )
      : std::range_error( format( Header, File, Line ) )
    {
    }

    std::string Thread::range_error::
    format( const std::string& Header,
	    const char* File, const int Line )
    {
      std::ostringstream	msg;

      msg << Header << ": no thread could be found."
	  << " (File: " << File << " Line: " << Line << ")"
	;
      return msg.str( );
    }
  } // namespace - AL
} // namespace - LDASTools
