/* -*- mode: c++; c-basic-offset: 2; -*- */

#include <time.h>
#ifndef CLOCK_REALTIME
#include <sys/time.h>
#endif

#include <iostream>
#include <stdexcept>

#include "ldastoolsal/ConditionalVariable.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/System.hh"

#include "MutexLockBaton.cc"

using LDASTools::System::ErrnoMessage;

namespace LDASTools
{
  namespace AL
  {
    class ConditionalVariable::impl
    {
    public:
      typedef pthread_cond_t cond_type;

      class timeout_error
	: public std::runtime_error
      {
      public:
	timeout_error( )
	  : runtime_error( init( ).c_str( ) )
	{
	}

      private:
	static std::string init( )
	{
	  std::ostringstream	msg;

	  msg << "timeout error: "
	      << ErrnoMessage( )
	    ;
	  return msg.str( );
	}
      };

      inline
      impl( )
	: baton( __FILE__, __LINE__ )
      {
	pthread_cond_init( &var, NULL );
      }

      inline
      ~impl( )
      {
	pthread_cond_destroy( &var );
      }

      inline void
      Broadcast( )
      {
	pthread_cond_broadcast( &var );
      }

      inline void
      Signal( )
      {
	pthread_cond_signal( &var );
      }

      inline void
      TimedWait( INT_4U Seconds )
      {
	struct timespec	ts;

#if defined( CLOCK_REALTIME )
	clock_gettime( CLOCK_REALTIME, &ts );
#else
	timeval	tv;

	gettimeofday( &tv, NULL );

	ts.tv_sec = tv.tv_sec;
	ts.tv_nsec = 1000 * tv.tv_usec;
#endif /* defined( CLOCK_REALTIME ) */
	ts.tv_sec += Seconds;

	int rc = pthread_cond_timedwait( &var, baton.pimpl_->ref( ), &ts );
	switch( rc )
	{
	case 0:
	  // No error
	  break;
	case ETIMEDOUT:
	  break;
	default:
	  {
	    std::ostringstream	msg;

	    msg << "ConditionalVariable::TimedWait: system error: "
		<< ErrnoMessage( )
	      ;
	    throw std::runtime_error( msg.str( ) );
	  }
	  break;
	}
      }

      inline void
      Wait( )
      {
	pthread_cond_wait( &var, baton.pimpl_->ref( ) );
      }

      cond_type 			var;
      MutexLock::baton_type	baton;
    };

    ConditionalVariable::
    ConditionalVariable( )
      : pimpl_( new impl( ) )
    {
    }

    ConditionalVariable::
    ~ConditionalVariable( )
    {
    }

    MutexLock::baton_type ConditionalVariable::
    Mutex( )
    {
      return pimpl_->baton;
    }

    void ConditionalVariable::
    Broadcast( )
    {
      pimpl_->Broadcast( );
    }

    void ConditionalVariable::
    Release( )
    {
      pimpl_->baton.pimpl_->unlock( );
    }

    void ConditionalVariable::
    Signal( )
    {
      pimpl_->Signal( );
    }

    bool ConditionalVariable::
    TimedWait( INT_4U Seconds )
    {
      try
      {
	pimpl_->TimedWait( Seconds );
      }
      catch( const impl::timeout_error& Exception )
      {
	return false;
      }
      catch( const std::exception& Exception )
      {
	throw;
      }
      return false;
    }

    void ConditionalVariable::
    Wait( )
    {
      pimpl_->Wait( );
    }
  } // namespace - AL
} // namespace - LDASTools
