#include <ldas_tools_config.h>

#include <sys/types.h>
#include <unistd.h>

#include <sstream>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/SharedPtr.hh"
#include "ldastoolsal/unordered_map.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/LogHTML.hh"
#include "genericAPI/Symbols.hh"

using LDASTools::AL::AtExit;
using LDASTools::AL::MutexLock;
using LDASTools::AL::unordered_map;

using GenericAPI::log_type;

static log_type find_log( const std::string& Filename );

typedef unordered_map< std::string, log_type::element_type::message_type> group_map_type;

static group_map_type& group_map_init( );

static group_map_type& group_map = group_map_init( );

static MutexLock::baton_type	logs_baton;

static log_type	default_log( new GenericAPI::LogHTML( "" ) );

static std::string log_default_filename;

namespace GenericAPI
{
  bool
  IsLogging( const int Group,
	     const int Level,
	     const std::string& Filename )
  {
    if ( AtExit::IsExiting( ) == true )
    {
      return false;
    }
    log_type	log = find_log( Filename );

    return log->VerbosityCheck( Group, Level );
  }

  const std::string&
  LogFileDefault( )
  {
    return log_default_filename;
  }

  void
  LogFileDefault( const std::string& LogFilename )
  {
    log_default_filename = LogFilename;
  }

  void
  queueLogEntry( const std::string& Message,
		 const std::string& Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time,
		 const std::string& Filename )
  {
    log_type::element_type::message_type mt = group_map[ std::string( Group ) ];

    queueLogEntry( Message, mt, Level, Caller, JobInfo, Time, Filename );
  }

  void
  queueLogEntry( const std::string& Message,
		 const int Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time,
		 const std::string& Filename )
  {
    if ( AtExit::IsExiting( ) == true )
    {
      return;
    }
    log_type	log = find_log( std::string( Filename ) );
    std::string	job_info( log->FormatJobInfo( JobInfo ) );

    //-------------------------------------------------------------------
    // Queue the message.
    //-------------------------------------------------------------------
    {
      MutexLock	l( logs_baton,
		   __FILE__, __LINE__ );

      if ( AtExit::IsExiting( ) == false )
      {
	log->Message( log_type::element_type::message_type( Group ),
		      log_type::element_type::level_type( Level ),
		      Caller, job_info, Message );
      }
    }
  }

  void
  setLogDebugLevel( const int Level,
		    const std::string& Filename )
  {
    log_type	log = find_log( std::string( Filename ) );
    
    {
      MutexLock	l( logs_baton,
		   __FILE__, __LINE__ );

      if ( AtExit::IsExiting( ) == false )
      {
	log->Verbosity( log_type::element_type::MT_DEBUG, Level );
      }
    }
  }

  void
  setLogOutputDirectory( const std::string& OutputDirectory ) 
  {
    if ( ! OutputDirectory.empty( ) )
    {
      GenericAPI::Symbols::LDAS_ARCHIVE_DIR::Set( OutputDirectory.c_str( ) );
      GenericAPI::Symbols::LDAS_LOG_DIR::Set( OutputDirectory.c_str( ) );
    }
  }

  void
  setLogTag( const std::string& Tag )
  {
    if ( ! Tag.empty( ) )
    {
      APINameSet( Tag );
    }
  }

  void
  SetLogFormatter( const GenericAPI::Log::LDAS* Formatter )
  {
    default_log.reset( const_cast< GenericAPI::Log::LDAS* >( Formatter ) );
  }


  log_type
  LogFormatter( const std::string& Filename )
  {
    log_type	log = find_log( std::string( Filename ) );

    return log;
  }
}

//-----------------------------------------------------------------------
// Local variables
//-----------------------------------------------------------------------
typedef unordered_map< std::string,
		       log_type > log_container_type;

static log_container_type	logs;

static void
unregister_logs( )
{
  MutexLock	l( logs_baton,
		   __FILE__, __LINE__ );

  logs.erase( logs.begin( ), logs.end( ) );
}

static log_type
find_log( const std::string& Filename )
{
  using GenericAPI::Log::LDAS;
  std::ostringstream logname( Filename );

  //---------------------------------------------------------------------
  // 
  //---------------------------------------------------------------------
  if ( logname.str( ).length( ) == 0 )
  {
    logname << LDAS::LogFilename( default_log->FileExtension( ),
				  0 );
  }

  //---------------------------------------------------------------------
  // Mutex lock to prevent multiple creations
  //---------------------------------------------------------------------
  MutexLock	l( logs_baton,
		   __FILE__, __LINE__ );

#if 1
  log_type n;

  if ( logs.empty( ) )
  {
    //-------------------------------------------------------------------
    // Since it is, register the AtExit handler to clean up the memory.
    //-------------------------------------------------------------------
    AtExit::Append( unregister_logs,
		    "unregister_logs@api/genericAPI/Logging.cc",
		    150 );
    
    //-------------------------------------------------------------------
    // Does not already exist so create a new instance and then add
    // to the list.
    //-------------------------------------------------------------------
    n.reset( default_log->Clone( logname.str( ) ) );
  
  
    if ( n )
    {
      logs[ logname.str( ) ] = n;
      n->Spawn( );
    }
  }
  n = logs.begin( )->second;
#else /* 1 */
#if MULTI_LOG_SUPPORT
  log_container_type::iterator i = logs.find( logname.str( ) );
  if ( i != logs.end( ) )
  {
    //-------------------------------------------------------------------
    // Have a log associated with the file.
    //-------------------------------------------------------------------
    return (i->second);
  }
  //---------------------------------------------------------------------
  // Check to see if the list is empty
  //---------------------------------------------------------------------
  if ( logs.empty( ) )
  {
    //-------------------------------------------------------------------
    // Since it is, register the AtExit handler to clean up the memory.
    //-------------------------------------------------------------------
    AtExit::Append( unregister_logs,
		    "unregister_logs@api/genericAPI/Logging.cc",
		    150 );
  }

  //---------------------------------------------------------------------
  // Does not already exist so create a new instance and then add to the
  //   list.
  //---------------------------------------------------------------------
  log_type n( default_log->Clone( logname.str( ) ) );
  
  
  if ( n )
  {
    logs[ logname.str( ) ] = n;
    n->Spawn( );

    return n;
  }
#endif /* MULTI_LOG_SUPPORT */
#endif /* 1 */
  return n;
}

static group_map_type&
group_map_init( )
{
  static group_map_type gm;

  if ( gm.size( ) == 0 )
  {
    gm[ "0" ] = log_type::element_type::MT_OK;
    gm[ "1" ] = log_type::element_type::MT_WARN;
    gm[ "2" ] = log_type::element_type::MT_ERROR;
    gm[ "3" ] = log_type::element_type::MT_EMAIL;
    gm[ "4" ] = log_type::element_type::MT_PHONE;
    gm[ "5" ] = log_type::element_type::MT_DEBUG;
    gm[ "6" ] = log_type::element_type::MT_NOTE;
    gm[ "orange" ] = log_type::element_type::MT_ORANGE;
    gm[ "green" ] = log_type::element_type::MT_GREEN;
    gm[ "yellow" ] = log_type::element_type::MT_YELLOW;
    gm[ "red" ] = log_type::element_type::MT_RED;
    gm[ "blue" ] = log_type::element_type::MT_BLUE;
    gm[ "purple" ] = log_type::element_type::MT_PURPLE;
    gm[ "phone" ] = log_type::element_type::MT_PHONE;
    gm[ "pager" ] = log_type::element_type::MT_PHONE;
    gm[ "mail" ] = log_type::element_type::MT_MAIL;
    gm[ "email" ] = log_type::element_type::MT_EMAIL;
    gm[ "certmail" ] = log_type::element_type::MT_CERTMAIL;
  }
  return gm;
}

