/* -*- mode: c++ -*- */
#ifndef FR_VECT_I
#define FR_VECT_I

/* Begin Group - SWIGInterface */
/**
 * \addtogroup SWIGInterface
 * @{
 */

%module frameCPP
%feature("autodoc","1");

#ifndef SWIGIMPORTED
%{
#include "ldastoolsal/autoarray.hh"
#include "ldastoolsal/types.hh"
#include "ldastoolsal/SharedPtr.hh"

#include "framecpp/FrVect.hh"

#if HAVE_NUMPY_ARRAYOBJECT_H
#include "numpy/arrayobject.h"
#endif /* HAVE_NUMPY_ARRAYOBJECT_H */

  using namespace FrameCPP;
  using SHARED_PTR_NAMESPACE::shared_ptr;
  using std::vector;

#ifdef SWIGPYTHON
  extern "C" {
    PyObject* get_buffer( FrVect* Self );
  } /* extern "C" */
#endif /* SWIGPYTHON */

#ifdef SWIGPYTHON
  PyObject* get_buffer( FrVect* Self )
  {
    PyObject*		retval = (PyObject*)NULL;
    FrVect::data_type	data = Self->GetDataUncompressed( );

#if HAVE_NUMPY_ARRAYOBJECT_H
    //-------------------------------------------------------------------
    // Interface: NumPy
    //-------------------------------------------------------------------
    {
      int typenum = 0;
      npy_intp*	dims = (npy_intp*)NULL;

      try
      {
	//---------------------------------------------------------------
	// Establish the dimensionality of the array
	//---------------------------------------------------------------
	int	nd( Self->GetNDim( ) );

	dims = PyDimMem_NEW( nd );
	if ( dims )
	{
	  //-------------------------------------------------------------
	  // Populate the dimension array
	  //-------------------------------------------------------------
	  for ( int x = 0; x < nd; ++x )
	  {
	    dims[ x ] = Self->GetDim( x ).GetNx( );
	  }
	  //-------------------------------------------------------------
	  // Map the datatype
	  //-------------------------------------------------------------
	  switch( Self->GetType( ) )
	  {
	    //-----------------------------------------------------------
	    // Integers
	    //-----------------------------------------------------------
	  case FrVect::FR_VECT_C:	// FR_VECT_C
#if NPY_SIZEOF_BYTE == 1
	    typenum = NPY_BYTE;
#endif
	    break;
	  case FrVect::FR_VECT_1U:	// FR_VECT_1U
#if NPY_SIZEOF_UBYTE == 1
	    typenum = NPY_UBYTE;
#endif
	    break;
	  case FrVect::FR_VECT_2S:	// FR_VECT_2S
#if NPY_SIZEOF_INT == 2
	    typenum = NPY_INT;
#elif NPY_SIZEOF_SHORT == 2
	    typenum = NPY_SHORT;
#endif
	    break;
	  case FrVect::FR_VECT_2U:	// FR_VECT_2U
#if NPY_SIZEOF_UINT == 2
	    typenum = NPY_UINT;
#elif NPY_SIZEOF_USHORT == 2
	    typenum = NPY_USHORT;
#endif
	    break;
	  case FrVect::FR_VECT_4S:	// FR_VECT_4S
#if NPY_SIZEOF_INT == 4
	    typenum = NPY_INT;
#elif NPY_SIZEOF_LONG == 4
	    typenum = NPY_LONG;
#endif
	    break;
	  case FrVect::FR_VECT_4U:	// FR_VECT_4U
#if NPY_SIZEOF_UINT == 4
	    typenum = NPY_UINT;
#elif NPY_SIZEOF_ULONG == 4
	    typenum = NPY_ULONG;
#endif
	    break;
	  case FrVect::FR_VECT_8S:	// FR_VECT_8S
#if NPY_SIZEOF_LONG == 8
	    typenum = NPY_LONG;
#elif NPY_SIZEOF_LONGLONG == 8
	    typenum = NPY_LONGLONG;
#endif
	    break;
	  case FrVect::FR_VECT_8U:	// FR_VECT_8U
#if NPY_SIZEOF_ULONG == 8
	    typenum = NPY_ULONG;
#elif NPY_SIZEOF_ULONGLONG == 8
	    typenum = NPY_ULONGLONG;
#endif
	    break;
	    //-----------------------------------------------------------
	    // Real
	    //-----------------------------------------------------------
	  case FrVect::FR_VECT_4R:	// FR_VECT_4R
#if NPY_SIZEOF_FLOAT == 4
	    typenum = NPY_FLOAT;
#endif
	    break;
	  case FrVect::FR_VECT_8R:	// FR_VECT_8R
#if NPY_SIZEOF_DOUBLE == 8
	    typenum = NPY_DOUBLE;
#endif
	    break;
	    //-----------------------------------------------------------
	    // Complex
	    //-----------------------------------------------------------
	  case FrVect::FR_VECT_8C:	// FR_VECT_8C
#if NPY_SIZEOF_FLOAT == 4
	    typenum = NPY_CFLOAT;
#endif
	    break;
	  case FrVect::FR_VECT_16C:	// FR_VECT_16C
#if NPY_SIZEOF_DOUBLE == 8
	    typenum = NPY_CDOUBLE;
#endif
	    break;
	  default:
	    //-----------------------------------------------------------
	    // It has not been recognized.
	    //-----------------------------------------------------------
            typenum = 0;
	    break;
	  }
	  retval = PyArray_SimpleNewFromData( nd, dims, typenum, data.get( ) );
	}
      }
      catch( ... )
      {
	if ( dims )
	{
	  PyDimMem_FREE( dims );
	}
	dims = (npy_intp*)NULL;
	throw;
      }
    }
#else /* HAVE_NUMPY_ARRAYOBJECT_H */
    //-------------------------------------------------------------------
    // Interface: Simple buffer interface
    //-------------------------------------------------------------------
#if PY_VERSION_HEX >= 0x02060000
    /* #error New buffer interface not yet supported */
#else /* PY_VERSION_HEX */
#endif /* PY_VERSION_HEX */
    //-------------------------------------------------------------------
    // This is the older buffer interface that is being phased out
    //-------------------------------------------------------------------
    retval = PyBuffer_FromReadWriteMemory( data.get( ),
					   Self->GetNBytes( ) );
#endif /* HAVE_NUMPY_ARRAYOBJECT_H */
    return retval;
  }


#endif /* SWIGPYTHON */
typedef LDASTools::AL::SharedPtr< FrVect > FrVectSP;
%}
#endif /* !defined(SWIGIMPORTED) */

%include "std_vector.i"
%import "DataTypes.i"
%include "Container.i"
%import "framecpp/Dimension.i"
%import "framecpp/FrVect.hh"

%SharedPtr(FrVect)

%feature("docstring")
  FrVect
  "The FrVect class implements the FrVect structure as specified in
   the lastest approved frame specification."

#if !defined(SWIGIMPORTED)
%template(DimensionContainer) std::vector< Dimension >;
#endif /* !defined(SWIGIMPORTED) */


/**
 * \brief Vector Data Structure Definition
 */
%feature("autodoc",
"""
FrVect (Vector Data) implementation of the frame specification.

  Attributes:

    name      Channel name -- not required to be unique.
    compress  Compression algorithm number.
    type      Vector class.
    nData     Number of sample elements in data series.
    nBytes    Number of bytes in the compressed vector.
    nDim      Dimensionality of data vector.
    unitY     String describing how to interpret the value of
              each element. If dimensionless, then unitY == <NONE>,
              in CAPITALS (without <...>).
    dims      List of information describing each dimension of
              the data vector.
""" ) FrVect;

class FrVect
{
public:
  typedef FR_VECT_COMPRESS_TYPE			compress_type;
  typedef FR_VECT_TYPE_TYPE			type_type;
  typedef FR_VECT_NDATA_TYPE			nData_type;
  typedef FR_VECT_NBYTES_TYPE			nBytes_type;
  typedef FR_VECT_NDIM_TYPE			nDim_type;

  typedef FR_VECT_COMPRESSION_LEVEL_TYPE	compression_level_type;

#if 0
  typedef std::vector< Dimension > dimension_container_type;
#else
  typedef DimensionContainer dimension_container_type;
#endif

  enum compression_scheme_type {
    RAW = FrameCPP::FrVect::RAW,
    BIGENDIAN_RAW = FrameCPP::FrVect::BIGENDIAN_RAW,
    LITTLEENDIAN_RAW = FrameCPP::FrVect::LITTLEENDIAN_RAW,

    GZIP = FrameCPP::FrVect::GZIP,
    BIGENDIAN_GZIP = FrameCPP::FrVect::BIGENDIAN_GZIP,
    LITTLEENDIAN_GZIP = FrameCPP::FrVect::LITTLEENDIAN_GZIP,

    DIFF_GZIP = FrameCPP::FrVect::DIFF_GZIP,
    BIGENDIAN_DIFF_GZIP = FrameCPP::FrVect::BIGENDIAN_DIFF_GZIP,
    LITTLEENDIAN_DIFF_GZIP = FrameCPP::FrVect::LITTLEENDIAN_DIFF_GZIP,

    ZERO_SUPPRESS_WORD_2 = FrameCPP::FrVect::ZERO_SUPPRESS_WORD_2,
    BIGENDIAN_ZERO_SUPPRESS_WORD_2 = FrameCPP::FrVect::BIGENDIAN_ZERO_SUPPRESS_WORD_2,
    LITTLEENDIAN_ZERO_SUPPRESS_WORD_2 = FrameCPP::FrVect::LITTLEENDIAN_ZERO_SUPPRESS_WORD_2,

    ZERO_SUPPRESS_WORD_4 = FrameCPP::FrVect::ZERO_SUPPRESS_WORD_4,
    BIGENDIAN_ZERO_SUPPRESS_WORD_4 = FrameCPP::FrVect::BIGENDIAN_ZERO_SUPPRESS_WORD_4,
    LITTLEENDIAN_ZERO_SUPPRESS_WORD_4 = FrameCPP::FrVect::LITTLEENDIAN_ZERO_SUPPRESS_WORD_4,

    ZERO_SUPPRESS_WORD_8 = FrameCPP::FrVect::ZERO_SUPPRESS_WORD_8,
    BIGENDIAN_ZERO_SUPPRESS_WORD_8 = FrameCPP::FrVect::BIGENDIAN_ZERO_SUPPRESS_WORD_8,
    LITTLEENDIAN_ZERO_SUPPRESS_WORD_8 = FrameCPP::FrVect::LITTLEENDIAN_ZERO_SUPPRESS_WORD_8,

    ZERO_SUPPRESS_OTHERWISE_GZIP = FrameCPP::FrVect::ZERO_SUPPRESS_OTHERWISE_GZIP,
  };

  enum {
    FR_VECT_C = FrameCPP::FrVect::FR_VECT_C,
    FR_VECT_1U = FrameC::FrVect::FR_VECT_1U,
    FR_VECT_2S = FrameC::FrVect::FR_VECT_2S,
    FR_VECT_2U = FrameC::FrVect::FR_VECT_2U,
    FR_VECT_4S = FrameC::FrVect::FR_VECT_4S,
    FR_VECT_4U = FrameC::FrVect::FR_VECT_4U,
    FR_VECT_8S = FrameC::FrVect::FR_VECT_8S,
    FR_VECT_8U = FrameC::FrVect::FR_VECT_8U,

    FR_VECT_4R = FrameC::FrVect::FR_VECT_4R,
    FR_VECT_8R = FrameC::FrVect::FR_VECT_8R,

    FR_VECT_8C = FrameC::FrVect::FR_VECT_8C,
    FR_VECT_16C = FrameC::FrVect::FR_VECT_16C,

    FR_VECT_STRING = FrameC::FrVect::FR_VECT_STRING
  };

  //---------------------------------------------------------------------
  /// \brief Constructor.
  ///
  /// \param[in] name
  ///     The name of the data.
  /// \param[in] type
  ///     The data type.
  /// \param[in] nDim
  ///     The number of dimensions.
  /// \param[in] dims
  ///     A pointer to 'nDim' Dimension objects containing
  ///     information about the dimensions for this data.
  /// \param[in] byte_order
  ///     Byte order of the data. Default is BYTE_ORDER_HOST
  /// \param[in] data
  ///     A pointer to the data.  Default: 0
  /// \param[in] unitY
  ///     Units for the data.  Default: ""
  ///
  /// \return
  ///    A new instance of this object.
  //---------------------------------------------------------------------
  FrVect( const std::string& name,
	  type_type type,
	  nDim_type nDim,
	  const Dimension* dims, 
	  const std::string& unitY = "" );

  //---------------------------------------------------------------------
  /// \brief Retrieve the channel name.
  ///
  /// \return
  ///     The channel name
  //---------------------------------------------------------------------
  const std::string& GetName( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the compression algorithm number.
  ///
  /// \return
  ///     The compression algorithm number.
  //---------------------------------------------------------------------
  compress_type GetCompress( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the vector class.
  ///
  /// \return
  ///     The vector class.
  //---------------------------------------------------------------------
  type_type GetType( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the number of sample elements in data series.
  ///
  /// \return
  ///     The number of sample elements in data series.
  //---------------------------------------------------------------------
  nData_type GetNData( ) const;

  //---------------------------------------------------------------------
  /// \brief Retrieve the number of bytes in the compressed vector.
  ///
  /// \return
  ///     The number of bytes in the compressed vector.
  //---------------------------------------------------------------------
  nBytes_type GetNBytes( ) const;

#if 0
  //---------------------------------------------------------------------
  /// \brief Retrieve the pointer to the data.
  ///
  /// \return
  ///     The pointer to the data.
  //---------------------------------------------------------------------
  const CHAR_U* GetData( ) const;

  //-----------------------------------------------------------------
  /// \brief Retrieve the pointer to the compressed data.
  ///
  /// \return
  ///     The pointer to the compressed data.
  //-----------------------------------------------------------------
  const CHAR_U* GetDataRaw( ) const;

  //-----------------------------------------------------------------
  /// \brief Retrieve the pointer to the compressed data.
  ///
  /// \return
  ///     The pointer to the compressed data.
  //-----------------------------------------------------------------
  CHAR_U* GetDataRaw( );
#endif /* 0 */


#if SWIGPYTHON
  %extend {
    //-------------------------------------------------------------------
    /// \brief Retrieve the memory associated with FrVect as an Array
    //-------------------------------------------------------------------
    PyObject* GetDataArray( )
    {
      PyObject*	retval = get_buffer( self );

      return retval;
    }
  }
#endif /* SWIGPYTHON */

  //---------------------------------------------------------------------
  /// \brief Retrieve the pointer to the uncompressed data.
  ///
  /// \return
  ///     The pointer to the uncompressed data.
  //---------------------------------------------------------------------
  FrVect::data_type GetDataUncompressed( );

#if 0
  //-----------------------------------------------------------------
  /// \brief Retrieve the pointer to the uncompressed data.
  ///
  /// \return
  ///     The pointer to the uncompressed data.
  //-----------------------------------------------------------------
  const CHAR_U* GetDataUncompressed( LDASTools::AL::AutoArray< CHAR_U >&
				     Expanded ) const;

  template< class T > static INT_2U GetDataType();
#endif /* 0 */

  //---------------------------------------------------------------------
  /// \brief Retrieve the number of dimensions
  ///
  /// \return
  ///     The number of dimensions
  //---------------------------------------------------------------------
  nDim_type GetNDim( ) const;

#if 0
  const Dimension& GetDim( INT_4U Offset ) const;
#endif /* 0 */

  //---------------------------------------------------------------------
  /// \brief Retrieve the Nth dimension
  ///
  /// \return
  ///     The Nth dimension
  //---------------------------------------------------------------------
  Dimension& GetDim( nDim_type Offset );

  %extend {
    dimension_container_type& GetDims( );
  }
      

  //---------------------------------------------------------------------
  /// \brief Retrieve the description of how to interpret each element
  ///
  /// \return
  ///     The description of how to interpret each element
  //---------------------------------------------------------------------
  const std::string& GetUnitY( ) const;

#if 0
  CHAR_U* SteelData( bool& Owns ) const;

  void Compress( compression_scheme_type Scheme, int GZipLevel );
  void Uncompress( );

  virtual void CompressData( INT_4U Scheme, INT_2U GZipLevel );

  virtual Compression::compression_base_type
  Compression( ) const;

  virtual Common::FrameSpec::Object*
  CloneCompressed( cmn_compression_scheme_type Scheme,
		   cmn_compression_level_type Level ) const;

  //-----------------------------------------------------------------
  /// \brief Establish the channel name.
  ///
  /// \param[in] Name
  ///     The channel name
  //-----------------------------------------------------------------
  void SetName( const std::string& Name );

  void SetNData( INT_4U NData );

  size_t GetTypeSize( ) const;
  static size_t GetTypeSize( INT_2U type );

  //-----------------------------------------------------------------
  /// \brief Merge with another FrAdcData
  ///
  /// \param[in] RHS
  ///     The source of the information to append to this FrAdcData
  ///     structure.
  ///
  /// \return
  ///     A reference to this object
  //-----------------------------------------------------------------
  FrVect& Merge( const FrVect& RHS );

  //-----------------------------------------------------------------
  /// \brief Request a subset of the data.
  ///
  /// \param[in] Start
  ///     The offset in the data set where to start.
  /// \param[in] Stop
  ///     The offset in the data set where to stop.
  ///
  /// \return
  ///     The subset of data bounded by data[Start, Stop).
  //-----------------------------------------------------------------
  std::auto_ptr< FrVect > SubFrVect( INT_4U Start, INT_4U Stop ) const;
#endif /* 0 */

  LDAS_PROPERTY_READ_ONLY("name",GetName)
  LDAS_PROPERTY_READ_ONLY("compress",GetCompress)
  LDAS_PROPERTY_READ_ONLY("type",GetType)
  LDAS_PROPERTY_READ_ONLY("nData",GetNData)
  LDAS_PROPERTY_READ_ONLY("nBytes",GetNBytes)
  LDAS_PROPERTY_READ_ONLY("nDim",GetNDim)
  LDAS_PROPERTY_READ_ONLY("unitY",GetUnitY)
  LDAS_PROPERTY_READ_ONLY("dims",GetDims)
};

#if ! SWIGIMPORTED
%{
  FrVect::dimension_container_type&
  FrVect_GetDims( FrVect* Self )
  {
    return Self->GetDims( );
  }
%}
#endif /* ! SWIGIMPORTED */

#if !defined(SWIGIMPORTED)
CONTAINER_WRAP(FrVect,FrVect)
#endif /* !defined(SWIGIMPORTED) */

/* End Group - SWIGInterface */
/**
 * @}
 */
#endif /* FR_VECT_I */
