/* -*- mode: C++; c-basic-offset: 2; -*- */

#ifndef FRAME_CPP__COMMON__SearchContainer_HH
#define FRAME_CPP__COMMON__SearchContainer_HH

#if ! defined(SWIGIMPORTED)
#include <sstream>
#endif /* ! defined(SWIGIMPORTED) */

#include "ldastoolsal/unordered_map.hh"
#include "ldastoolsal/regexmatch.hh"
#include "ldastoolsal/ldasexception.hh"
#include "ldastoolsal/util.hh"

#include "framecpp/Common/FrameSpec.hh"
#include "framecpp/Common/Container.hh"


namespace FrameCPP
{
  namespace Common
  {

    //-------------------------------------------------------------------
    //
    //: A searchable container.
    //
    // The SearchContainer class allows the user to find an element in the class
    // based upon an attribute of the contained object.  For this class, there are
    // two template parameters:
    //
    // <ul>
    //   <li>class T - The type of object to store.</li>
    //   <li>const string& (T::*F)() const - A constant method of class T returning
    //       a const string&.  This is used to supply the attribute to query on.
    //       </li>
    // </ul>
    //
    // <p>For example, a container holding AdcData objects which searched on the
    // name field would be declared as:
    //
    // <p><code>SearchContainer< AdcData, &AdcData::getName ></code>
    //
    // <p>The SearchContainer class adds 3 ways to search for a contained object:
    //
    // <ol>
    //   <li>Exact match - This is implemented via the find methods.  This
    //       iterates through a container until an exact match is found.  The user
    //       may specify the starting location for the search.</li>
    //   <li>Regex match - This is implemented via regexFind.</li>
    //   <li>Hashed search - If the user would like to use a hash to find objects
    //       in the container, then they can use the hashFind method.  After a
    //       change to the container which would modify an element's key the user
    //       must manually 'rehash' the container.</li>
    // </ol>
    //
    //!todo: Should this be an adaptor class?
    //
    template< class T, const std::string& (T::*F)() const >
    class SearchContainer
      : public Container< T > 
    {
    public:

      /* typedef's */
      typedef typename Container< T >::value_type value_type;
      typedef typename Container< T >::const_value_type const_value_type;

      typedef typename Container< T >::iterator iterator;
      typedef typename Container< T >::const_iterator const_iterator;
      typedef typename Container< T >::reverse_iterator reverse_iterator;
      typedef typename Container< T >::const_reverse_iterator const_reverse_iterator;
    
      typedef typename LDASTools::AL::
      unordered_multimap< std::string, value_type,
			  LDASTools::AL::hash< std::string >,
			  LDASTools::AL::CaseInsensitiveCmp > hash_type;

      typedef typename hash_type::const_iterator const_hash_iterator;
      typedef typename hash_type::iterator hash_iterator;


      /* Constructors / Destructor */
      //-------------------------------------------------------------------
      //
      //: Default Constructor.
      //
      //!exc: None.   
      //      
      SearchContainer( bool AllowDuplicates = true );

      //-------------------------------------------------------------------
      //
      //: Copy Constructor.
      //
      //!param: SearchContainer< T, F >& sc - Search container to be copied.
      //
      //!exc: bad_alloc - Memory allocation failed.
      //
      SearchContainer( const SearchContainer< T, F >& sc );

      //-------------------------------------------------------------------
      //
      //: Destructor
      //
      //!exc: None.   
      //         
      virtual ~SearchContainer();


      /* Operator Overloads */

      //-------------------------------------------------------------------
      //
      //: Assignment Operator.
      //
      //!param: const SearchContainer< T, F >& sc - Search container to be assigned.
      //
      //!return: const SearchContainer< T, F >& -- "this" search container.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      const SearchContainer< T, F >&
      operator=( const SearchContainer< T, F >& sc );

#if 0
      //-------------------------------------------------------------------
      //
      //: Concatenate search containers.
      //
      //!param: const SearchContainer< T, F >& sc - Search contaier to be concatenated.
      //
      //!return: const SearchContainer< T, F >& -- "this" search container.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //!exc: frame_mismatch - `T' mismatch.
      //
      const SearchContainer< T, F >&
      operator+=( const SearchContainer< T, F >& sc );
#endif


      /* Accessors */
      //-------------------------------------------------------------------
      //
      //: Finds an element with the given name.
      //
      //!param: const std::string& name - Name to search for.
      //
      //!return: const_iterator -- Iterator pointing to first element found.
      //
      //!exc: None.   
      //         
      const_iterator find( const std::string& name )
	const;

      //-------------------------------------------------------------------
      //
      //: Finds an element with the given name.
      //
      //!param: const std::string& name - Name to search for.
      //!param: const_iterator start - Where to begin search.
      //
      //!return: const_iterator -- Iterator pointing to first element found.
      //
      //!exc: None.   
      //         
      const_iterator find( const std::string& name, const_iterator start )
	const;

      //-------------------------------------------------------------------
      //
      //: Finds an element with the given name.
      //
      //!param: const std::string& name - Name to search for.
      //
      //!return: iterator -- Iterator pointing to first element found.
      //
      //!exc: None.   
      //         
      iterator find( const std::string& name );

      //-------------------------------------------------------------------
      //
      //: Finds an element with the given name.
      //
      //!param: const std::string& name - Name to search for.
      //!param: iterator start - Where to begin search.
      //
      //!return: iterator -- Iterator pointing to first element found.
      //
      //!exc: None.   
      //         
      iterator find( const std::string& name, iterator start );

      //-------------------------------------------------------------------
      //
      //: Finds an element with a name matching the given regex.
      //
      //!param: const std::string& regex - Regular expression to search for.
      //
      //!return: const_iterator -- Iterator pointing to first element found.
      //
      //!exc: LdasException
      //
      const_iterator regexFind( const std::string& regex )
	const;

      //-------------------------------------------------------------------
      //
      //: Finds an element with a name matching the given regex.
      //
      //!param: const std::string& regex - Regular expression to search for.
      //!param: const_iterator start - Where to begin search.
      //
      //!return: const_iterator -- Iterator pointing to first element found.
      //
      //!exc: LdasException
      //
      const_iterator
      regexFind( const std::string& regex, const_iterator start ) const;

      //-------------------------------------------------------------------
      //
      //: Finds an element with a name matching the given regex.
      //
      //!param: const std::string& regex - Regular expression to search for.
      //!param: const_iterator start - Where to begin search.
      //
      //!return: iterator -- Iteartor poiting to the element found.
      //
      //!exc: LdasException
      //   
      iterator regexFind( const std::string& regex );
 
      //-------------------------------------------------------------------
      //
      //: Finds an element with a name matching the given regex.
      //
      //!param: const std::string& regex - Regular expression to search for.
      //!param: const_iterator start - Where to begin search.
      //
      //!return: iterator -- Iteartor poiting to the element found.
      //
      //!exc: LdasException
      //   
      iterator regexFind( const std::string& regex, iterator start );
 
      //-------------------------------------------------------------------
      //
      //: Finds an element with the given name via a hash.
      //
      //!param: const std::string& name - The name to search for.
      //
      //!return: std::pair< const_hash_iterator, const_hash_iterator > --
      //+  Iterator range for the elements found.
      //
      //!exc: None.
      //
      std::pair< const_hash_iterator, const_hash_iterator >
      hashFind( const std::string& name ) const;

      //-------------------------------------------------------------------
      //
      //: Finds an element with a the given name via a hash.
      //
      //!param: const std::string& name - The name to search for.
      //
      //!return: std::pair< hash_iterator, hash_iterator > --
      //+  Iterator range for the elements found.
      //
      //!exc: None.
      //   
      std::pair< hash_iterator, hash_iterator >
      hashFind( const std::string& name );

    
      /* Mutators */
      //-------------------------------------------------------------------
      //
      //: Rehash search container.
      //
      //!exc: None.
      //   
      void rehash() const;

      //-------------------------------------------------------------------
      //
      //: Append an item to the container.
      //
      //!param: const T& data - An element to append.
      //
      //!return: iterator -- The iterator correspinding to the appended object.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      iterator append( value_type data )
      {
	const std::string
	  k( (data.get( )->*F)( ) );

	if ( ( ! mAllowDuplicates ) &&
	     ( mHash.find( k ) != mHash.end( ) ) )
	{
	  std::ostringstream	oss;

	  oss << "Inserting non-unique key: " << k
	      << " into list of " << mHash.size( ) << " element(s)";
	  throw std::logic_error( oss.str( ) );
	}
	iterator iter = Container< T >::append( data );

	mHash.insert( make_pair( k, *iter ) );
	return iter;
      }

      iterator append( const T& data )
      {
	const std::string	k( (data.*F)( ) );

	if ( ( ! mAllowDuplicates ) &&
	     ( mHash.find( k ) != mHash.end( ) ) )
	{
	  std::ostringstream	oss;

	  oss << "Inserting non-unique key: " << k;
	  throw std::logic_error( oss.str( ) );
	}
	iterator iter = Container< T >::append( data );

	mHash.insert( make_pair( k, *iter ) );

	return iter;
      }

      //-------------------------------------------------------------------
      /// \brief Insert an item into the container.
      ///
      /// \param[in] pos
      ///     An iterator pointing to the location where the
      ///     element should be added.
      /// \param[in] data
      ///     The item to add.
      ///
      /// \return
      ///     An iterator pointing to the added item.
      ///
      /// \exception std::bad_alloc
      ///     Memory could not be allocated for the element.
      ///
      iterator insert( iterator pos, value_type data )
      {
	iterator iter = Container< T >::insert( pos, data );
	mHash.insert( make_pair( ((*iter)->*F)(), *iter ) );
	return iter;
      }

      //-------------------------------------------------------------------
      //
      //: Insert an item into the container.
      //
      //!param: size_t index - The index at which the element should be added.  If
      //+     the index is out of range then the element will be inserted at the end.
      //!param: const T& data - The item to add.
      //
      //!return: iterator -- An iterator pointing to the added item.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      iterator insert( size_t index, const T& data )
      {
	iterator iter = Container< T >::insert( index, data );
	mHash.insert( make_pair( ((*iter)->*F)(), *iter ) );
	return iter;
      }

#if 0
      //-------------------------------------------------------------------
      //
      //: Insert an item into the container.
      //
      //!param: size_t index - The index at which the element should be added.  If
      //+     the index is out of range then the element will be inserted at the end.
      //!param: const T* data - The item to add.
      //
      //!return: iterator -- An iterator pointing to the added item.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      iterator insert( size_t index, const T* data );

      //-------------------------------------------------------------------
      //
      //: Insert an item into the container.
      //
      //!param: size_t index - The index at which the element should be added.  If
      //+     the index is out of range then the element will be inserted at the end.
      //!param: const T* data - The item to add.
      //!param: bool allocate - Duplicate "data" or not (default: true).
      //!param: bool owns - Will destruct "data" when destructed or not (default: true).
      //
      //!return: iterator -- An iterator pointing to the added item.
      //
      //!exc: std::bad_alloc - Memory could not be allocated for the element.
      //
      iterator insert( size_t index, T* data, bool allocate = true,
		       bool owns = true );
#endif /* 0 */

      //-------------------------------------------------------------------
      //
      //: Remove an element from the container.
      //
      // If the element is owned, then it will be destructed.
      //
      //!param: unsigned int index - which element to erase.
      //
      //!exc: None.
      //   
      void erase( unsigned int index );

      //-------------------------------------------------------------------
      //
      //: Remove a sequence of elements from the container.
      //
      // All owned elements in sequence will be destructed.
      //
      //!param: iterator start - Sequence beginning.
      //!param: iterator finish - Sequence end.
      //
      //!exc: None.
      //   
      void erase( iterator start, iterator finish );
    
      SearchContainer< T, F>&
      Merge( const SearchContainer< T, F >& RHS );

    private:

      //: Element hash.
      mutable hash_type	mHash;
      bool		mAllowDuplicates;

      inline const std::string&
      key( const_iterator i ) const
      {
	return (i->get( )->*F)( );
      }
      
    };

    template< class T, const std::string& (T::*F)() const >
    inline typename SearchContainer< T, F >::const_iterator
    SearchContainer< T, F >::
    find( const std::string& name )
      const
    {
      return find( name, this->begin( ) );
    }

    template< class T, const std::string& (T::*F)() const >
    inline typename SearchContainer< T, F>::iterator
    SearchContainer< T, F >::
    find( const std::string& name )
    {
      return find( name, this->begin( ) );
    }

    template< class T, const std::string& (T::*F)() const >
    inline typename SearchContainer< T, F>::const_iterator
    SearchContainer< T, F >::
    regexFind( const std::string& regex )
      const
    {
      return regexFind( regex, this->begin( ) );
    }

    template< class T, const std::string& (T::*F)() const >
    inline typename SearchContainer< T, F>::iterator
    SearchContainer< T, F >::
    regexFind( const std::string& regex )
    {
      return regexFind( regex, this->begin( ) );
    }
  } // namespace - Common
} // namespace - FrameCPP

#include "framecpp/Common/SearchContainer.tcc"

#endif // FRAME_CPP__COMMON__SearchContainer_HH
