#include <ldas_tools_config.h>

#include <fstream>

#include "genericAPI/Logging.hh"

#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"

#include "diskcacheAPI/Streams/ASCII.hh"
#include "diskcacheAPI/Streams/Binary.hh"
#include "diskcacheAPI/Streams/FStream.hh"

#include "diskcacheAPI/MetaCommands.hh"

#include "Commands.hh"
#include "diskcachecmd.hh"
#include "DumpCacheDaemon.hh"
#include "IO.hh"
#include "MountPointScanner.hh"

//=======================================================================
//=======================================================================

using GenericAPI::queueLogEntry;

using diskCache::Commands::updateFileExtList;
using diskCache::Commands::updateMountPtList;

typedef diskCache::MetaCommand::client_type	client_type;
typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;

//=======================================================================
// Global variables
//=======================================================================

#if 0
static client_type	client_handle;
#endif /* 0 */

//=======================================================================
//=======================================================================
#if 0
static void send_request( const std::string& Command,
			  const ServerInfo& Server );
#endif /* 0 */

//=======================================================================
/// \brief Maintains a list of commands that are supported.
//=======================================================================
namespace diskCache
{
  namespace MetaCommand
  {
    //===================================================================
    // CommandTable
    //===================================================================

    // Instantiate generic to Singleton class methods:   
    // see "general" library   
    SINGLETON_TS_INST( CommandTable );

    /** \cond ignore_singleton_constructor */
    CommandTable::
    CommandTable( )
    {
      command_table[ "daemon" ] = CMD_DAEMON;
      command_table[ "dump" ] = CMD_DUMP;
      command_table[ "filenames" ] = CMD_FILENAMES;
      command_table[ "filenames-rds" ] = CMD_FILENAMES_RDS;
      command_table[ "intervals" ] = CMD_INTERVALS;
      command_table[ "mount-point-stats" ] = CMD_MOUNT_POINT_STATS;
      command_table[ "quit" ] = CMD_QUIT;
      command_table[ "scan" ] = CMD_SCAN;
    }
    /** \endcond */

    CommandTable::
    ~CommandTable( )
    {
      command_table.erase( command_table.begin( ),
			   command_table.end( ) );
    }

    ClientServerInterface::
    ~ClientServerInterface( )
    {
      if ( server_request_handle )
      {
	server_request_handle->close( );
      }
    }

    //===================================================================
    // Daemon::Config_
    //===================================================================
    Daemon::Config_::
    Config_( Daemon& Command )
      : command( Command ),
	state( NON_BLOCK )
    {
    }

    void Daemon::Config_::
    Parse( std::istream& Stream )
    {
      base_type::Parse( Stream );
    }

    void Daemon::Config_::
    ParseBlock( const std::string& Value )
    {
      if ( Value.compare( "DAEMON" ) == 0 )
      {
	state = BLOCK_DAEMON;
      }
      else if ( Value.compare( "EXTENSIONS" ) == 0 )
      {
	state = BLOCK_EXTENSIONS;
      }
      else if ( Value.compare( "EXCLUDED_DIRECTORIES" ) == 0 )
      {
	state = BLOCK_EXCLUDED_DIRECTORIES;
      }
      else if ( Value.compare( "MOUNT_POINTS" ) == 0 )
      {
	state = BLOCK_MOUNT_POINTS;
      }
      else
      {
	//---------------------------------------------------------------
	// Tried everything we know. Flag as unknown block
	//---------------------------------------------------------------
	state = BLOCK_UNKNOWN;
      }
    }

    void Daemon::Config_::
    ParseKeyValue( const std::string& Key,
		   const std::string& Value )
    {
      switch( state )
      {
      case NON_BLOCK:
      case BLOCK_DAEMON:
	if ( Key.compare( "CONCURRENCY" ) == 0 )
	{
	  command.set_concurrency( Value );
	}
	else if ( Key.compare( "LOG" ) == 0 )
	{
	  command.set_log( Value );
	}
	else if ( Key.compare( "LOG_DIRECTORY" ) == 0 )
	{
	  command.set_log_directory( Value );
	}
	else if ( Key.compare( "LOG_ROTATE_ENTRY_COUNT" ) == 0 )
	{
	  command.set_log_rotate_entry_count( Value );
	}
	else if ( Key.compare( "OUTPUT_ASCII" ) == 0 )
	{
	  command.set_output_ascii( Value );
	}
	else if ( Key.compare( "OUTPUT_ASCII_VERSION" ) == 0 )
	{
	  command.set_output_ascii_version( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY" ) == 0 )
	{
	  command.set_output_binary( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY_VERSION" ) == 0 )
	{
	  command.set_output_binary_version( Value );
	}
	else
	{
	  if ( state == BLOCK_DAEMON )
	  {
	    //-----------------------------------------------------------
	    /// \todo
	    ///     Need to produce and exception regarding unknown
	    ///     keyword found
	    //-----------------------------------------------------------
	  }
	}
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
      case BLOCK_EXTENSIONS:
      case BLOCK_MOUNT_POINTS:
	//---------------------------------------------------------------
	/// \todo
	///     Need to produce an exception regarding parse error
	///     in configuration file as a key/value pair was encountered
	///     where a word value was expected.
	//---------------------------------------------------------------
	break;
      default:
	//---------------------------------------------------------------
	// Ignore all other cases
	//---------------------------------------------------------------
	break;
      }
    }

    void Daemon::Config_::
    ParseWord( const std::string& Value )
    {
      switch( state )
      {
      case BLOCK_EXTENSIONS:
	command.push_extension( Value );
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
	command.push_excluded_directory( Value );
	break;
      case BLOCK_MOUNT_POINTS:
	command.push_mount_point( Value );
	break;
      case BLOCK_DAEMON:
      case BLOCK_UNKNOWN:
      case NON_BLOCK:
	// This shouild never happen.
	/// \todo
	/// Have Daemon::Config_::ParseWord throw exception
	/// if it reaches an unreachable state.
	break;
      }
    }


    //===================================================================
    // Daemon
    //===================================================================
    bool Daemon::daemon_mode;
    OptionSet& Daemon::m_options( Daemon::init_options( ) );

    OptionSet& Daemon::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: daemon" );

      retval.
	Summary( "The daemon sub command is intended to"
		 " provide a continuous scanning mode." );

      retval.Add( Option( OPT_CONCURRENCY,
			  "concurrency",
			  Option::ARG_REQUIRED,
			  "Number of mount points to scan concurrently",
			  "integer" ) );

      retval.Add( Option( OPT_CONFIGURATION_FILE,
			  "configuration-file",
			  Option::ARG_REQUIRED,
			  "Name of file containing additional configuration information.",
			  "filename" ) );

      retval.Add( Option( OPT_EXCLUDED_DIRECTORIES,
			  "excluded-directories",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of directories not to be searched",
			  "list" ) );

      retval.Add( Option( OPT_EXTENSIONS,
			  "extensions",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of file extensions",
			  "list" ) );

      retval.Add( Option( OPT_LOG,
			  "log",
			  Option::ARG_REQUIRED,
			  "Specify where the log messages should be written.",
			  "filename" ) );

      retval.Add( Option( OPT_LOG_DIRECTORY,
			  "log-directory",
			  Option::ARG_REQUIRED,
			  "Specify the directory to capture logging information",
			  "directory" ) );

      retval.Add( Option( OPT_LOG_ROTATE_ENTRY_COUNT,
			  "log-rotate-entry-count",
			  Option::ARG_REQUIRED,
			  "Specify the point at which to rotate the log files.",
			  "count" ) );

      retval.Add( Option( OPT_MOUNT_POINTS,
			  "mount-points",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of mount points to scan",
			  "list" ) );

      retval.Add( Option( OPT_OUTPUT_ASCII,
			  "output-ascii",
			  Option::ARG_REQUIRED,
			  "Filename for the ascii output; '-' to direct to standard output",
			  "filename" ) );

      retval.Add( Option( OPT_OUTPUT_BINARY,
			  "output-binary",
			  Option::ARG_REQUIRED,
			  "Filename for the binary output",
			  "filename" ) );

      retval.Add( Option( OPT_VERSION_ASCII,
			  "version-ascii",
			  Option::ARG_REQUIRED,
			  "Version of the ascii diskcache dump format to output",
			  "version" ) );

      retval.Add( Option( OPT_VERSION_BINARY,
			  "version-binary",
			  Option::ARG_REQUIRED,
			  "Version of the binary diskcache dump format to output",
			  "version" ) );
      return retval;
    }

    Daemon::
    Daemon( CommandLineOptions& Args,
	    const ClientServerInterface::ServerInfo& Server )
      : m_args( Args ),
	m_concurrency( 4 ),
	log_rotate_entry_count( 800 ),
	version_ascii( Streams::Interface::VERSION_NONE ),
	version_binary( Streams::Interface::VERSION_NONE ),
	finished( false ),
	server_info( Server )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_CONCURRENCY:
	    set_concurrency( arg_value );
	    break;
	  case OPT_CONFIGURATION_FILE:
	    {
	      Config_		cfg( *this );
	      std::ifstream	stream( arg_value.c_str( ) );

	      cfg.Parse( stream );
	    }
	    break;
	  case OPT_EXCLUDED_DIRECTORIES:
	    {
	      //-----------------------------------------------------------
	      // Generate list of directories
	      //-----------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::directory_container_type
		directories;

	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		//-------------------------------------------------------
		// Extract each directory
		//-------------------------------------------------------
		end = arg_value.find_first_of( ",", pos );
		const size_t dend( ( end == std::string::npos )
				   ? end
				   : ( end - pos ) );
		push_excluded_directory( arg_value.substr( pos, dend ) );
		pos = end + 1;
	      }
	      //---------------------------------------------------------
	      // Update the list of directories to be excluded
	      //---------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::Update( directories );
	    }
	    break;
	  case OPT_EXTENSIONS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of extensions
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_extension( arg_value.substr( pos,( ( end == std::string::npos )
							? end
							: end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_LOG:
	    set_log( arg_value );
	    break;
	  case OPT_LOG_DIRECTORY:
	    set_log_directory( arg_value );
	    break;
	  case OPT_LOG_ROTATE_ENTRY_COUNT:
	    set_log_rotate_entry_count( arg_value );
	    break;
	  case OPT_MOUNT_POINTS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of mount points
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_mount_point( arg_value.substr( pos,( ( end == std::string::npos )
							  ? end
							  : end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_OUTPUT_ASCII:
	    set_output_ascii( arg_value );
	    break;
	  case OPT_OUTPUT_BINARY:
	    set_output_binary( arg_value );
	    break;
	  case OPT_VERSION_ASCII:
	    set_output_ascii_version( arg_value );
	    break;
	  case OPT_VERSION_BINARY:
	    set_output_binary_version( arg_value );
	    break;
	  default:
	    break;
	  }
	}
      }

      GenericAPI::LogFormatter( )->EntriesMax( log_rotate_entry_count );
      {
	using GenericAPI::Log::LDAS;

	LDAS::stream_file_type
	  fs( LDASTools::AL::DynamicPointerCast< LDAS::stream_file_type::element_type >
	      ( GenericAPI::LogFormatter( )->Stream( ) ) );
	if ( fs )
	{
	  std::string	ad;
	  GenericAPI::Symbols::LDAS_ARCHIVE_DIR::Get( ad );
	  
	  fs->ArchiveDirectory( ad );
	  fs->FilenamePattern( "diskcache" );
	  fs->FilenameExtension( GenericAPI::LogFormatter( )->FileExtension( ) );
	}
      }
      if ( excluded_directories.size( ) > 0 )
      {
	//---------------------------------------------------------------
	// Update the list of directories to be excluded
	//---------------------------------------------------------------
	Cache::ExcludedDirectoriesSingleton::Update( excluded_directories );
      }
    }

    const OptionSet& Daemon::
    Options( )
    {
      return m_options;
    }

    void Daemon::
    do_client_request( )
    {
      static const char* method_name = "diskCache::MetaCommand::Daemon::do_client_request";

      if ( server->good( ) )
      {
	queueLogEntry( "Server: waiting for client request",
		       GenericAPI::LogEntryGroup_type::MT_DEBUG,
		       30,
		       method_name,
		       "CXX" );
	client = server->accept( );

	if ( client->good( ) )
	{
	  Spawn( );
	  Join( );
	}

	queueLogEntry( "Server: client request completed",
		       GenericAPI::LogEntryGroup_type::MT_DEBUG,
		       30,
		       method_name,
		       "CXX" );
	client.reset( ( server_responce_type::element_type* )NULL );
      }
    }

    bool Daemon::
    process_cmd( CommandLineOptions& Options )
    {
#if 0
      bool	client_server = ( ( daemon_mode ) && ( server_info.Port( ) >= 0 ) );
#endif /* 0 */
      bool	retval = true;

      switch( CommandTable::Lookup( Options.ProgramName( ) ) )
      {
      case CommandTable::CMD_QUIT:
	{
	  finished = true;
	}
	break;
      case CommandTable::CMD_DUMP:
	{
	  //-------------------------------------------------------------
	  // Dump the current contents of the cache
	  //-------------------------------------------------------------
	  MetaCommand::Dump	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES:
	{
	  MetaCommand::Filenames	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES_RDS:
	{
	  MetaCommand::FilenamesRDS	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_INTERVALS:
	{
	  MetaCommand::Intervals	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      default:
	{
	  retval = false;
	}
	break;
      }
      return retval;
    }

    bool Daemon::
    read_command( char* Buffer, size_t BufferSize )
    {
      bool	retval = false;

      if ( ! server )
      {
	std::cin.getline( Buffer, BufferSize, '\n' );
	retval = ( ( std::cin.good( ) )
		  ? true
		  : false );
		    
      }
      return retval;
    }

    void Daemon::
    operator()( )
    {
      //-----------------------------------------------------------------
      // 
      //-----------------------------------------------------------------
      daemon_mode = true;

      //-----------------------------------------------------------------
      // Setup the logging stream
      //-----------------------------------------------------------------
      std::ofstream	log_stream_base;
      std::ostream*	log_stream = &std::cout;

      if ( m_log.empty( ) == false )
      {
	log_stream_base.open( m_log.c_str( ) );
	if ( log_stream_base.is_open( ) )
	{
	  log_stream = &log_stream_base;
	}
      }

      //-----------------------------------------------------------------
      // Check to see if the server port has been requested
      //-----------------------------------------------------------------
      if ( server_info.Port( ) >= 0 )
      {
	server = server_type( new server_type::element_type( ) );
	if ( server )
	{
	  server->open( server_info.Port( ) );
	}
      }

      //-----------------------------------------------------------------
      // Add the extensions
      //-----------------------------------------------------------------
      updateFileExtList( m_extensions );

      //-----------------------------------------------------------------
      // Add the mount points
      //-----------------------------------------------------------------
      {
	MountPointManagerSingleton::UpdateResults    status;

	updateMountPtList( status, m_mount_points, false );
      }

      //-----------------------------------------------------------------
      // Setup scanning parameters
      //-----------------------------------------------------------------
      ScanConcurrency( m_concurrency );

      //-----------------------------------------------------------------
      // Scan the directories
      //-----------------------------------------------------------------

      ScanMountPointListContinuously( log_stream );

      //-----------------------------------------------------------------
      // Update the cache files
      //-----------------------------------------------------------------
      if ( output_ascii.size( ) > 0 )
      {
	diskCache::DumpCacheDaemon::FilenameAscii( output_ascii );
      }
      if ( output_binary.size( ) > 0 )
      {
	diskCache::DumpCacheDaemon::FilenameBinary( output_binary );
      }
      if ( version_ascii != Streams::Interface::VERSION_NONE )
      {
	diskCache::DumpCacheDaemon::ASCIIVersion( version_ascii );
      }
      if ( version_binary != Streams::Interface::VERSION_NONE )
      {
	diskCache::DumpCacheDaemon::BinaryVersion( version_binary );
      }
      DumpCacheDaemonStart( log_stream );

      //-----------------------------------------------------------------
      // Read commands to be executed.
      //-----------------------------------------------------------------
      if ( server )
      {
	while ( ( finished == false )
		&& ( server->good( ) ) )
	{
	  do_client_request( );
	}
      }
      else
      {
	char	line_buffer[ 2048 ];

	while ( ( finished == false )
		&& ( read_command( line_buffer, sizeof( line_buffer ) ) ) )
	{
	  //-------------------------------------------------------------
	  // Create list of options.
	  //-------------------------------------------------------------
	  char*	lb_opts[ 20 ];
	  int	cur_opt = 0;
	  char*	lb_pos = line_buffer;

	  lb_opts[ cur_opt ] = lb_pos;
	  while( *lb_pos )
	  {
	    if ( *lb_pos == ' ' )
	    {
	      *lb_pos = '\0';
	      if ( ++lb_pos )
	      {
		lb_opts[ ++cur_opt ] = lb_pos;
	      }
	      continue;
	    }
	    ++lb_pos;
	  }
	  lb_opts[ ++cur_opt ] = (char*)NULL;
      
	  //-------------------------------------------------------------
	  // Switch on the command specified by the user
	  //-------------------------------------------------------------
	  CommandLineOptions	line_options( cur_opt, lb_opts );

	  process_cmd( line_options);
	}
      }
      //-----------------------------------------------------------------
      // Get read to terminate
      //-----------------------------------------------------------------
      if ( log_stream_base.is_open( ) )
      {
	log_stream_base.close( );
      }
    }

    void Daemon::
    action( )
    {
      char	line_buffer[ 2048 ];

      INT_4U	bytes;
      (*client) >> bytes;
      if ( bytes > sizeof( line_buffer) )
      {
      }
      else
      {
	client->read( line_buffer, bytes );

	//---------------------------------------------------------------
	// Create list of options.
	//---------------------------------------------------------------
	char*	lb_opts[ 20 ];
	int	cur_opt = 0;
	char*	lb_pos = line_buffer;
      
	lb_opts[ cur_opt ] = lb_pos;
	while( *lb_pos )
	{
	  if ( *lb_pos == ' ' )
	  {
	    *lb_pos = '\0';
	    if ( ++lb_pos )
	    {
	      lb_opts[ ++cur_opt ] = lb_pos;
	    }
	    continue;
	  }
	  ++lb_pos;
	}
	lb_opts[ ++cur_opt ] = (char*)NULL;
      
	//---------------------------------------------------------------
	// Switch on the command specified by the user
	//---------------------------------------------------------------
	CommandLineOptions	line_options( cur_opt, lb_opts );

	process_cmd( line_options);
      }
    }

    //===================================================================
    // MountPointStats
    //===================================================================
    OptionSet& MountPointStats::m_options( MountPointStats::init_options( ) );

    OptionSet& MountPointStats::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: mountPointStats" );

      retval.
	Summary( "The mount-point-stats sub command is intended to get some statistical information"
		 " the memory cache grouped by the mount points."
		 );

      retval.Add( Option( OPT_IFO,
			  "ifo",
			  Option::ARG_REQUIRED,
			  "IFO pattern to use for search. (Default: all)",
			  "pattern" ) );

      retval.Add( Option( OPT_TYPE,
			  "type",
			  Option::ARG_REQUIRED,
			  "Type pattern to use for search. (Default: all)",
			  "pattern" ) );

      return retval;
    }

    MountPointStats::
    MountPointStats( CommandLineOptions& Args )
      : m_args( Args ),
	m_ifo( "all" ),
	m_type( "all" )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_IFO:
	    m_ifo = arg_value;
	    break;
	  case OPT_TYPE:
	    m_type = arg_value;
	    break;
	  default:
	    break;
	  }
	}
      }
    }

    const OptionSet& MountPointStats::
    Options( )
    {
      return m_options;
    }

    void MountPointStats::
    operator()( )
    {
      diskCache::Cache::QueryAnswer	answer;

      getHashNumbers( answer, m_ifo.c_str( ), m_type.c_str( ) );
    }

    //===================================================================
    // Scan
    //===================================================================
    OptionSet& Scan::m_options( Scan::init_options( ) );

    OptionSet& Scan::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: scan" );

      retval.
	Summary( "The scan sub command is intended to"
		 " scan a set of directories for files of interest"
		 " and generate a memory cache image."
		 );

      retval.Add( Option( OPT_CONCURRENCY,
			  "concurrency",
			  Option::ARG_REQUIRED,
			  "Number of mount points to scan concurrently",
			  "integer" ) );

      retval.Add( Option( OPT_EXTENSIONS,
			  "extensions",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of file extensions",
			  "list" ) );

      retval.Add( Option( OPT_MOUNT_POINTS,
			  "mount-points",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of mount points to scan",
			  "list" ) );

      retval.Add( Option( OPT_OUTPUT_ASCII,
			  "output-ascii",
			  Option::ARG_REQUIRED,
			  "Filename for the ascii output; '-' to direct to standard output",
			  "filename" ) );

      retval.Add( Option( OPT_OUTPUT_BINARY,
			  "output-binary",
			  Option::ARG_REQUIRED,
			  "Filename for the binary output",
			  "filename" ) );

      retval.Add( Option( OPT_TYPE,
			  "type",
			  Option::ARG_REQUIRED,
			  "Type pattern to use for search. (Default: all)",
			  "pattern" ) );

      retval.Add( Option( OPT_VERSION_ASCII,
			  "version-ascii",
			  Option::ARG_REQUIRED,
			  "Version of the ascii diskcache dump format to output",
			  "version" ) );
      retval.Add( Option( OPT_VERSION_BINARY,
			  "version-binary",
			  Option::ARG_REQUIRED,
			  "Version of the binary diskcache dump format to output",
			  "version" ) );
      return retval;
    }

    Scan::
    Scan( CommandLineOptions& Args )
      : m_args( Args ),
	m_output_ascii( "" ),
	m_output_binary( "" ),
	m_version_ascii( VERSION_DEFAULT_ASCII ),
	m_version_binary( VERSION_DEFAULT_BINARY )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_CONCURRENCY:
	    {
	      std::istringstream	value_stream( arg_value );

	      INT_4U	concurrency;

	      value_stream >> concurrency;

	      ScanConcurrency( concurrency );
	    }
	    break;
	  case OPT_EXTENSIONS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of extensions
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		m_extensions.push_back( arg_value.substr( pos,( ( end == std::string::npos )
								? end
								: end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_MOUNT_POINTS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of mount points
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		m_mount_points.push_back( arg_value.substr( pos,( ( end == std::string::npos )
								  ? end
								  : end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_TYPE:
	    m_type = arg_value;
	    break;
	  case OPT_OUTPUT_ASCII:
	    m_output_ascii = arg_value;
	    break;
	  case OPT_OUTPUT_BINARY:
	    m_output_binary = arg_value;
	    break;
	  case OPT_VERSION_ASCII:
	    {
	      std::istringstream	version_stream( arg_value );

	      version_stream >> std::hex >> m_version_ascii >> std::dec;
	    }
	    break;
	  case OPT_VERSION_BINARY:
	    {
	      std::istringstream	version_stream( arg_value );

	      version_stream >> std::hex >> m_version_binary >> std::dec;
	    }
	    break;
	  default:
	    break;
	  }
	}
      }
    }

    const OptionSet& Scan::
    Options( )
    {
      return m_options;
    }

    void Scan::
    operator()( )
    {
      //-----------------------------------------------------------------
      // Add the extensions
      //-----------------------------------------------------------------
      updateFileExtList( m_extensions );

      //-----------------------------------------------------------------
      // Add the mount points
      //-----------------------------------------------------------------
      if ( m_mount_points.size( ) > 0 )
      {
	MountPointManagerSingleton::UpdateResults    status;

	updateMountPtList( status, m_mount_points, false );
      }

      //-----------------------------------------------------------------
      // Scan the directories
      //-----------------------------------------------------------------

      {
	diskCache::MountPointScanner::ScanResults	results;
	ScanMountPointList( results );

	/// \todo format results for output to stream
	/// diskCache::ASCII::Translate( &std::cout, results );
      }

      //-----------------------------------------------------------------
      // Write the scanned data
      //-----------------------------------------------------------------
      if ( m_output_ascii.empty( )
	   || ( m_output_ascii.compare( "-" ) == 0 ) )
      {
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OASCII	stream( std::cout, m_version_ascii );

	diskCache::Write( stream );
      }
      else if ( m_output_ascii.empty( ) == false )
      {
	//-------------------------------------------------------------
	/// \todo
	///     When the output is ascii, the query options should allow
	///     the user to request a subset of the entire cache.
	//-------------------------------------------------------------
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OFStream	file_stream( m_output_ascii );
	diskCache::Streams::OASCII	stream( file_stream, m_version_ascii );

	diskCache::Write( stream );
      }
      if ( m_output_binary.empty( ) == false )
      {
	//-------------------------------------------------------------
	/// \todo
	///     When the output is binary, the query options should allow
	///     the user to request a subset of the entire cache.
	//-------------------------------------------------------------
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OFStream	file_stream( m_output_binary );
	diskCache::Streams::OBinary	stream( file_stream, m_version_binary );

	diskCache::Write( stream );
      }
    }

    //
    ClientServerInterface::ServerInfo::
    ServerInfo( )
      : port( -1 )
    {
    }
  }
}

#if 0
//=======================================================================
//=======================================================================
static void
send_request( const std::string& Command,
	      const ServerInfo& Server )
{
  if ( ! client_handle )
  {
    client_handle = client_type( new client_type::element_type( ) );
    client_handle->open( Server.Hostname( ), Server.Port( ) );
  }
  (*client_handle) << INT_4U( Command.size( ) );
  client_handle->write( &(Command[0]), Command.size( ) );
  client_handle->flush( );
}
#endif /* 0 */
