#ifndef RegexMatchHH
#define RegexMatchHH

#if ! defined(SWIGIMPORTED)
#include <cstddef>
#endif /* ! defined(SWIGIMPORTED) */

#include "regex.hh"

//-----------------------------------------------------------------------------
/// \brief  Regular Expression Match
///
/// This class represents a regular expression match.  It attempts to match a
/// string with a regular expression, storing the information resulting from
/// captures (the parenthesis expressions).
///
class RegexMatch
{
public:

    /* Constructor/Destructor */
    RegexMatch( size_t nmatches = 0 ) throw(std::bad_alloc);
    ~RegexMatch() throw();

    /* Regex Match */
    bool match( const Regex& r, const char* string, int eflags = 0 ) throw();

    //-------------------------------------------------------------------
    // \brief Return the maximum number of matches
    //-------------------------------------------------------------------
    size_t capacity( ) const;

    //-------------------------------------------------------------------
    // \brief Return the number of matches actually found
    //-------------------------------------------------------------------
    size_t size( ) const;

    /* Accessors */
    inline bool isSubValid( size_t m ) throw();
    inline const char* getSubStart( size_t m ) throw();
    inline const char* getSubEnd( size_t m ) throw();
    inline int getSubStartOffset( size_t m ) throw();
    inline int getSubEndOffset( size_t m ) throw();
    inline size_t getSubLength( size_t m ) throw();    
    std::string getSubString( size_t m ) throw();
    inline size_t getNMatches() const throw();
    
private:

    /* Data */
    regmatch_t* mMatches;
    size_t mNMatches;
    const char* mString;
    size_t found;
};


//-----------------------------------------------------------------------
/// Returns the maximum number of matches which this object can hold.
///  This is equal to the number of matches which was passed to the
///   constructor.
///
/// \return size_t
//-----------------------------------------------------------------------
inline size_t RegexMatch::
capacity() const
{
    return mNMatches;
}

//-----------------------------------------------------------------------
/// Returns the number of matches that were found after the
///   call to match()
///
/// \return size_t
//-----------------------------------------------------------------------
inline size_t RegexMatch::
size() const
{
    return found;
}

//-----------------------------------------------------------------------------
/// \brief  Check Substring Match Validity
///
/// This returns whether or not a substring match exists at the specified index.
///
/// \param m
///
/// \return bool
///
inline bool RegexMatch::isSubValid( size_t m ) throw()
{
    return ( ( m < mNMatches ) && ( mMatches[ m ].rm_so != -1 ) );
}


//-----------------------------------------------------------------------------
/// \brief  Get Substring Start
///
/// Returns a pointer to the beginning of a substring match.  If the requested
/// match is not valid, a null pointer is returned.
///
/// \param m
///
/// \return const char*
///
inline const char* RegexMatch::getSubStart( size_t m ) throw()
{
    if ( !isSubValid( m ) )
    {
        return 0;
    }

    return mString + mMatches[ m ].rm_so;
}


//-----------------------------------------------------------------------------
/// \brief  Get Substring End
///
/// Returns a pointer to the end of a substring match.  If the requested
/// match is not valid, a null pointer is returned.
///
/// \param m
///
/// \return const char*
///
inline const char* RegexMatch::getSubEnd( size_t m ) throw()
{
    if ( !isSubValid( m ) )
    {
        return 0;
    }

    return mString + mMatches[ m ].rm_eo;
}


//-----------------------------------------------------------------------------
/// \brief  Get Substring Starting Offset
///
/// Returns the offset to the beginning of a substring match.  If the requested
/// match is not valid, then -1 is returned.
///
/// \param m
///
/// \return const char*
///
inline int RegexMatch::getSubStartOffset( size_t m ) throw()
{
    if ( !isSubValid( m ) )
    {
        return -1;
    }

    return mMatches[ m ].rm_so;
}


//-----------------------------------------------------------------------------
/// \brief  Get Substring Ending Offset
///
/// Returns the offset to the end of a substring match.  If the requested
/// match is not valid, then -1 is returned.
///
/// \param m
///
/// \return const char*
///
inline int RegexMatch::getSubEndOffset( size_t m ) throw()
{
    if ( !isSubValid( m ) )
    {
        return -1;
    }

    return mMatches[ m ].rm_eo;
}


//-----------------------------------------------------------------------------
/// \brief  Get Substring Length
///
/// Returns the number of characters in a substring match.
///
/// \param m
///
/// \return size_t
///
inline size_t RegexMatch::getSubLength( size_t m ) throw()
{
    return ( getSubEndOffset( m ) - getSubStartOffset( m ) );
}


//-----------------------------------------------------------------------------
/// \brief  Get Number of Matches
///
/// Returns the maximum number of matches which this object can hold.  This is
/// equal to the number of matches which was passed to the constructor.
///
/// \return size_t
///
inline size_t RegexMatch::getNMatches() const throw()
{
    return mNMatches;
}


#endif // RegexMatchHH
