#include "general/config.h"

#if HAVE_SYS_SYSTEMINFO_H
#include <sys/systeminfo.h>
#endif /* HAVE_SYS_SYSTEMINFO_H */

#include <cstdlib>
#include <dlfcn.h>

#include <sstream>
#include <stdexcept>

#include "general/SOLoader.hh"

static std::string find_lib( const std::string& Path,
			     const std::string& LibName );

namespace General
{
  SOLoader::
  SOLoader( const std::string& Path, const std::string& LibName, int Type )
    : m_handle( (handle_type)NULL )
  {
    std::string lib = find_lib( Path, LibName );

    m_handle = dlopen( lib.c_str( ), RTLD_NOW );
    if ( ( m_handle == (handle_type)NULL ) && ( Type == SOLOADER_MANDITORY ) )
    {
      std::stringstream	msg;

      msg << "Unable to load " << lib;
      throw std::runtime_error( msg.str( ).c_str( ) );
    }
  }

  SOLoader::
  ~SOLoader( )
  {
    if ( m_handle )
    {
      dlclose( m_handle );
      m_handle = (handle_type)NULL;
    }
  }

  SOLoader::function_type SOLoader::
  GetFunction( const std::string& Function )
  {
    //-------------------------------------------------------------------
    // :TRICKY: This union is needed to avoid warning about converting
    // :TRICKY:    from void* to void (*)(void).
    //-------------------------------------------------------------------
    union {
      function_type	f;
      symbol_type	s;
    } retval;

    if ( m_handle )
    {
      retval.s = dlsym( m_handle, Function.c_str( ) );
    }
    else
    {
      retval.s = (symbol_type)NULL;
    }
    return retval.f;
  }

  SOLoader::symbol_type SOLoader::
  GetSymbol( const std::string& Symbol )
  {
    symbol_type	retval = (symbol_type)( NULL );

    if ( m_handle )
    {
      retval = (symbol_type)( dlsym( m_handle, Symbol.c_str( ) ) );
    }
    return retval;
  }
}

static std::string
find_lib( const std::string& Path,
	  const std::string& LibName )
{
  std::ostringstream	retval;
  
  retval << Path;
#ifdef SI_ARCHITECTURE_64
  if ( sizeof( void(*)() ) == 8 )
  {
    // Running as a 64-bit application
    char	buffer[256];

    sysinfo( SI_ARCHITECTURE_64, buffer, sizeof( buffer ) );
    retval << "/" << buffer;
  }
#endif /* SI_ARCHITECTURE_64 */
  retval << "/" << LibName;

  return retval.str( );
}
