#include "general/config.h"

#include <time.h>
#include <algorithm>

#include "general/AtExit.hh"
#include "general/Log.hh"

static const int OFFLOAD_CPU_FOR_N_SECONDS = 1;
static const int DEFAULT_BYTES_MAX   = 100000;
static const int DEFAULT_ENTRIES_MAX =      0;
static const int WAIT_FOR_N_SECONDS = 5;

namespace General
{
  //---------------------------------------------------------------------
  /// Constructor for the Log class.
  /// This will save the base name to use for the log file
  /// and start the writing thread.
  ///
  /// This class works by seperating the request to log a message
  /// and the actual writing of the message.
  /// This seperation allows for multiple threads to log messages
  /// asyncronously while minimizing the serialization caused by
  /// critical resources being locked.
  //---------------------------------------------------------------------
  Log::
  Log( const std::string& BaseName )
    : m_base_name( BaseName ),
      m_bytes( 0 ),
      m_bytes_max( DEFAULT_BYTES_MAX ),
      m_entries( 0 ),
      m_entries_max( DEFAULT_ENTRIES_MAX ),
      m_message_baton( MutexLock::Initialize( ) )
  {
    //-------------------------------------------------------------------
    // Start the thread which is responsible for streaming the messages
    // to the output stream.
    //-------------------------------------------------------------------
#if 0
    Spawn( );
#endif /* 0 */
  }

  Log::
  ~Log( )
  {
    //-------------------------------------------------------------------
    // Stop background logging.
    //-------------------------------------------------------------------
    Cancel( );
    Join( );
    //-------------------------------------------------------------------
    // Close the stream to ensure it has been flushed.
    //-------------------------------------------------------------------
    m_stream.close( );
    //-------------------------------------------------------------------
    // if the message queue is not empty, flush them now
    //-------------------------------------------------------------------
    while( m_message_queue.empty( ) == false )
    {
      std::cerr << m_message_queue.front( ).s_message
		<< std::flush
	;
      m_message_queue.pop( );
    }
  }
  //---------------------------------------------------------------------
  /// This member first determins if the Message should be logged based
  /// on the Level for the Group.
  /// If it should be logged, it will be added to the message queue
  /// in a thread safe manner.
  /// Another thread is responsible for actually outputting the
  /// queue of messages to the log stream.
  //---------------------------------------------------------------------
  void Log::
  Message( const group_type Group,
	   const level_type Level,
	   const std::string& Message )
  {
    //-------------------------------------------------------------------
    // Determine if the message is to be put into the message queue.
    //-------------------------------------------------------------------
    if ( VerbosityCheck( Group, Level ) == false  )
    {
      return;
    }
    //-------------------------------------------------------------------
    // From the looks of everything, the message should be added to
    // the queue.
    //
    // Lock the queue to prevent others from modifying its contents
    // and potentially corrupt memory.
    //-------------------------------------------------------------------
    MutexLock	lock( m_message_baton );

    m_message_queue.push( message_queue_entry_type( ) );
    m_message_queue.back( ).s_message = Message;
    m_message_queue.back( ).s_timestamp.Now( );
  }

  //---------------------------------------------------------------------
  /// By default, there is no special action that needs to take place
  /// when the file is closed.
  //---------------------------------------------------------------------
  void Log::
  onStreamClose( )
  {
  }

  //---------------------------------------------------------------------
  /// By default, there is no special action that needs to take place
  /// when the file is opened.
  //---------------------------------------------------------------------
  void Log::
  onStreamOpen( )
  {
  }

  //---------------------------------------------------------------------
  /// This method is used for updating the output stream.
  /// It is done by a child thread and is started when the object is
  /// created.
  /// The thread is terminated when the object is destroyed.
  //---------------------------------------------------------------------
  void Log::
  action( )
  {
    //-------------------------------------------------------------------
    // Setup the thread.
    // Currently allow asyncronous cancellation since we are only
    // sleeping.
    //-------------------------------------------------------------------
    CancellationType( Thread::CANCEL_ASYNCHRONOUS );
    CancellationEnable( true );

    {
      //-----------------------------------------------------------------
      // Give time for everything else to get started
      //-----------------------------------------------------------------
      struct timespec wakeup;
      
      wakeup.tv_sec = WAIT_FOR_N_SECONDS;
      wakeup.tv_nsec = 0;
      
      nanosleep( &wakeup, NULL );
    }

    //-------------------------------------------------------------------
    // Reset the way to be cancelled to be via exception since there
    // are more things that need to be cleaned up.
    //-------------------------------------------------------------------
    CancellationType( Thread::CANCEL_EXCEPTION );
    try
    {
      static const int buffer_size = ( 64 * 1024 );

      m_stream_buffer.reset( new char[ buffer_size ] );
      m_stream.rdbuf( )->pubsetbuf( m_stream_buffer.get( ), buffer_size );
      //-----------------------------------------------------------------
      // Storage for the pending messages
      //-----------------------------------------------------------------
      Log::message_queue_type	mq;

      //-----------------------------------------------------------------
      // main thread execution
      //-----------------------------------------------------------------
      while ( AtExit::IsExiting( ) == false )
      {
	try
	{
	  //-------------------------------------------------------------
	  // Be cooperative with cancelation
	  //-------------------------------------------------------------
	  CancellationCheck(  "General::Log:: action", __FILE__, __LINE__ );

	  //-------------------------------------------------------------
	  // Critical section
	  //---------------------------------------------------------------
	  {
	    //-------------------------------------------------------------
	    // Reduce the time the message queue is locked by moving the
	    // messages to a temporary variable.
	    //-------------------------------------------------------------
	    MutexLock	lock( m_message_baton );

	    if ( m_message_queue.empty( ) == false )
	    {
	      std::swap( m_message_queue, mq );
	    }
	  }
	  //---------------------------------------------------------------
	  // Check if there any work to be done
	  //---------------------------------------------------------------
	  if ( mq.empty( ) )
	  {
	    //-------------------------------------------------------------
	    // Nothing to do so offload the cpu for a couple of seconds
	    //-------------------------------------------------------------
	    struct timespec wakeup;

	    wakeup.tv_sec = OFFLOAD_CPU_FOR_N_SECONDS;
	    wakeup.tv_nsec = 0;

	    nanosleep( &wakeup, NULL );
	  }
	  else // if ( mq.empty( ) )
	  {
	    //-------------------------------------------------------------
	    // Output the messages to the ouput stream.
	    //-------------------------------------------------------------
	    while ( mq.empty( ) == false )
	    {
	      write_message( mq.front( ) );
	      mq.pop( );	// Remove from the queue
	    } // while ( mq.empty( ) == false )
	  } // if ( mq.empty( ) )
	}
	catch( const Thread::cancellation& Exception )
	{
	  //-------------------------------------------------------------
	  // Rethrow because we want out
	  //-------------------------------------------------------------
	  throw;
	}
	catch( std::exception& Exception )
	{
	  //-------------------------------------------------------------
	  // Something has happened.
	  // Quietly absorb.
	  //-------------------------------------------------------------
	  std::cerr << "DEBUG: caught Log error: " << Exception.what( )
		    << std::endl
	    ;
	  //-------------------------------------------------------------
	  // Throw away the rest of the message
	  //-------------------------------------------------------------
	  while( mq.empty( ) == false )
	  {
	    mq.pop( );
	  }
	}
	catch( ... )
	{
	  //-------------------------------------------------------------
	  // Something has happened.
	  // Quietly absorb.
	  //-------------------------------------------------------------
	  std::cerr << "DEBUG: caught unknown Log error"
		    << std::endl
	    ;
	  //-------------------------------------------------------------
	  // Throw away the rest of the message
	  //-------------------------------------------------------------
	  while( mq.empty( ) == false )
	  {
	    mq.pop( );
	  }
	}
      } // while( 1 )
    } // try
    catch( const Thread::cancellation& Exception )
    {
      //-----------------------------------------------------------------
      // Quietly ignore this one since it is an indication of shutting
      // down.
      //-----------------------------------------------------------------
    }
    catch( std::exception& Exception )
    {
      std::cerr << "DEBUG: caught Log error: " << Exception.what( )
		<< std::endl
	;
    }
    catch( ... )
    {
      std::cerr << "DEBUG: caught unknown Log error"
		<< std::endl
	;
    }

  } // Log::writer( )
      
  //---------------------------------------------------------------------
  /// Handle the rotation of the log files.
  /// \note
  ///     Log rotation only occurs when the logging stream is closed.
  //---------------------------------------------------------------------
  void Log::
  rotate( )
  {
    //-------------------------------------------------------------------
    // If m_base_name does not exist, return since there is no need
    // to rotate.
    //-------------------------------------------------------------------
    if ( m_stream.is_open( )
	 /* || access( m_name_base ) */ )
    {
      return;
    }

    //-------------------------------------------------------------------
    // Remove the oldest
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    // Loop backwards moving the N to N + 1
    //-------------------------------------------------------------------
  }

  //---------------------------------------------------------------------
  /// This method does the actual writing of the message to the log.
  //---------------------------------------------------------------------
  void Log::
  write_message( const message_queue_entry_type& MessageInfo )
  {
    //-------------------------------------------------------------------
    // Check if the output stream is open.
    //-------------------------------------------------------------------
    if ( ( m_stream.is_open( ) == false )
	 && ( IsParentThread( ) == false ) )
    {
      //-----------------------------------------------------------------
      // Since it is not open, rotate the logs
      //-----------------------------------------------------------------
      rotate( );
    }
    //-------------------------------------------------------------------
    // Output the message.
    //-------------------------------------------------------------------
    writeDirect( MessageInfo.s_message );
    //-------------------------------------------------------------------
    // Check to see if the log file should be closed.
    //-------------------------------------------------------------------
    if ( ( ( m_bytes_max ) && ( m_bytes >= m_bytes_max ) )
	 || ( ( m_entries_max ) && ( m_entries >= m_entries_max ) ) )
    {
      m_stream.close( );
      m_bytes = 0;
      m_entries = 0;
    }
  }

  //---------------------------------------------------------------------
  /// This method can be used by derived classes for writing a message
  /// to the stream without having that message going to the message
  /// queue.
  //---------------------------------------------------------------------
  void Log::
  writeDirect( const std::string& Message )
  {
    //-------------------------------------------------------------------
    // This method can only be called by the child thread.
    //-------------------------------------------------------------------
    if ( IsParentThread( ) )
    {
      std::ostringstream msg;

      msg << "General::Log::writeDirect can only be called by the child thread";
      std::cerr << msg.str( ) << std::endl;

      throw std::runtime_error( msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Write the information only to an open stream
    //-------------------------------------------------------------------
    if ( m_stream.is_open( ) )
    {
      //-----------------------------------------------------------------
      // Output to the stream without any additional formatting.
      // The stream is flushed to ensure that the data gets out
      // in case of a crash.
      //-----------------------------------------------------------------
      m_stream << Message << std::flush;
      m_bytes += Message.length( );
      ++m_entries;
    }
    else
    {
      std::cerr << Message << std::flush;
    }
  }
}
