#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>

#include <fcntl.h>
#include <poll.h>
#include <unistd.h>

#include <cstdarg>

#include <iostream>

#include "Exec.hh"

namespace /* Anonymous */
{
  using namespace General;

  void
  child_cpu_time( Exec::time_type& User,
		  Exec::time_type& System )
  {
    struct rusage ru;        
    
    getrusage(RUSAGE_CHILDREN, &ru);

    User =
      (double)( ru.ru_utime.tv_sec )
      + (double)( ru.ru_utime.tv_usec ) / 1000000.0;
    System =
      (double)( ru.ru_stime.tv_sec )
      + (double)( ru.ru_stime.tv_usec ) / 1000000.0;
  }
} /* namespace - anonymous */

namespace General
{
  const char* Exec::COMMAND_END = (const char*)NULL;

  Exec::
  Exec( arg_type Command, ... )
  {
    m_child_exit.s_exit_code = 0;
    m_child_exit.s_signal = 0;
    m_child_exit.s_core_dump = 0;
    m_child_exit.s_exited = false;

    arg_type arg;

    va_list( ap );

    //-------------------------------------------------------------------
    // Preserve the command to execute
    //-------------------------------------------------------------------
    m_options.push_back( Command );

    va_start( ap, Command );
    while( ( arg = va_arg( ap, arg_type ) )
	   != COMMAND_END )
    {
      //-----------------------------------------------------------------
      // Add arguments
      //-----------------------------------------------------------------
      m_options.push_back( arg );
    }

    m_options.push_back( COMMAND_END );

    va_end( ap );
  }

  void Exec::
  Spawn( )
  {
    //-------------------------------------------------------------------
    // Need to record some information to properly calculate
    // the amount of time spent in the sub-process
    //-------------------------------------------------------------------
    child_cpu_time( m_start_time_user,
		    m_start_time_system );
    
    //-------------------------------------------------------------------
    // Do the usual stuff
    //-------------------------------------------------------------------
    Fork::Spawn( );
  }

  void Exec::
  evalChild( )
  {
    //-------------------------------------------------------------------
    // Replace stdin with pipe
    //-------------------------------------------------------------------
    close( 0 );	// Close stdin
    if( dup( getStdIn( ) ) == -1 )
    {
      _exit( 1 );
    }
    closeStdIn( );

    //-------------------------------------------------------------------
    // Replace stdout with pipe
    //-------------------------------------------------------------------
    close( 1 ); // Close stdout
    if ( dup( getStdOut( ) ) == -1 )
    {
      _exit( 1 );
    }
    closeStdOut( );

    //-------------------------------------------------------------------
    // Replace stderr with pipe
    //-------------------------------------------------------------------
    close( 2 ); // Close stdout
    if ( dup( getStdErr( ) ) == -1 )
    {
      _exit( 1 );
    }
    closeStdErr( );

    //-------------------------------------------------------------------
    // Run the command in the new shell
    //-------------------------------------------------------------------
    char* const *
      xargs( const_cast< char* const*>( &(m_options[ 0 ] ) ) );
    execv( m_options[ 0 ],
	   xargs );
    _exit( 1 );
  }

  void Exec::
  evalParent( )
  {
    //-------------------------------------------------------------------
    // Give time for the child process to spawn 
    //-------------------------------------------------------------------
    sleep( 2 );

    //-------------------------------------------------------------------
    // Wait for the child process to complete
    //-------------------------------------------------------------------
    int			stat;
    pid_t		retval;
    bool		done = false;
    char		buffer[ 256 ];
    static const int	MAX_MONITOR = 2;
    static const int 	timeout = 5000; /* milliseconds ( 1/1000 ) */

    struct pollfd	monitor[ MAX_MONITOR ];

    monitor[ E_STDOUT ].fd = getStdOut( );
    monitor[ E_STDOUT ].events = POLLIN;

    monitor[ E_STDERR ].fd = getStdErr( );
    monitor[ E_STDERR ].events = POLLIN;
    
    while ( ! done )
    {
      int retval = poll( monitor, MAX_MONITOR, timeout );
      if ( retval > 0 )
      {
	for ( int
		x = 0;
	      x < MAX_MONITOR;
	      ++x )
	{
	  if ( monitor[ x ].revents != 0 )
	  {
	    if ( monitor[ x ].revents & POLLIN )
	    {
	      ssize_t count = read( monitor[ x ].fd, buffer, sizeof( buffer ) );
	      if ( count > 0 )
	      {
		m_child_exit.s_output[ x ].append( buffer, count );
		++retval;
	      }
	    }
	    if ( monitor[ x ].revents & ( POLLERR | POLLHUP | POLLNVAL ) )
	    {
	      --retval;
	    }
	  }
	}
	if ( retval == 0 )
	{
	  done = true;
	}
      }
    }
    retval = waitpid( ChildPid( ), &stat, 0 );

    if ( retval == -1 )
    {
      std::cerr << "ERROR: waitpid: Returned -1"
		<< std::endl
	;
      return;
    }

    child_cpu_time( m_child_exit.s_time_user,
		    m_child_exit.s_time_system );
    
    m_child_exit.s_time_user =
      m_child_exit.s_time_user - m_start_time_user;
    m_child_exit.s_time_system =
      m_child_exit.s_time_system - m_start_time_system;

    if ( WIFEXITED( stat ) )
    {
      m_child_exit.s_exit_code = WEXITSTATUS( stat );
      m_child_exit.s_exited = true;
    }
    else if ( WIFSIGNALED( stat ) )
    {
      m_child_exit.s_signal = WTERMSIG( stat );
      m_child_exit.s_core_dump = WCOREDUMP( stat );
    }
  }
}
