#include "general/config.h"

#include <stdlib.h>
#include <string.h>
#include <iostream>

#include "general/AtExit.hh"

namespace anonymous
{
  void exit_handler( );
}

extern "C"
{
  typedef void(*atexit_parm_type)();
}

// Instantiate generic to Singleton class methods:   
// see "general" library   
SINGLETON_TS_INST( General::AtExit );   

namespace General
{
  bool AtExit::m_is_exiting = false;

  AtExit::exit_function_node::
  exit_function_node( ExitFunction Function,
		      const std::string& Name,
		      exit_function_node* Next )
    : m_function( Function ),
      m_name( ::strdup( Name.c_str( ) ) ),
      m_next( Next )
  {
  }

  //---------------------------------------------------------------------
  /// Constructs and allocates resources needed to allow for clean
  /// exiting. This allows for more usefull memory diagnostics using
  /// tools such as Valgrind.
  //---------------------------------------------------------------------
  AtExit::
  AtExit( )
    : m_head( (AtExit::exit_ring_node*)NULL ),
      m_is_exiting_baton( MutexLock::Initialize( ) )

  {
    {
      MutexLock	l( m_is_exiting_baton );
      
      m_is_exiting = false;
    }
    (void)atexit( (atexit_parm_type)(singleton_suicide) );
  }

  //---------------------------------------------------------------------
  /// When exiting, call exit handlers in reverse order.
  //---------------------------------------------------------------------
  AtExit::
  ~AtExit( )
  {
    cleanup( );
  }

  void AtExit::
  append( ExitFunction Function, const std::string& Name, int Ring )
  {
    MutexLock	l( m_is_exiting_baton );

    if ( m_is_exiting )
    {
      return;
    }
#if 0
    std::cerr << "AtExit: Appending: Ring: " << Ring
	      << " Function: " << Name
	      << std::endl;
#endif /* 0 */
    exit_ring_node* cur = m_head;

    if ( m_head == (exit_ring_node*)NULL )
    {
      m_head = new exit_ring_node( Ring, m_head );
      cur = m_head;
    }
    else
    {
      //-----------------------------------------------------------------
      // Advance down the list of rings
      //-----------------------------------------------------------------
      exit_ring_node* prev = (exit_ring_node*)NULL;

      while ( cur && ( cur->m_ring > Ring ) )
      {
	prev = cur;
	cur = cur->m_next;
      }
      if ( cur )
      {
	if ( cur->m_ring != Ring )
	{
	  exit_ring_node* n =
	    new exit_ring_node( Ring, cur );

	  if ( prev )
	  {
	    prev->m_next = n;
	  }
	  else
	  {
	    m_head = n;
	  }
	  cur = n;
	}
      }
      else if ( prev )
      {
	cur =
	  new exit_ring_node( Ring, cur );
	prev->m_next = cur;
      }
    }
    cur->m_exit_function_list =
      new exit_function_node( Function, Name, cur->m_exit_function_list );
  }

  //---------------------------------------------------------------------
  /// When exiting, call exit handlers in reverse order.
  //---------------------------------------------------------------------
  void AtExit::
  cleanup( )
  {
    {
      MutexLock	l( m_is_exiting_baton );

      m_is_exiting = true;
    }
    //-------------------------------------------------------------------
    // Delete objects in the reverse order they were created to ensure
    // object dependencies are maintained.
    //-------------------------------------------------------------------
    while ( m_head )
    {
      volatile exit_function_node*
	cur_function_node = m_head->m_exit_function_list;

      while ( cur_function_node )
      {
#if 0
	std::cerr << "AtExit: Ring: " << m_head->m_ring
		  << " Function: " << cur_function_node->m_name
		  << std::endl;
#endif /* 0 */

	::free( cur_function_node->m_name );

	volatile exit_function_node*	old_node = cur_function_node;
	volatile ExitFunction		f = old_node->m_function;

	cur_function_node = old_node->m_next;
	delete old_node;

	(*f)( );
      }
      volatile exit_ring_node* old_head_node = m_head;
      m_head = m_head->m_next;
      delete old_head_node;
    }
  }

} // namespace - General
