#ifndef FrameCPP_VERSION_8_FrameH_HH
#define FrameCPP_VERSION_8_FrameH_HH

#if !defined(SWIG)
#include "general/types.hh"

#include "framecpp/Version8/FrameSpec.hh"

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/SearchContainer.hh"
#include "framecpp/Common/FrameH.hh"
#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version7/FrameH.hh"

#include "framecpp/Version8/GPSTime.hh"
#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/PTR_STRUCT.hh"
#include "framecpp/Version8/FrSE.hh"

#include "framecpp/Version8/FrDetector.hh"
#include "framecpp/Version8/FrEvent.hh"
#include "framecpp/Version8/FrHistory.hh"
#include "framecpp/Version8/FrProcData.hh"
#include "framecpp/Version8/FrRawData.hh"
#include "framecpp/Version8/FrSimData.hh"
#include "framecpp/Version8/FrSimEvent.hh"
#include "framecpp/Version8/FrSummary.hh"
#include "framecpp/Version8/FrTable.hh"
#include "framecpp/Version8/FrVect.hh"
#endif /* !defined(SWIG) */

namespace FrameCPP
{
  namespace Common
  {
    class Description;
  }

  namespace Version_8
  {
    //===================================================================
    /// \brief Non-PTR_STRUCT elements of the FrameH structure
    ///
    /// This class contains the data members of the FrameH structure
    /// that are not refernced by a pointer.
    /// This allows for easier migration into future
    /// frame specifications.
    //===================================================================

    typedef Previous::FrameHNPS FrameHNPS;

    //===================================================================
    /// \brief PTR_STRUCT elements of the FrmaeH structure.
    ///
    /// This class contains the PTR_STRUCT references of the FrameH
    /// structure.
    /// This allows for easier migration into future
    /// frame specifications.
    //===================================================================

#if 1
    typedef Version_6::FrameHPST< FrDetector, FrEvent, FrHistory,
				  FrProcData, FrRawData, FrSimData,
				  FrSimEvent, FrSummary, FrTable, FrVect >
    FrameHPS;
#else /* 0 */
    typedef Previous::FrameHPS FrameHPS;
#endif /* 0 */

    //===================================================================
    /// \brief Frame Header
    //===================================================================
    class FrameH
      : public ObjectWithChecksum< Common::FrameH >,
        public FrameHNPS,
	public FrameHPS
    {
    public:
      typedef Common::FrameH::assign_stream_type assign_stream_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      //-----------------------------------------------------------------
      FrameH( );

      //-----------------------------------------------------------------
      /// \param[in] frame
      ///     Frame to use as the base
      //-----------------------------------------------------------------
      FrameH( const FrameH& frame );

      //-----------------------------------------------------------------
      /// \brief Explicate construction
      ///
      /// \param[in] name
      ///     Name of project or other experiment description
      /// \param[in] run
      ///     Run number (number < 0 reserved for simulated data);
      ///     monotonic for experimental runs.
      /// \param[in] frame
      ///     Frame number, monotonically increasing until end of run,
      ///     re-starting from 0 with each new run.
      /// \param[in] time
      ///     Frame start time in GPS units.
      /// \param[in] uLeapS      
      ///     The integer number of leap seconds between GPS/TAI and
      ///     UTC in the epoch when the frame is written.
      /// \param[in] dt
      ///     Frame length in seconds
      /// \param[in] dqual
      ///     Data quality flag
      ///
      /// \return
      ///     A new instance of this object.
      //-----------------------------------------------------------------
      FrameH( const std::string& name,
	      INT_4S run,
	      INT_4U frame,
	      const GPSTime& time,
	      INT_2U uLeapS,
	      const REAL_8 dt,
	      INT_4U dqual = 0 );

      //-----------------------------------------------------------------
      /// \brief Copy constructor
      ///
      /// \param[in] Source
      ///     The non-pointer portion of the frame header.
      //-----------------------------------------------------------------
      FrameH( const FrameHNPS& Source );

      //-----------------------------------------------------------------
      /// \brief Upconvert constructor
      ///
      /// \param[in] Source
      ///     An instance of a similiar object from a previous version
      ///     of the frame specification.
      /// \param[in] Stream
      ///     The stream from which the origional object was read.
      //-----------------------------------------------------------------
      FrameH( const Previous::FrameH& Source,
	      istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief Destructor
      //-----------------------------------------------------------------
      ~FrameH( );

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      //-----------------------------------------------------------------
      virtual FrameH* Clone( ) const;

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrameH object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrameH& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

    protected:
      //-----------------------------------------------------------------
      /// \brief Assign the contents of the instance
      ///
      /// \param[in] Source
      ///     An input stream from which to read information for
      ///     the instance.
      ///
      /// \note
      ///     This has been hidden so programmers are not tempted to
      ///     bypass usage of IFrameStream as the input stream.
      //-----------------------------------------------------------------
      virtual void assign( assign_stream_type& Stream );

      //-----------------------------------------------------------------
      /// \brief Limitted reading of the object
      ///
      /// \param[in] Stream
      ///     The input stream from which to read the object.
      /// \param[in] ElementMask
      ///     Mask of elements to read from the Stream
      //-----------------------------------------------------------------
      virtual void readSubset( istream_type& Stream,
			       INT_4U ElementMask );

      //-----------------------------------------------------------------
      /// \brief Frame start time in seconds
      //-----------------------------------------------------------------
      virtual start_time_normalized_type normalizedStartTime( ) const;

      //-----------------------------------------------------------------
      /// \brief Duration of the frame
      //-----------------------------------------------------------------
      virtual delta_t_normalized_type normalizedDeltaT( ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor.
      ///
      /// \param[in] Stream
      ///     The input stream from which to read the object.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      virtual FrameH* pCreate( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( ostream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Verify the object as being valid
      ///
      /// \param[in] Verifier
      ///     The instance of the Verify object where state information
      ///     is stored.
      /// \param[in] Stream
      ///     The input stream from which the object is being read.
      //-----------------------------------------------------------------
      virtual void VerifyObject( Common::Verify& Verifier,
				 Common::IFrameStream& Stream ) const;
    private:
      //-----------------------------------------------------------------
      /// \brief Object with checksum data.
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameH > object_type;

      using Common::FrameSpec::Object::Create;

      //-----------------------------------------------------------------
      /// \brief Constructor via stream
      //-----------------------------------------------------------------
      FrameH( istream_type& Stream );

      //-----------------------------------------------------------------
      /// \brief Down grade an object
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Upgrade an object
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of this class.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline FrameH::
    FrameH( const Previous::FrameH& Source,
	    istream_type* Stream )
      : object_type( StructDescription( ) ),
	FrameHNPS( Source )
    {
      if ( Stream )
      {
	const INT_2U max_ref = Previous::FrameHPS::MAX_REF;

	Stream->ReplaceRef( RefType( ), Source.RefType( ), max_ref );
	Stream->ReplaceRef( RefUser( ), Source.RefUser( ), max_ref );
	Stream->ReplaceRef( RefDetectSim( ), Source.RefDetectSim( ), max_ref );
	Stream->ReplaceRef( RefDetectProc( ), Source.RefDetectProc( ), max_ref );
	Stream->ReplaceRef( RefHistory( ), Source.RefHistory( ), max_ref );
	Stream->ReplacePtr( AddressOfRawData( ),
			    Source.AddressOfRawData( ),
			    max_ref );
	Stream->ReplaceRef( RefProcData( ), Source.RefProcData( ), max_ref );
	Stream->ReplaceRef( RefSimData( ), Source.RefSimData( ), max_ref );
	Stream->ReplaceRef( RefEvent( ), Source.RefEvent( ), max_ref );
	Stream->ReplaceRef( RefSimEvent( ), Source.RefSimEvent( ), max_ref );
	Stream->ReplaceRef( RefSummaryData( ), Source.RefSummaryData( ), max_ref );
	Stream->ReplaceRef( RefAuxData( ), Source.RefAuxData( ), max_ref );
	Stream->ReplaceRef( RefAuxTable( ), Source.RefAuxTable( ), max_ref );
      }
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline FrameH::promote_ret_type FrameH::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     istream_type* Stream )
    {
      return Object::PromoteObject< Previous::FrameH, FrameH >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline const char* FrameH::
    StructName( )
    {
      static const char* name = "FrameH";
      return name;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline bool FrameH::
    operator==( const FrameH& RHS ) const
    {
      return ( ( m_data == RHS.m_data )
	       && ( m_refs == RHS.m_refs ) );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline start_time_normalized_type FrameH::
    normalizedStartTime( ) const
    {
      return GetGTime( );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline delta_t_normalized_type FrameH::
    normalizedDeltaT( ) const
    {
      return GetDt( );
    }
  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrameH_HH */
